/*
 * Decompiled with CFR 0.152.
 */
package org.cesecore.keys.util;

import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.math.BigInteger;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.UnrecoverableEntryException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.ECGenParameterSpec;
import java.util.Date;
import java.util.Enumeration;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.bouncycastle.asn1.ASN1Set;
import org.bouncycastle.asn1.DERSet;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509v3CertificateBuilder;
import org.bouncycastle.jce.ECKeyUtil;
import org.bouncycastle.operator.BufferingContentSigner;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.ContentVerifierProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.cesecore.certificates.util.AlgorithmTools;
import org.cesecore.config.CesecoreConfiguration;
import org.cesecore.internal.InternalResources;
import org.cesecore.keys.KeyCreationException;
import org.cesecore.keys.token.CachingKeyStoreWrapper;
import org.cesecore.util.Base64;
import org.cesecore.util.CertTools;

public class KeyStoreTools {
    private static final Logger log = Logger.getLogger(KeyStoreTools.class);
    private static final InternalResources intres = InternalResources.getInstance();
    protected final CachingKeyStoreWrapper keyStore;
    private final String providerName;

    public KeyStoreTools(CachingKeyStoreWrapper _keyStore, String _providerName) {
        this.keyStore = _keyStore;
        this.providerName = _providerName;
    }

    public String getProviderName() {
        return this.providerName;
    }

    public CachingKeyStoreWrapper getKeyStore() {
        return this.keyStore;
    }

    public void setKeyEntry(String alias, Key key, Certificate[] chain) throws KeyStoreException {
        this.getKeyStore().setKeyEntry(alias, key, null, chain);
    }

    private void deleteAlias(String alias) throws KeyStoreException {
        this.getKeyStore().deleteEntry(alias);
    }

    public void deleteEntry(String alias) throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException {
        if (alias != null) {
            this.deleteAlias(alias);
        } else {
            Enumeration<String> e = this.getKeyStore().aliases();
            while (e.hasMoreElements()) {
                String str = e.nextElement();
                this.deleteAlias(str);
            }
        }
    }

    public void copyEntry(String oldAlias, String newAlias) throws KeyStoreException, NoSuchAlgorithmException, UnrecoverableEntryException, CertificateException, IOException {
        this.getKeyStore().setEntry(newAlias, this.getKeyStore().getEntry(oldAlias, null), null);
    }

    private X509Certificate getSelfCertificate(String myname, long validity, String sigAlg, KeyPair keyPair) throws InvalidKeyException, CertificateException {
        long currentTime = new Date().getTime();
        Date firstDate = new Date(currentTime - 86400000L);
        Date lastDate = new Date(currentTime + validity * 1000L);
        X500Name issuer = new X500Name(myname);
        BigInteger serno = BigInteger.valueOf(firstDate.getTime());
        PublicKey publicKey = keyPair.getPublic();
        if (publicKey == null) {
            throw new InvalidKeyException("Public key is null");
        }
        try {
            JcaX509v3CertificateBuilder cg = new JcaX509v3CertificateBuilder(issuer, serno, firstDate, lastDate, issuer, publicKey);
            log.debug((Object)("Keystore signing algorithm " + sigAlg));
            BufferingContentSigner signer = new BufferingContentSigner(new JcaContentSignerBuilder(sigAlg).setProvider(this.providerName).build(keyPair.getPrivate()), 20480);
            X509CertificateHolder cert = cg.build((ContentSigner)signer);
            return (X509Certificate)CertTools.getCertfromByteArray(cert.getEncoded());
        }
        catch (OperatorCreationException e) {
            log.error((Object)"Error creating content signer: ", (Throwable)e);
            throw new CertificateException(e);
        }
        catch (IOException e) {
            throw new CertificateException("Could not read certificate", e);
        }
    }

    private void generateEC(String name, String keyEntryName) throws InvalidAlgorithmParameterException {
        KeyPairGenerator kpg;
        if (log.isTraceEnabled()) {
            log.trace((Object)(">generate EC: curve name " + name + ", keyEntryName " + keyEntryName));
        }
        try {
            kpg = KeyPairGenerator.getInstance("EC", this.providerName);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("Algorithm ECwas not recognized.", e);
        }
        catch (NoSuchProviderException e) {
            throw new IllegalStateException("BouncyCastle was not found as a provider.", e);
        }
        try {
            ECGenParameterSpec ecSpec;
            Provider prov = Security.getProvider(this.providerName);
            if (StringUtils.contains((String)prov.getClass().getName(), (String)"iaik")) {
                throw new InvalidAlgorithmParameterException("IAIK ECC key generation not implemented.");
            }
            if (StringUtils.equals((String)name, (String)"implicitlyCA")) {
                if (log.isDebugEnabled()) {
                    log.debug((Object)"Generating implicitlyCA encoded ECDSA key pair");
                }
                ecSpec = null;
            } else {
                String oidOrName = AlgorithmTools.getEcKeySpecOidFromBcName(name);
                if (log.isDebugEnabled()) {
                    log.debug((Object)("keySpecification '" + name + "' transformed into OID " + oidOrName));
                }
                ecSpec = new ECGenParameterSpec(oidOrName);
            }
            kpg.initialize(ecSpec);
        }
        catch (InvalidAlgorithmParameterException e) {
            log.debug((Object)("EC name " + name + " not supported."));
            throw e;
        }
        this.generateKeyPair(kpg, keyEntryName, "SHA1withECDSA");
        if (log.isTraceEnabled()) {
            log.trace((Object)("<generate: curve name " + name + ", keyEntryName " + keyEntryName));
        }
    }

    private void generateExtraEC(String name, String keyEntryName, String algInstanceName, String sigAlgName) throws InvalidAlgorithmParameterException {
        KeyPairGenerator kpg;
        if (log.isTraceEnabled()) {
            log.trace((Object)(">generate " + algInstanceName + ": curve name " + name + ", keyEntryName " + keyEntryName));
        }
        try {
            kpg = KeyPairGenerator.getInstance(algInstanceName, this.providerName);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("Algorithm " + name + "was not recognized.", e);
        }
        catch (NoSuchProviderException e) {
            throw new IllegalStateException("BouncyCastle was not found as a provider.", e);
        }
        try {
            ECGenParameterSpec ecSpec = new ECGenParameterSpec(name);
            kpg.initialize(ecSpec);
        }
        catch (InvalidAlgorithmParameterException e) {
            log.debug((Object)("EC " + algInstanceName + " name " + name + " not supported."));
            throw e;
        }
        this.generateKeyPair(kpg, keyEntryName, sigAlgName);
        if (log.isTraceEnabled()) {
            log.trace((Object)("<generate: curve name " + name + ", keyEntryName " + keyEntryName));
        }
    }

    private void generateGOST3410(String name, String keyEntryName) throws InvalidAlgorithmParameterException {
        this.generateExtraEC(name, keyEntryName, "ECGOST3410", "GOST3411withECGOST3410");
    }

    private void generateDSTU4145(String name, String keyEntryName) throws InvalidAlgorithmParameterException {
        this.generateExtraEC(name, keyEntryName, "DSTU4145", "GOST3411withDSTU4145");
    }

    private void generateRSA(int keySize, String keyEntryName) {
        KeyPairGenerator kpg;
        if (log.isTraceEnabled()) {
            log.trace((Object)(">generate: keySize " + keySize + ", keyEntryName " + keyEntryName));
        }
        try {
            kpg = KeyPairGenerator.getInstance("RSA", this.providerName);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("Algorithm RSAwas not recognized.", e);
        }
        catch (NoSuchProviderException e) {
            throw new IllegalStateException("BouncyCastle was not found as a provider.", e);
        }
        kpg.initialize(keySize);
        this.generateKeyPair(kpg, keyEntryName, "SHA1withRSA");
        if (log.isTraceEnabled()) {
            log.trace((Object)("<generate: keySize " + keySize + ", keyEntryName " + keyEntryName));
        }
    }

    private void generateDSA(int keySize, String keyEntryName) {
        KeyPairGenerator kpg;
        if (log.isTraceEnabled()) {
            log.trace((Object)(">generate: keySize " + keySize + ", keyEntryName " + keyEntryName));
        }
        try {
            kpg = KeyPairGenerator.getInstance("DSA", this.providerName);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("Algorithm DSAwas not recognized.", e);
        }
        catch (NoSuchProviderException e) {
            throw new IllegalStateException("BouncyCastle was not found as a provider.", e);
        }
        kpg.initialize(keySize);
        this.generateKeyPair(kpg, keyEntryName, "SHA1withDSA");
        if (log.isTraceEnabled()) {
            log.trace((Object)("<generate: keySize " + keySize + ", keyEntryName " + keyEntryName));
        }
    }

    public void generateKeyPair(String keySpec, String keyEntryName) throws InvalidAlgorithmParameterException {
        if (keySpec.toUpperCase().startsWith("DSA")) {
            this.generateDSA(Integer.parseInt(keySpec.substring(3).trim()), keyEntryName);
        } else if (AlgorithmTools.isGost3410Enabled() && keySpec.startsWith("GostR3410-")) {
            this.generateGOST3410(keySpec, keyEntryName);
        } else if (AlgorithmTools.isDstu4145Enabled() && keySpec.startsWith(CesecoreConfiguration.getOidDstu4145() + ".")) {
            this.generateDSTU4145(keySpec, keyEntryName);
        } else {
            try {
                this.generateRSA(Integer.parseInt(keySpec.trim()), keyEntryName);
            }
            catch (NumberFormatException e) {
                this.generateEC(keySpec, keyEntryName);
            }
        }
    }

    public void generateKey(String algorithm, int keysize, String keyEntryName) throws NoSuchAlgorithmException, NoSuchProviderException, KeyStoreException {
        KeyGenerator generator = KeyGenerator.getInstance(algorithm, this.providerName);
        generator.init(keysize);
        SecretKey key = generator.generateKey();
        this.setKeyEntry(keyEntryName, key, null);
    }

    public void generateKeyPair(AlgorithmParameterSpec spec, String keyEntryName) throws InvalidAlgorithmParameterException, CertificateException, IOException {
        KeyPairGenerator kpg;
        if (log.isTraceEnabled()) {
            log.trace((Object)(">generate from AlgorithmParameterSpec: " + spec.getClass().getName()));
        }
        String algorithm = "EC";
        String sigAlg = "SHA1withECDSA";
        String specName = spec.getClass().getName();
        if (specName.contains("DSA")) {
            algorithm = "DSA";
            sigAlg = "SHA1withDSA";
        } else if (specName.contains("RSA")) {
            algorithm = "RSA";
            sigAlg = "SHA1withRSA";
        }
        try {
            kpg = KeyPairGenerator.getInstance(algorithm, this.providerName);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("Algorithm " + algorithm + " was not recognized.", e);
        }
        catch (NoSuchProviderException e) {
            throw new IllegalStateException("BouncyCastle was not found as a provider.", e);
        }
        try {
            kpg.initialize(spec);
        }
        catch (InvalidAlgorithmParameterException e) {
            log.debug((Object)("Algorithm parameters not supported: " + e.getMessage()));
            throw e;
        }
        this.generateKeyPair(kpg, keyEntryName, sigAlg);
        if (log.isTraceEnabled()) {
            log.trace((Object)("<generate from AlgorithmParameterSpec: " + spec.getClass().getName()));
        }
    }

    private void generateKeyPair(KeyPairGenerator kpg, String keyEntryName, String sigAlgName) {
        for (int bar = 0; bar < 3; ++bar) {
            try {
                log.debug((Object)"generating...");
                KeyPair keyPair = kpg.generateKeyPair();
                Certificate[] chain = new X509Certificate[]{this.getSelfCertificate("CN=some guy, L=around, C=US", 946080000L, sigAlgName, keyPair)};
                log.debug((Object)("Creating certificate with entry " + keyEntryName + '.'));
                this.setKeyEntry(keyEntryName, keyPair.getPrivate(), chain);
                break;
            }
            catch (KeyStoreException e) {
                log.info((Object)("Failed to generate or store new key, will try 3 times. This was try: " + bar), (Throwable)e);
                continue;
            }
            catch (CertificateException e) {
                throw new KeyCreationException("Can't create keystore because dummy certificate chain creation failed.", e);
            }
            catch (InvalidKeyException e) {
                throw new KeyCreationException("Dummy certificate chain was created with an invalid key", e);
            }
        }
    }

    public void generateCertReq(String alias, String sDN, boolean explicitEccParameters) throws Exception {
        String sigAlg;
        PublicKey publicKey = this.getCertificate(alias).getPublicKey();
        PrivateKey privateKey = this.getPrivateKey(alias);
        if (log.isDebugEnabled()) {
            log.debug((Object)("alias: " + alias + " SHA1 of public key: " + CertTools.getFingerprintAsString(publicKey.getEncoded())));
        }
        if ((sigAlg = AlgorithmTools.getSignatureAlgorithms(publicKey).iterator().next()) == null) {
            sigAlg = "SHA1WithRSA";
        }
        if (sigAlg.contains("ECDSA") && explicitEccParameters) {
            log.info((Object)"Using explicit parameter encoding for ECC key.");
            publicKey = ECKeyUtil.publicToExplicitParameters((PublicKey)publicKey, (String)"BC");
        } else {
            log.info((Object)"Using named curve parameter encoding for ECC key.");
        }
        X500Name sDNName = sDN != null ? new X500Name(sDN) : new X500Name("CN=" + alias);
        PKCS10CertificationRequest certReq = CertTools.genPKCS10CertificationRequest(sigAlg, sDNName, publicKey, (ASN1Set)new DERSet(), privateKey, this.keyStore.getProvider().getName());
        ContentVerifierProvider verifier = CertTools.genContentVerifierProvider(publicKey);
        if (!certReq.isSignatureValid(verifier)) {
            String msg = intres.getLocalizedMessage("token.errorcertreqverify", alias);
            throw new Exception(msg);
        }
        String filename = alias + ".pem";
        FileWriter writer = new FileWriter(filename);
        writer.write("-----BEGIN CERTIFICATE REQUEST-----\n");
        writer.write(new String(Base64.encode(certReq.getEncoded())));
        writer.write("\n-----END CERTIFICATE REQUEST-----\n");
        ((Writer)writer).close();
        log.info((Object)("Wrote csr to file: " + filename));
    }

    public void installCertificate(String fileName) throws Exception {
        Certificate[] chain = CertTools.getCertsFromPEM(new FileInputStream(fileName)).toArray(new X509Certificate[0]);
        PublicKey importPublicKey = chain[0].getPublicKey();
        String importKeyHash = CertTools.getFingerprintAsString(importPublicKey.getEncoded());
        Enumeration<String> eAlias = this.getKeyStore().aliases();
        boolean notFound = true;
        while (eAlias.hasMoreElements() && notFound) {
            String alias = eAlias.nextElement();
            PublicKey hsmPublicKey = this.getCertificate(alias).getPublicKey();
            if (log.isDebugEnabled()) {
                log.debug((Object)("alias: " + alias + " SHA1 of public hsm key: " + CertTools.getFingerprintAsString(hsmPublicKey.getEncoded()) + " SHA1 of first public key in chain: " + importKeyHash + (chain.length == 1 ? "" : "SHA1 of last public key in chain: " + CertTools.getFingerprintAsString(chain[chain.length - 1].getPublicKey().getEncoded()))));
            }
            if (!hsmPublicKey.equals(importPublicKey)) continue;
            log.info((Object)("Found a matching public key for alias \"" + alias + "\"."));
            this.getKeyStore().setKeyEntry(alias, this.getPrivateKey(alias), null, chain);
            notFound = false;
        }
        if (notFound) {
            String msg = intres.getLocalizedMessage("token.errorkeynottoken", importKeyHash);
            throw new Exception(msg);
        }
    }

    public void installTrustedRoot(String fileName) throws Exception {
        X509Certificate[] chain = CertTools.getCertsFromPEM(new FileInputStream(fileName)).toArray(new X509Certificate[0]);
        if (chain.length < 1) {
            throw new Exception("No certificate in file");
        }
        this.getKeyStore().setCertificateEntry("trusted", chain[chain.length - 1]);
    }

    private PrivateKey getPrivateKey(String alias) throws Exception {
        PrivateKey key = (PrivateKey)this.getKey(alias);
        if (key == null) {
            String msg = intres.getLocalizedMessage("token.errornokeyalias", alias);
            log.info((Object)msg);
        }
        return key;
    }

    private Key getKey(String alias) throws Exception, IOException {
        return this.getKeyStore().getKey(alias, null);
    }

    private X509Certificate getCertificate(String alias) throws KeyStoreException {
        X509Certificate cert = (X509Certificate)this.keyStore.getCertificate(alias);
        if (cert == null) {
            String msg = intres.getLocalizedMessage("token.errornocertalias", alias);
            log.info((Object)msg);
        }
        return cert;
    }
}

