package org.elasticsearch.index.analysis;
/*
 * Licensed to ElasticSearch and Shay Banon under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. ElasticSearch licenses this
 * file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import java.util.Arrays;
import java.util.Map;

import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.miscellaneous.KeepWordFilter;
import org.apache.lucene.analysis.util.CharArraySet;
import org.elasticsearch.ElasticSearchIllegalArgumentException;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.inject.assistedinject.Assisted;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.env.Environment;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.settings.IndexSettings;
import org.elasticsearch.indices.analysis.IndicesAnalysisService;

/**
 * A {@link TokenFilterFactory} for {@link KeepWordFilter}. This filter only
 * keep tokens that are contained in the term set configured via
 * {@value #KEEP_WORDS_KEY} setting. This filter acts like an inverse stop
 * filter.
 * 
 * Configuration options:
 * 
 * <ul>
 * <li>{@value #KEEP_WORDS_KEY} the array of words / tokens to keep.</li>
 * 
 * <li>{@value #KEEP_WORDS_PATH_KEY} an reference to a file containing the words
 * / tokens to keep. Note: this is an alternative to {@value #KEEP_WORDS_KEY} if
 * both are set an exception will be thrown.</li>
 * 
 * <li>{@value #ENABLE_POS_INC_KEY} <code>true</code> iff the filter should
 * maintain position increments for dropped tokens. The default is
 * <code>true</code>.</li>
 * 
 * <li>{@value #KEEP_WORDS_CASE_KEY} to use case sensitive keep words. The
 * default is <code>false</code> which corresponds to case-sensitive.</li>
 * </ul>
 * 
 * @see StopTokenFilterFactory
 * 
 */
@AnalysisSettingsRequired
public class KeepWordFilterFactory extends AbstractTokenFilterFactory {
    private Boolean enablePositionIncrements;
    private CharArraySet keepWords;
    private static final String KEEP_WORDS_KEY = "keep_words";
    private static final String KEEP_WORDS_PATH_KEY = KEEP_WORDS_KEY + "_path";
    private static final String KEEP_WORDS_CASE_KEY = KEEP_WORDS_KEY + "_case"; // for javadoc
    private static final String ENABLE_POS_INC_KEY = "enable_position_increments";
    
    @Inject
    public KeepWordFilterFactory(Index index, @IndexSettings Settings indexSettings, 
            Environment env, IndicesAnalysisService indicesAnalysisService, Map<String, TokenizerFactoryFactory> tokenizerFactories,
                                     @Assisted String name, @Assisted Settings settings) {
        super(index, indexSettings, name, settings);
        
        final String[] arrayKeepWords = settings.getAsArray(KEEP_WORDS_KEY);
        final String keepWordsPath = settings.get(KEEP_WORDS_PATH_KEY, null);
        if (!(arrayKeepWords == null ^ keepWordsPath == null)) {
            // we don't allow both or non
            throw new ElasticSearchIllegalArgumentException("keep requires either `" + KEEP_WORDS_KEY + "` or `"
                            + KEEP_WORDS_PATH_KEY + "` to be configured");
        }
        this.enablePositionIncrements = settings.getAsBoolean(ENABLE_POS_INC_KEY, true);
        this.keepWords = Analysis.getWordSet(env, settings, KEEP_WORDS_KEY, version);
        
    }
    
    @Override
    public TokenStream create(TokenStream tokenStream) {
        return new KeepWordFilter(enablePositionIncrements, tokenStream, keepWords);
    }

    
    
   

}
