/*
 * Public Firebird Java API.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *    1. Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the author may not be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.firebirdsql.gds.ng.wire.auth;

import java.sql.SQLException;

/**
 * Plugin for Firebird authentication.
 *
 * @author Mark Rotteveel
 * @since 3.0
 */
public interface AuthenticationPlugin {

    /**
     * @return Name of the plugin as used by Firebird
     */
    String getName();

    /**
     * Authenticate using the plugin.
     * <p>
     * The data generated by this method should be stored and available for calls from {@link #getClientData()}.
     * </p>
     * <p>
     * The method authenticate may be called multiple times depending on the state of the authentication (as indicated
     * by the returned {@link org.firebirdsql.gds.ng.wire.auth.AuthenticationPlugin.AuthStatus}.
     * </p>
     *
     * @param clientAuthBlock
     *         Client authentication block.
     * @return Status of the authentication
     * @throws SQLException
     *         For authentication failures
     */
    AuthStatus authenticate(ClientAuthBlock clientAuthBlock) throws SQLException;

    /**
     * @return The data generated by a call to {@link #authenticate(ClientAuthBlock)}.
     */
    byte[] getClientData();

    /**
     * Sets data received from the server.
     *
     * @param serverData
     *         Server data
     */
    void setServerData(byte[] serverData);

    /**
     * @return {@code true} if plugin has non-null and non-empty server data
     */
    boolean hasServerData();

    /**
     * @return {@code true} if the plugin generates a session key
     */
    boolean generatesSessionKey();

    /**
     * Session key for encryption.
     *
     * @return Session key
     * @throws SQLException
     *         If there is currently no session key or the authentication plugin doesn't support session key
     *         generation.
     */
    byte[] getSessionKey() throws SQLException;

    /**
     * Status of authentication.
     */
    enum AuthStatus {
        /**
         * Authentication completed.
         */
        AUTH_SUCCESS,
// NOTE: Intentionally commented out but kept to compare with FB, authentication failure should be thrown as an SQLException
//        /**
//         * Authentication failed.
//         */
//        AUTH_FAILED,
        /**
         * Plugin is not able to authenticate using the available data, continue with another plugin.
         */
        AUTH_CONTINUE,
        /**
         * The authentication needs more data to complete (e.g. key exchange/handshake).
         */
        AUTH_MORE_DATA
    }
}
