/*
 * Decompiled with CFR 0.152.
 */
package org.firebirdsql.pool;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.firebirdsql.jdbc.FBSQLException;
import org.firebirdsql.logging.Logger;
import org.firebirdsql.pool.ConnectionPoolConfiguration;
import org.firebirdsql.pool.PoolDebugConfiguration;
import org.firebirdsql.pool.PooledConnectionManager;
import org.firebirdsql.pool.PooledObject;
import org.firebirdsql.util.SQLExceptionChainBuilder;

final class PooledConnectionQueue {
    private static final boolean LOG_DEBUG_INFO = PoolDebugConfiguration.LOG_DEBUG_INFO;
    private static final boolean SHOW_STACK_ON_BLOCK = PoolDebugConfiguration.SHOW_TRACE;
    private static final boolean SHOW_STACK_ON_ALLOCATION = PoolDebugConfiguration.SHOW_TRACE;
    private final PooledConnectionManager connectionManager;
    private final Logger logger;
    private final ConnectionPoolConfiguration configuration;
    private final Object key;
    private final String queueName;
    private final int blockingTimeout;
    private final Object addConnectionMutex = new Object();
    private IdleRemover idleRemover;
    private final BlockingQueue<PooledObject> idleConnections;
    private final Set<PooledObject> allConnections;
    private final Set<PooledObject> workingConnections;
    private final Set<PooledObject> workingConnectionsToClose;
    private final AtomicReference<QueueState> queueState = new AtomicReference<QueueState>(QueueState.NEW);

    public PooledConnectionQueue(PooledConnectionManager connectionManager, Logger logger, ConnectionPoolConfiguration configuration, String queueName, Object key) {
        this.connectionManager = connectionManager;
        this.logger = logger;
        this.configuration = configuration;
        this.queueName = queueName;
        this.blockingTimeout = configuration.getBlockingTimeout();
        this.key = key;
        this.idleConnections = new LinkedBlockingQueue<PooledObject>();
        int initialSize = Math.max(16, configuration.getMaxPoolSize());
        this.allConnections = Collections.synchronizedSet(new HashSet(initialSize));
        this.workingConnections = Collections.synchronizedSet(new HashSet(initialSize));
        this.workingConnectionsToClose = Collections.synchronizedSet(new HashSet(initialSize));
    }

    public int size() {
        return this.idleConnections.size();
    }

    public int totalSize() {
        return this.allConnections.size();
    }

    public int workingSize() {
        return this.workingConnections.size();
    }

    public void start() throws SQLException {
        if (!this.queueState.compareAndSet(QueueState.NEW, QueueState.STARTED)) {
            return;
        }
        for (int i = 0; i < this.configuration.getMinPoolSize(); ++i) {
            this.addConnection();
        }
        this.idleRemover = new IdleRemover();
        Thread t = new Thread((Runnable)this.idleRemover, "Pool " + this.queueName + " idleRemover");
        t.setDaemon(true);
        t.start();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void restart() throws SQLException {
        if (!this.queueState.compareAndSet(QueueState.STARTED, QueueState.RESTARTING)) {
            QueueState currentState = this.queueState.get();
            throw new SQLException("Queue " + this.queueName + " cannot be restarted, current state is: " + (Object)((Object)currentState));
        }
        try {
            ArrayList connectionsToClose = new ArrayList(this.size());
            this.idleConnections.drainTo(connectionsToClose);
            this.workingConnectionsToClose.addAll(this.workingConnections);
            for (PooledObject connection : connectionsToClose) {
                try {
                    if (!connection.isValid()) continue;
                    connection.deallocate();
                }
                catch (Exception ex) {
                    if (this.logger == null) continue;
                    this.logger.warn("Could not close connection.", ex);
                }
                finally {
                    this.physicalConnectionDeallocated(connection);
                }
            }
            while (this.totalSize() < this.configuration.getMinPoolSize()) {
                if (this.queueState.get() != QueueState.RESTARTING) {
                    break;
                }
                try {
                    this.addConnection();
                }
                catch (Exception e) {
                    if (this.logger == null) continue;
                    this.logger.warn("Could not add connection.", e);
                }
            }
        }
        finally {
            if (!this.queueState.compareAndSet(QueueState.RESTARTING, QueueState.STARTED) && this.queueState.get() == QueueState.RETRY_SHUTDOWN) {
                this.shutdown();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() throws SQLException {
        QueueState previousState = this.queueState.getAndSet(QueueState.SHUTDOWN);
        switch (previousState) {
            case SHUTDOWN: {
                return;
            }
            case RESTARTING: {
                if (this.queueState.compareAndSet(QueueState.SHUTDOWN, QueueState.RETRY_SHUTDOWN)) {
                    return;
                }
                if (this.queueState.compareAndSet(QueueState.STARTED, QueueState.SHUTDOWN)) break;
                throw new SQLException("Current queue state prevented shutdown. Please retry.");
            }
        }
        if (this.idleRemover != null) {
            this.idleRemover.stop();
        }
        this.idleRemover = null;
        Object object = this.addConnectionMutex;
        synchronized (object) {
            this.idleConnections.clear();
            this.workingConnections.clear();
            this.workingConnectionsToClose.clear();
            for (PooledObject item : this.allConnections) {
                try {
                    if (!item.isValid()) continue;
                    item.deallocate();
                }
                catch (Exception ex) {
                    if (this.logger == null) continue;
                    this.logger.warn("Could not deallocate connection.", ex);
                }
            }
            this.allConnections.clear();
        }
    }

    private boolean keepBlocking(long startTime) {
        return System.currentTimeMillis() - startTime < (long)this.blockingTimeout;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void destroyConnection(PooledObject connection) {
        try {
            connection.deallocate();
        }
        finally {
            this.physicalConnectionDeallocated(connection);
        }
    }

    public void physicalConnectionDeallocated(PooledObject connection) {
        this.allConnections.remove(connection);
        this.workingConnections.remove(connection);
    }

    public void put(PooledObject connection) throws SQLException {
        QueueState currentState = this.queueState.get();
        if (currentState == QueueState.NEW) {
            throw new SQLException("Queue has not been started yet");
        }
        if (!currentState.isAllowPut()) {
            this.destroyConnection(connection);
            if (LOG_DEBUG_INFO && this.logger != null) {
                this.logger.debug("Thread " + Thread.currentThread().getName() + " released connection while pool was in state ." + (Object)((Object)currentState));
            }
            return;
        }
        if (this.configuration.isPooling()) {
            connection.setInPool(true);
            if (this.workingConnectionsToClose.remove(connection)) {
                this.destroyConnection(connection);
                this.addConnection();
            } else if (!this.idleConnections.offer(connection)) {
                this.destroyConnection(connection);
            } else {
                this.workingConnections.remove(connection);
            }
        } else {
            this.destroyConnection(connection);
        }
        if (LOG_DEBUG_INFO && this.logger != null) {
            this.logger.debug("Thread " + Thread.currentThread().getName() + " released connection.");
        }
    }

    public PooledObject take() throws SQLException {
        String message;
        QueueState currentState = this.queueState.get();
        if (!currentState.isAllowTake()) {
            throw new SQLException("Current queue state " + (Object)((Object)currentState) + " does not allow take() on connection queue");
        }
        long startTime = System.currentTimeMillis();
        if (LOG_DEBUG_INFO && this.logger != null) {
            this.logger.debug("Thread " + Thread.currentThread().getName() + " wants to take connection.");
        }
        SQLExceptionChainBuilder<SQLException> pendingExceptions = new SQLExceptionChainBuilder<SQLException>();
        PooledObject result = null;
        try {
            while (true) {
                block19: {
                    if (this.idleConnections.isEmpty()) {
                        try {
                            result = this.createPooledConnection();
                        }
                        catch (SQLException sqlex) {
                            if (this.logger != null) {
                                this.logger.warn("Could not create connection." + sqlex.getMessage());
                            }
                            if (!pendingExceptions.hasException()) {
                                pendingExceptions.append(sqlex);
                            }
                            if (((SQLException)pendingExceptions.getException()).getErrorCode() == sqlex.getErrorCode()) break block19;
                            pendingExceptions.append(sqlex);
                        }
                    }
                }
                if (result == null) {
                    result = this.idleConnections.poll(this.configuration.getRetryInterval(), TimeUnit.MILLISECONDS);
                }
                if (result != null) {
                    if (this.logger != null) {
                        this.logger.info("Obtained connection. Thread " + Thread.currentThread().getName());
                    }
                    result.setInPool(false);
                    this.workingConnections.add(result);
                    break;
                }
                if (!this.keepBlocking(startTime)) {
                    message = "Could not obtain connection during blocking timeout (" + this.blockingTimeout + " ms)";
                    FBSQLException ex = new FBSQLException(message, "08006");
                    if (pendingExceptions.hasException()) {
                        ex.setNextException((SQLException)pendingExceptions.getException());
                    }
                    throw ex;
                }
                message = "Pool " + this.queueName + " is empty and will block here. Thread " + Thread.currentThread().getName();
                if (this.logger == null) continue;
                if (SHOW_STACK_ON_BLOCK) {
                    this.logger.warn(message, new Exception());
                    continue;
                }
                this.logger.warn(message);
            }
        }
        catch (InterruptedException iex) {
            throw new SQLException("No free connection was available and waiting thread was interrupted.");
        }
        if (LOG_DEBUG_INFO && this.logger != null) {
            message = "Thread " + Thread.currentThread().getName() + " got connection.";
            if (SHOW_STACK_ON_ALLOCATION) {
                this.logger.debug(message, new Exception());
            } else {
                this.logger.debug(message);
            }
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean addConnection() throws SQLException {
        if (LOG_DEBUG_INFO && this.logger != null) {
            this.logger.debug("Trying to create connection, total connections " + this.allConnections.size() + ", max allowed " + this.configuration.getMaxPoolSize());
        }
        if (this.idleConnections.remainingCapacity() == 0) {
            if (LOG_DEBUG_INFO && this.logger != null) {
                this.logger.debug("Unable to add more connections, maximum capacity reached.");
            }
            return false;
        }
        Object object = this.addConnectionMutex;
        synchronized (object) {
            if (!this.queueState.get().isAllowAdd()) {
                return false;
            }
            PooledObject pooledConnection = this.createPooledConnection();
            if (pooledConnection == null) {
                if (LOG_DEBUG_INFO && this.logger != null) {
                    this.logger.debug("Unable to add more connections, maximum capacity reached.");
                }
                return false;
            }
            if (LOG_DEBUG_INFO && this.logger != null) {
                this.logger.debug("Thread " + Thread.currentThread().getName() + " created connection.");
            }
            if (this.idleConnections.offer(pooledConnection)) {
                return true;
            }
            this.destroyConnection(pooledConnection);
            if (LOG_DEBUG_INFO && this.logger != null) {
                this.logger.debug("Thread " + Thread.currentThread().getName() + " forced to abandon created connection, capacity reached.");
            }
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private PooledObject createPooledConnection() throws SQLException {
        Object object = this.addConnectionMutex;
        synchronized (object) {
            boolean maximumCapacityReached;
            if (!this.queueState.get().isAllowAdd()) {
                return null;
            }
            boolean bl = maximumCapacityReached = this.configuration.isPooling() && this.configuration.getMaxPoolSize() != 0 && this.configuration.getMaxPoolSize() <= this.totalSize();
            if (maximumCapacityReached) {
                if (LOG_DEBUG_INFO && this.logger != null) {
                    this.logger.debug("Unable to add more connections, maximum capacity reached.");
                }
                return null;
            }
            PooledObject pooledConnection = this.connectionManager.allocateConnection(this.key, this);
            this.allConnections.add(pooledConnection);
            if (LOG_DEBUG_INFO && this.logger != null) {
                this.logger.debug("Thread " + Thread.currentThread().getName() + " created connection.");
            }
            return pooledConnection;
        }
    }

    private boolean releaseNextIdleConnection() throws SQLException {
        if (this.totalSize() <= this.configuration.getMinPoolSize()) {
            return false;
        }
        PooledObject initialCandidate = (PooledObject)this.idleConnections.peek();
        if (initialCandidate == null || initialCandidate.getInstantInPool() >= System.currentTimeMillis() - (long)this.configuration.getMaxIdleTime()) {
            return false;
        }
        PooledObject candidate = (PooledObject)this.idleConnections.poll();
        if (candidate == null) {
            return false;
        }
        long lastUsageTime = candidate.getInstantInPool();
        if (lastUsageTime == -1L && this.idleConnections.offer(candidate)) {
            return false;
        }
        long idleTime = System.currentTimeMillis() - lastUsageTime;
        if (idleTime < (long)this.configuration.getMaxIdleTime() && this.idleConnections.offer(candidate)) {
            return false;
        }
        if (this.logger != null && this.logger.isDebugEnabled()) {
            this.logger.debug(String.format("Going to remove connection with idleTime %d (max is %d)%n", idleTime, this.configuration.getMaxIdleTime()));
        }
        this.destroyConnection(candidate);
        if (this.totalSize() < this.configuration.getMinPoolSize()) {
            this.addConnection();
        }
        return true;
    }

    private static enum QueueState {
        NEW(false, false, false),
        STARTED(true, true, true),
        RESTARTING(true, true, true),
        RETRY_SHUTDOWN(false, false, false),
        SHUTDOWN(false, false, false);

        private final boolean allowPut;
        private final boolean allowTake;
        private final boolean allowAdd;

        private QueueState(boolean allowPut, boolean allowTake, boolean allowAdd) {
            this.allowPut = allowPut;
            this.allowTake = allowTake;
            this.allowAdd = allowAdd;
        }

        public boolean isAllowPut() {
            return this.allowPut;
        }

        public boolean isAllowTake() {
            return this.allowTake;
        }

        public boolean isAllowAdd() {
            return this.allowAdd;
        }
    }

    private class IdleRemover
    implements Runnable {
        private volatile boolean running = true;

        public void stop() {
            this.running = false;
        }

        @Override
        public void run() {
            while (this.running) {
                try {
                    int releasedInIteration = 0;
                    while (PooledConnectionQueue.this.releaseNextIdleConnection()) {
                        ++releasedInIteration;
                    }
                    if (PooledConnectionQueue.this.logger != null && releasedInIteration > 0) {
                        PooledConnectionQueue.this.logger.trace("IdleRemover for " + PooledConnectionQueue.this.queueName + " released " + releasedInIteration + " connections");
                    }
                }
                catch (SQLException ex) {
                    // empty catch block
                }
                try {
                    int idleTimeout = PooledConnectionQueue.this.configuration.getMaxIdleTime();
                    Thread.sleep(Math.max(500, idleTimeout / 4));
                }
                catch (InterruptedException interruptedException) {}
            }
        }
    }
}

