package org.firebirdsql.jdbc;

import org.firebirdsql.gds.impl.GDSHelper;

import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLNonTransientException;
import java.sql.SQLWarning;
import java.util.logging.Logger;

/**
 * JDBC 4.2 implementation of {@link java.sql.Statement}
 *
 * @author <a href="mailto:rrokytskyy@users.sourceforge.net">Roman Rokytskyy</a>
 * @author <a href="mailto:mrotteveel@users.sourceforge.net">Mark Rotteveel</a>
 */
public class FBStatement extends AbstractStatement {

    public FBStatement(GDSHelper c, int rsType, int rsConcurrency, int rsHoldability,
            FBObjectListener.StatementListener statementListener) throws SQLException {
        super(c, rsType, rsConcurrency, rsHoldability, statementListener);
    }
    
    public Logger getParentLogger() throws SQLFeatureNotSupportedException {
        throw new FBDriverNotCapableException();
    }

    /**
     * {@inheritDoc}
     * <p>
     * Jaybird does not support large update counts, the return value of this method is the same as
     * {@link #getUpdateCount()}.
     * </p>
     */
    @Override
    public long getLargeUpdateCount() throws SQLException {
        return getUpdateCount();
    }

    /**
     * {@inheritDoc}
     * <p>
     * Jaybird does not support maxRows exceeding {@link Integer#MAX_VALUE}, if a larger value is set, Jaybird will
     * add a warning to the statement and reset the maximum to 0.
     * </p>
     */
    @Override
    public void setLargeMaxRows(long max) throws SQLException {
        if (max > Integer.MAX_VALUE) {
            addWarning(new SQLWarning(
                    String.format("Implementation limit: maxRows cannot exceed Integer.MAX_VALUE, value was %d, reset to 0", max),
                    FBSQLException.SQL_STATE_INVALID_ARG_VALUE));
            max = 0;
        }
        setMaxRows((int) max);
    }

    /**
     * {@inheritDoc}
     * <p>
     * Jaybird does not support maxRows exceeding {@link Integer#MAX_VALUE}, the return value of this method is the
     * same as {@link #getMaxRows()}.
     * </p>
     */
    @Override
    public long getLargeMaxRows() throws SQLException {
        return getMaxRows();
    }

    /**
     * {@inheritDoc}
     * <p>
     * Jaybird does not support update counts exceeding {@link Integer#MAX_VALUE}, this method calls
     * {@link #executeBatch()} and converts the int-array to a long-array.
     * </p>
     */
    @Override
    public long[] executeLargeBatch() throws SQLException {
        int[] updateCountsInt = executeBatch();
        long[] updateCountsLong = new long[updateCountsInt.length];
        for (int i = 0; i < updateCountsInt.length; i++) {
            updateCountsLong[i] = updateCountsInt[i];
        }
        return updateCountsLong;
    }

    /**
     * {@inheritDoc}
     * <p>
     * Jaybird does not support update counts exceeding {@link Integer#MAX_VALUE}, this method calls
     * {@link #executeUpdate(String)}.
     * </p>
     */
    @Override
    public long executeLargeUpdate(String sql) throws SQLException {
        return executeUpdate(sql);
    }

    /**
     * {@inheritDoc}
     * <p>
     * Jaybird does not support update counts exceeding {@link Integer#MAX_VALUE}, this method calls
     * {@link #executeUpdate(String,int)}.
     * </p>
     */
    @Override
    public long executeLargeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        return executeUpdate(sql, autoGeneratedKeys);
    }

    /**
     * {@inheritDoc}
     * <p>
     * Jaybird does not support update counts exceeding {@link Integer#MAX_VALUE}, this method calls
     * {@link #executeUpdate(String,int[])}.
     * </p>
     */
    @Override
    public long executeLargeUpdate(String sql, int[] columnIndexes) throws SQLException {
        return executeUpdate(sql, columnIndexes);
    }

    /**
     * {@inheritDoc}
     * <p>
     * Jaybird does not support update counts exceeding {@link Integer#MAX_VALUE}, this method calls
     * {@link #executeUpdate(String,String[])}.
     * </p>
     */
    @Override
    public long executeLargeUpdate(String sql, String[] columnNames) throws SQLException {
        return executeUpdate(sql, columnNames);
    }
}
