/*
 * Decompiled with CFR 0.152.
 */
package org.graalvm.compiler.lir.stackslotalloc;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Deque;
import java.util.EnumSet;
import java.util.PriorityQueue;
import java.util.function.Predicate;
import jdk.vm.ci.code.CodeUtil;
import jdk.vm.ci.code.StackSlot;
import jdk.vm.ci.code.TargetDescription;
import jdk.vm.ci.meta.Value;
import jdk.vm.ci.meta.ValueKind;
import org.graalvm.collections.EconomicSet;
import org.graalvm.compiler.core.common.LIRKind;
import org.graalvm.compiler.core.common.cfg.BasicBlock;
import org.graalvm.compiler.debug.DebugCloseable;
import org.graalvm.compiler.debug.DebugContext;
import org.graalvm.compiler.debug.GraalError;
import org.graalvm.compiler.debug.Indent;
import org.graalvm.compiler.debug.TimerKey;
import org.graalvm.compiler.lir.LIR;
import org.graalvm.compiler.lir.LIRInstruction;
import org.graalvm.compiler.lir.LIRValueUtil;
import org.graalvm.compiler.lir.ValueProcedure;
import org.graalvm.compiler.lir.VirtualStackSlot;
import org.graalvm.compiler.lir.framemap.FrameMapBuilderTool;
import org.graalvm.compiler.lir.framemap.SimpleVirtualStackSlot;
import org.graalvm.compiler.lir.framemap.SimpleVirtualStackSlotAlias;
import org.graalvm.compiler.lir.framemap.VirtualStackSlotRange;
import org.graalvm.compiler.lir.gen.LIRGenerationResult;
import org.graalvm.compiler.lir.phases.AllocationPhase;
import org.graalvm.compiler.lir.phases.LIRPhase;
import org.graalvm.compiler.lir.stackslotalloc.FixPointIntervalBuilder;
import org.graalvm.compiler.lir.stackslotalloc.StackInterval;
import org.graalvm.compiler.lir.stackslotalloc.StackIntervalDumper;
import org.graalvm.compiler.lir.stackslotalloc.StackSlotAllocatorUtil;
import org.graalvm.compiler.options.NestedBooleanOptionKey;

public final class LSStackSlotAllocator
extends AllocationPhase {
    private static final TimerKey MainTimer = DebugContext.timer("LSStackSlotAllocator");
    private static final TimerKey NumInstTimer = DebugContext.timer("LSStackSlotAllocator[NumberInstruction]");
    private static final TimerKey BuildIntervalsTimer = DebugContext.timer("LSStackSlotAllocator[BuildIntervals]");
    private static final TimerKey VerifyIntervalsTimer = DebugContext.timer("LSStackSlotAllocator[VerifyIntervals]");
    private static final TimerKey AllocateSlotsTimer = DebugContext.timer("LSStackSlotAllocator[AllocateSlots]");
    private static final TimerKey AssignSlotsTimer = DebugContext.timer("LSStackSlotAllocator[AssignSlots]");

    @Override
    protected void run(TargetDescription target, LIRGenerationResult lirGenRes, AllocationPhase.AllocationContext context) {
        LSStackSlotAllocator.allocateStackSlots((FrameMapBuilderTool)lirGenRes.getFrameMapBuilder(), lirGenRes);
        lirGenRes.buildFrameMap();
    }

    public static void allocateStackSlots(FrameMapBuilderTool builder, LIRGenerationResult res) {
        if (builder.getNumberOfStackSlots() > 0) {
            try (DebugCloseable t = MainTimer.start(res.getLIR().getDebug());){
                new Allocator(res.getLIR(), builder).allocate();
            }
        }
    }

    private static final class Allocator {
        private final LIR lir;
        private final DebugContext debug;
        private final FrameMapBuilderTool frameMapBuilder;
        private final StackInterval[] stackSlotMap;
        private final PriorityQueue<StackInterval> unhandled;
        private final PriorityQueue<StackInterval> active;
        private final BasicBlock<?>[] sortedBlocks;
        private final int maxOpId;
        private static final Predicate<StackInterval> IS_REFERENCE_INTERVAL = new Predicate<StackInterval>(){

            @Override
            public boolean test(StackInterval interval) {
                return !((LIRKind)interval.kind()).isValue();
            }
        };
        private static final Predicate<StackInterval> IS_PRIMITIVE_INTERVAL = new Predicate<StackInterval>(){

            @Override
            public boolean test(StackInterval interval) {
                return ((LIRKind)interval.kind()).isValue();
            }
        };
        private ArrayList<Deque<StackSlot>> freeSlots;
        ValueProcedure assignSlot = new ValueProcedure(){

            @Override
            public Value doValue(Value value, LIRInstruction.OperandMode mode, EnumSet<LIRInstruction.OperandFlag> flags) {
                if (LIRValueUtil.isVirtualStackSlot(value)) {
                    VirtualStackSlot virtualSlot = LIRValueUtil.asVirtualStackSlot(value);
                    StackInterval interval = this.get(virtualSlot);
                    assert (interval != null);
                    StackSlot slot = interval.location();
                    if (virtualSlot instanceof SimpleVirtualStackSlotAlias) {
                        GraalError.guarantee(mode == LIRInstruction.OperandMode.USE || mode == LIRInstruction.OperandMode.ALIVE, "Invalid application of SimpleVirtualStackSlotAlias");
                        return StackSlot.get((ValueKind)virtualSlot.getValueKind(), (int)slot.getRawOffset(), (boolean)slot.getRawAddFrameSize());
                    }
                    return slot;
                }
                return value;
            }
        };

        private Allocator(LIR lir, FrameMapBuilderTool frameMapBuilder) {
            this.lir = lir;
            this.debug = lir.getDebug();
            this.frameMapBuilder = frameMapBuilder;
            this.stackSlotMap = new StackInterval[frameMapBuilder.getNumberOfStackSlots()];
            this.sortedBlocks = lir.getControlFlowGraph().getBlocks();
            this.unhandled = new PriorityQueue((a, b) -> a.from() - b.from());
            this.active = new PriorityQueue((a, b) -> a.to() - b.to());
            try (DebugCloseable t = NumInstTimer.start(this.debug);){
                this.maxOpId = Allocator.numberInstructions(lir, this.sortedBlocks);
            }
        }

        private void allocate() {
            DebugCloseable t;
            EconomicSet<LIRInstruction> usePos;
            this.debug.dump(3, this.lir, "After StackSlot numbering");
            boolean allocationFramesizeEnabled = StackSlotAllocatorUtil.allocatedFramesize.isEnabled(this.debug);
            long currentFrameSize = allocationFramesizeEnabled ? (long)this.frameMapBuilder.getFrameMap().currentFrameSize() : 0L;
            try (DebugContext.Scope s = this.debug.scope("StackSlotAllocationBuildIntervals");
                 Indent indent = this.debug.logAndIndent("BuildIntervals");
                 DebugCloseable t2 = BuildIntervalsTimer.start(this.debug);){
                usePos = this.buildIntervals();
            }
            if (this.debug.areScopesEnabled()) {
                t = VerifyIntervalsTimer.start(this.debug);
                try {
                    assert (this.verifyIntervals());
                }
                finally {
                    if (t != null) {
                        t.close();
                    }
                }
            }
            if (this.debug.isDumpEnabled(3)) {
                this.dumpIntervals("Before stack slot allocation");
            }
            t = AllocateSlotsTimer.start(this.debug);
            try {
                this.allocateStackSlots(IS_PRIMITIVE_INTERVAL);
                this.allocateStackSlots(IS_REFERENCE_INTERVAL);
            }
            finally {
                if (t != null) {
                    t.close();
                }
            }
            if (this.debug.isDumpEnabled(3)) {
                this.dumpIntervals("After stack slot allocation");
            }
            t = AssignSlotsTimer.start(this.debug);
            try {
                this.assignStackSlots(usePos);
            }
            finally {
                if (t != null) {
                    t.close();
                }
            }
            if (allocationFramesizeEnabled) {
                StackSlotAllocatorUtil.allocatedFramesize.add(this.debug, (long)this.frameMapBuilder.getFrameMap().currentFrameSize() - currentFrameSize);
            }
        }

        private static int numberInstructions(LIR lir, BasicBlock<?>[] sortedBlocks) {
            int opId = 0;
            int index = 0;
            for (BasicBlock<?> block : sortedBlocks) {
                ArrayList<LIRInstruction> instructions = lir.getLIRforBlock(block);
                int numInst = instructions.size();
                for (int j = 0; j < numInst; ++j) {
                    LIRInstruction op = instructions.get(j);
                    op.setId(opId);
                    ++index;
                    opId += 2;
                }
            }
            assert (index << 1 == opId) : "must match: " + (index << 1);
            return opId - 2;
        }

        private EconomicSet<LIRInstruction> buildIntervals() {
            return new FixPointIntervalBuilder(this.lir, this.stackSlotMap, this.maxOpId()).build();
        }

        private boolean verifyIntervals() {
            for (StackInterval interval : this.stackSlotMap) {
                if (interval != null) assert (interval.verify(this.maxOpId()));
            }
            return true;
        }

        private void allocateStackSlots(Predicate<StackInterval> predicate) {
            for (StackInterval interval : this.stackSlotMap) {
                if (interval == null || predicate != null && !predicate.test(interval)) continue;
                this.unhandled.add(interval);
            }
            StackInterval current = this.activateNext();
            while (current != null) {
                try (Indent indent = this.debug.logAndIndent("allocate %s", current);){
                    this.allocateSlot(current);
                }
                current = this.activateNext();
            }
            this.freeSlots = null;
            this.active.clear();
        }

        private void allocateSlot(StackInterval current) {
            StackSlot location;
            VirtualStackSlot virtualSlot = current.getOperand();
            if (virtualSlot instanceof VirtualStackSlotRange) {
                VirtualStackSlotRange slotRange = (VirtualStackSlotRange)virtualSlot;
                location = this.frameMapBuilder.getFrameMap().allocateStackMemory(slotRange.getSizeInBytes(), slotRange.getAlignmentInBytes());
                StackSlotAllocatorUtil.virtualFramesize.add(this.debug, slotRange.getSizeInBytes());
                StackSlotAllocatorUtil.allocatedSlots.increment(this.debug);
            } else {
                SimpleVirtualStackSlot simpleSlot;
                if (virtualSlot instanceof SimpleVirtualStackSlot) {
                    simpleSlot = (SimpleVirtualStackSlot)virtualSlot;
                } else if (virtualSlot instanceof SimpleVirtualStackSlotAlias) {
                    simpleSlot = ((SimpleVirtualStackSlotAlias)virtualSlot).getAliasedSlot();
                } else {
                    throw GraalError.shouldNotReachHere("Unexpected VirtualStackSlot type: " + virtualSlot);
                }
                StackSlot slot = this.findFreeSlot(simpleSlot);
                if (slot != null) {
                    location = StackSlot.get(current.kind(), (int)slot.getRawOffset(), (boolean)slot.getRawAddFrameSize());
                    StackSlotAllocatorUtil.reusedSlots.increment(this.debug);
                    this.debug.log(1, "Reuse stack slot %s (reallocated from %s) for virtual stack slot %s", (Object)location, (Object)slot, (Object)virtualSlot);
                } else {
                    ValueKind slotKind = simpleSlot.getValueKind();
                    location = this.frameMapBuilder.getFrameMap().allocateSpillSlot(slotKind);
                    StackSlotAllocatorUtil.virtualFramesize.add(this.debug, this.frameMapBuilder.getFrameMap().spillSlotSize(slotKind));
                    StackSlotAllocatorUtil.allocatedSlots.increment(this.debug);
                    this.debug.log(1, "New stack slot %s for virtual stack slot %s", (Object)location, (Object)virtualSlot);
                }
            }
            this.debug.log("Allocate location %s for interval %s", (Object)location, (Object)current);
            current.setLocation(location);
        }

        private Deque<StackSlot> getNullOrFreeSlots(int index) {
            if (this.freeSlots == null) {
                return null;
            }
            if (index < this.freeSlots.size()) {
                return this.freeSlots.get(index);
            }
            return null;
        }

        private Deque<StackSlot> getOrInitFreeSlots(int index) {
            Deque<StackSlot> freeList = null;
            if (this.freeSlots == null) {
                this.freeSlots = new ArrayList(6);
            } else if (index < this.freeSlots.size()) {
                freeList = this.freeSlots.get(index);
            }
            if (freeList == null) {
                int requiredSize = index + 1;
                for (int i = this.freeSlots.size(); i < requiredSize; ++i) {
                    this.freeSlots.add(null);
                }
                freeList = new ArrayDeque<StackSlot>();
                this.freeSlots.set(index, freeList);
            }
            return freeList;
        }

        private StackSlot findFreeSlot(SimpleVirtualStackSlot slot) {
            assert (slot != null);
            int size = this.log2SpillSlotSize(slot.getValueKind());
            Deque<StackSlot> freeList = this.getNullOrFreeSlots(size);
            if (freeList == null) {
                return null;
            }
            return freeList.pollLast();
        }

        private void freeSlot(StackSlot slot) {
            int size = this.log2SpillSlotSize(slot.getValueKind());
            this.getOrInitFreeSlots(size).addLast(slot);
        }

        private int log2SpillSlotSize(ValueKind<?> kind) {
            int size = this.frameMapBuilder.getFrameMap().spillSlotSize(kind);
            assert (CodeUtil.isPowerOf2((int)size)) : "kind: " + kind + ", size: " + size;
            return CodeUtil.log2((int)size);
        }

        private StackInterval activateNext() {
            if (this.unhandled.isEmpty()) {
                return null;
            }
            StackInterval next = this.unhandled.poll();
            int id = next.from();
            while (this.activePeekId() < id) {
                this.finished(this.active.poll());
            }
            this.debug.log("active %s", next);
            this.active.add(next);
            return next;
        }

        private int activePeekId() {
            StackInterval first = this.active.peek();
            if (first == null) {
                return Integer.MAX_VALUE;
            }
            return first.to();
        }

        private void finished(StackInterval interval) {
            if (interval.getOperand() instanceof VirtualStackSlotRange) {
                this.debug.log("finished %s (not freeing VirtualStackSlotRange)", interval);
            } else {
                StackSlot location = interval.location();
                this.debug.log("finished %s (freeing %s)", (Object)interval, (Object)location);
                this.freeSlot(location);
            }
        }

        private void assignStackSlots(EconomicSet<LIRInstruction> usePos) {
            for (LIRInstruction op : usePos) {
                op.forEachInput(this.assignSlot);
                op.forEachAlive(this.assignSlot);
                op.forEachState(this.assignSlot);
                op.forEachTemp(this.assignSlot);
                op.forEachOutput(this.assignSlot);
            }
        }

        private int maxOpId() {
            return this.maxOpId;
        }

        private StackInterval get(VirtualStackSlot stackSlot) {
            return this.stackSlotMap[stackSlot.getId()];
        }

        private void dumpIntervals(String label) {
            this.debug.dump(3, new StackIntervalDumper(Arrays.copyOf(this.stackSlotMap, this.stackSlotMap.length)), label);
        }
    }

    public static class Options {
        public static final NestedBooleanOptionKey LIROptLSStackSlotAllocator = new NestedBooleanOptionKey(LIRPhase.Options.LIROptimization, true);
    }
}

