/*
 * Copyright (c) 2018, 2020, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.regex.tregex.parser;

import com.oracle.truffle.api.CompilerDirectives.TruffleBoundary;
import com.oracle.truffle.regex.charset.CodePointSet;
import com.oracle.truffle.regex.charset.CodePointSetAccumulator;
import com.oracle.truffle.regex.charset.Range;
import com.oracle.truffle.regex.charset.RangesBuffer;
import com.oracle.truffle.regex.charset.SortedListOfRanges;
import com.oracle.truffle.regex.tregex.util.Exceptions;

public class CaseFoldTable {

    public enum CaseFoldingAlgorithm {
        ECMAScriptNonUnicode,
        ECMAScriptUnicode,
        PythonAscii,
        PythonUnicode
    }

    private static CaseFoldTableImpl getTable(CaseFoldingAlgorithm algorithm) {
        switch (algorithm) {
            case ECMAScriptNonUnicode:
                return NON_UNICODE_TABLE_ENTRIES;
            case ECMAScriptUnicode:
                return UNICODE_TABLE_ENTRIES;
            case PythonAscii:
                return PYTHON_ASCII_TABLE_ENTRIES;
            case PythonUnicode:
                return PYTHON_UNICODE_TABLE_ENTRIES;
            default:
                throw Exceptions.shouldNotReachHere();
        }
    }

    public static void applyCaseFold(CodePointSetAccumulator codePointSet, CodePointSetAccumulator tmp, CaseFoldingAlgorithm algorithm) {
        getTable(algorithm).applyCaseFold(codePointSet, tmp);
    }

    @TruffleBoundary
    public static boolean equalsIgnoreCase(int codePointA, int codePointB, CaseFoldingAlgorithm algorithm) {
        return getTable(algorithm).equalsIgnoreCase(codePointA, codePointB);
    }

    private static CodePointSet rangeSet(int... ranges) {
        return CodePointSet.createNoDedup(ranges);
    }

    private static final int INTEGER_OFFSET = 1;
    private static final int DIRECT_MAPPING = 2;
    private static final int ALTERNATING_UL = 3;
    private static final int ALTERNATING_AL = 4;

    private static final class CaseFoldTableImpl implements SortedListOfRanges {

        private final int[] ranges;

        CaseFoldTableImpl(int[] ranges) {
            this.ranges = ranges;
        }

        void applyCaseFold(CodePointSetAccumulator acc, CodePointSetAccumulator copy) {
            acc.copyTo(copy);
            for (Range r : copy) {
                int search = binarySearch(r.lo);
                if (binarySearchExactMatch(search, r.lo, r.hi)) {
                    apply(acc, search, r.lo, r.hi);
                    continue;
                }
                int firstIntersection = binarySearchGetFirstIntersecting(search, r.lo, r.hi);
                if (binarySearchNoIntersectingFound(firstIntersection)) {
                    continue;
                }
                for (int j = firstIntersection; j < size(); j++) {
                    if (rightOf(j, r.lo, r.hi)) {
                        break;
                    }
                    assert intersects(j, r.lo, r.hi);
                    int intersectionLo = Math.max(getLo(j), r.lo);
                    int intersectionHi = Math.min(getHi(j), r.hi);
                    apply(acc, j, intersectionLo, intersectionHi);
                }
            }
        }

        private void apply(CodePointSetAccumulator codePointSet, int tblEntryIndex, int intersectionLo, int intersectionHi) {
            switch (ranges[tblEntryIndex * 4 + 2]) {
                case INTEGER_OFFSET:
                    int delta = ranges[tblEntryIndex * 4 + 3];
                    codePointSet.addRange(intersectionLo + delta, intersectionHi + delta);
                    break;
                case DIRECT_MAPPING:
                    CodePointSet set = CHARACTER_SET_TABLE[ranges[tblEntryIndex * 4 + 3]];
                    codePointSet.addSet(set);
                    break;
                case ALTERNATING_UL:
                    int loUL = Math.min(((intersectionLo - 1) ^ 1) + 1, ((intersectionHi - 1) ^ 1) + 1);
                    int hiUL = Math.max(((intersectionLo - 1) ^ 1) + 1, ((intersectionHi - 1) ^ 1) + 1);
                    if (!SortedListOfRanges.contains(intersectionLo, intersectionHi, loUL, hiUL)) {
                        codePointSet.addRange(loUL, hiUL);
                    }
                    break;
                case ALTERNATING_AL:
                    int loAL = Math.min(intersectionLo ^ 1, intersectionHi ^ 1);
                    int hiAL = Math.max(intersectionLo ^ 1, intersectionHi ^ 1);
                    if (!SortedListOfRanges.contains(intersectionLo, intersectionHi, loAL, hiAL)) {
                        codePointSet.addRange(loAL, hiAL);
                    }
                    break;
                default:
                    throw Exceptions.shouldNotReachHere();
            }
        }

        boolean equalsIgnoreCase(int codePointA, int codePointB) {
            if (codePointA == codePointB) {
                return true;
            }
            int search = binarySearch(codePointA);
            if (binarySearchExactMatch(search, codePointA, codePointA)) {
                return equalsIgnoreCase(search, codePointA, codePointB);
            }
            int firstIntersection = binarySearchGetFirstIntersecting(search, codePointA, codePointA);
            if (binarySearchNoIntersectingFound(firstIntersection) || rightOf(firstIntersection, codePointA, codePointA)) {
                return false;
            }
            assert intersects(firstIntersection, codePointA, codePointA);
            return equalsIgnoreCase(firstIntersection, codePointA, codePointB);
        }

        private boolean equalsIgnoreCase(int tblEntryIndex, int codePointA, int codePointB) {
            switch (ranges[tblEntryIndex * 4 + 2]) {
                case INTEGER_OFFSET:
                    int delta = ranges[tblEntryIndex * 4 + 3];
                    return codePointA + delta == codePointB;
                case DIRECT_MAPPING:
                    CodePointSet set = CHARACTER_SET_TABLE[ranges[tblEntryIndex * 4 + 3]];
                    return set.contains(codePointB);
                case ALTERNATING_UL:
                    return ((codePointA - 1) ^ 1) + 1 == codePointB;
                case ALTERNATING_AL:
                    return (codePointA ^ 1) == codePointB;
                default:
                    throw Exceptions.shouldNotReachHere();
            }
        }

        @Override
        public int getLo(int i) {
            return ranges[i * 4];
        }

        @Override
        public int getHi(int i) {
            return ranges[i * 4 + 1];
        }

        @Override
        public int size() {
            return ranges.length / 4;
        }

        @Override
        public void appendRangesTo(RangesBuffer buffer, int startIndex, int endIndex) {
            throw Exceptions.shouldNotReachHere();
        }
    }

    /* GENERATED CODE BEGIN - KEEP THIS MARKER FOR AUTOMATIC UPDATES */

    private static final CodePointSet[] CHARACTER_SET_TABLE = new CodePointSet[]{
                    rangeSet(0x0000b5, 0x0000b5, 0x00039c, 0x00039c, 0x0003bc, 0x0003bc),
                    rangeSet(0x0001c4, 0x0001c6),
                    rangeSet(0x0001c7, 0x0001c9),
                    rangeSet(0x0001ca, 0x0001cc),
                    rangeSet(0x0001f1, 0x0001f3),
                    rangeSet(0x000345, 0x000345, 0x000399, 0x000399, 0x0003b9, 0x0003b9, 0x001fbe, 0x001fbe),
                    rangeSet(0x000392, 0x000392, 0x0003b2, 0x0003b2, 0x0003d0, 0x0003d0),
                    rangeSet(0x000395, 0x000395, 0x0003b5, 0x0003b5, 0x0003f5, 0x0003f5),
                    rangeSet(0x000398, 0x000398, 0x0003b8, 0x0003b8, 0x0003d1, 0x0003d1),
                    rangeSet(0x00039a, 0x00039a, 0x0003ba, 0x0003ba, 0x0003f0, 0x0003f0),
                    rangeSet(0x0003a0, 0x0003a0, 0x0003c0, 0x0003c0, 0x0003d6, 0x0003d6),
                    rangeSet(0x0003a1, 0x0003a1, 0x0003c1, 0x0003c1, 0x0003f1, 0x0003f1),
                    rangeSet(0x0003a3, 0x0003a3, 0x0003c2, 0x0003c3),
                    rangeSet(0x0003a6, 0x0003a6, 0x0003c6, 0x0003c6, 0x0003d5, 0x0003d5),
                    rangeSet(0x000412, 0x000412, 0x000432, 0x000432, 0x001c80, 0x001c80),
                    rangeSet(0x000414, 0x000414, 0x000434, 0x000434, 0x001c81, 0x001c81),
                    rangeSet(0x00041e, 0x00041e, 0x00043e, 0x00043e, 0x001c82, 0x001c82),
                    rangeSet(0x000421, 0x000421, 0x000441, 0x000441, 0x001c83, 0x001c83),
                    rangeSet(0x000422, 0x000422, 0x000442, 0x000442, 0x001c84, 0x001c85),
                    rangeSet(0x00042a, 0x00042a, 0x00044a, 0x00044a, 0x001c86, 0x001c86),
                    rangeSet(0x000462, 0x000463, 0x001c87, 0x001c87),
                    rangeSet(0x001c88, 0x001c88, 0x00a64a, 0x00a64b),
                    rangeSet(0x001e60, 0x001e61, 0x001e9b, 0x001e9b),
                    rangeSet(0x00004b, 0x00004b, 0x00006b, 0x00006b, 0x00212a, 0x00212a),
                    rangeSet(0x000053, 0x000053, 0x000073, 0x000073, 0x00017f, 0x00017f),
                    rangeSet(0x0000c5, 0x0000c5, 0x0000e5, 0x0000e5, 0x00212b, 0x00212b),
                    rangeSet(0x000398, 0x000398, 0x0003b8, 0x0003b8, 0x0003d1, 0x0003d1, 0x0003f4, 0x0003f4),
                    rangeSet(0x0003a9, 0x0003a9, 0x0003c9, 0x0003c9, 0x002126, 0x002126),
                    rangeSet(0x000041, 0x000041, 0x000061, 0x000061, 0x001e9a, 0x001e9a),
                    rangeSet(0x000046, 0x000046, 0x000066, 0x000066, 0x00fb00, 0x00fb04),
                    rangeSet(0x000048, 0x000048, 0x000068, 0x000068, 0x001e96, 0x001e96),
                    rangeSet(0x000049, 0x000049, 0x000069, 0x000069, 0x0000cc, 0x0000cd, 0x0000ec, 0x0000ed, 0x000128, 0x000129, 0x000130, 0x000131),
                    rangeSet(0x00004a, 0x00004a, 0x00006a, 0x00006a, 0x0001f0, 0x0001f0),
                    rangeSet(0x000053, 0x000053, 0x000073, 0x000073, 0x0000df, 0x0000df, 0x00017f, 0x00017f, 0x001e9e, 0x001e9e, 0x00fb05, 0x00fb06),
                    rangeSet(0x000054, 0x000054, 0x000074, 0x000074, 0x001e97, 0x001e97),
                    rangeSet(0x000057, 0x000057, 0x000077, 0x000077, 0x001e98, 0x001e98),
                    rangeSet(0x000059, 0x000059, 0x000079, 0x000079, 0x001e99, 0x001e99),
                    rangeSet(0x000345, 0x000345, 0x000390, 0x000390, 0x000399, 0x000399, 0x0003b9, 0x0003b9, 0x001fbe, 0x001fbe, 0x001fd2, 0x001fd3, 0x001fd6, 0x001fd7),
                    rangeSet(0x000386, 0x000386, 0x0003ac, 0x0003ac, 0x001fb4, 0x001fb4),
                    rangeSet(0x000389, 0x000389, 0x0003ae, 0x0003ae, 0x001fc4, 0x001fc4),
                    rangeSet(0x00038f, 0x00038f, 0x0003ce, 0x0003ce, 0x001ff4, 0x001ff4),
                    rangeSet(0x000391, 0x000391, 0x0003b1, 0x0003b1, 0x001fb3, 0x001fb3, 0x001fb6, 0x001fb7, 0x001fbc, 0x001fbc),
                    rangeSet(0x000397, 0x000397, 0x0003b7, 0x0003b7, 0x001fc3, 0x001fc3, 0x001fc6, 0x001fc7, 0x001fcc, 0x001fcc),
                    rangeSet(0x0003a1, 0x0003a1, 0x0003c1, 0x0003c1, 0x0003f1, 0x0003f1, 0x001fe4, 0x001fe4),
                    rangeSet(0x0003a5, 0x0003a5, 0x0003b0, 0x0003b0, 0x0003c5, 0x0003c5, 0x001f50, 0x001f50, 0x001f52, 0x001f52, 0x001f54, 0x001f54, 0x001f56, 0x001f56, 0x001fe2, 0x001fe3, 0x001fe6,
                                    0x001fe7),
                    rangeSet(0x0003a9, 0x0003a9, 0x0003c9, 0x0003c9, 0x001ff3, 0x001ff3, 0x001ff6, 0x001ff7, 0x001ffc, 0x001ffc, 0x002126, 0x002126),
                    rangeSet(0x000535, 0x000535, 0x000565, 0x000565, 0x000587, 0x000587),
                    rangeSet(0x000544, 0x000544, 0x000574, 0x000574, 0x00fb13, 0x00fb15, 0x00fb17, 0x00fb17),
                    rangeSet(0x00054e, 0x00054e, 0x00057e, 0x00057e, 0x00fb16, 0x00fb16),
                    rangeSet(0x001f00, 0x001f00, 0x001f08, 0x001f08, 0x001f80, 0x001f80, 0x001f88, 0x001f88),
                    rangeSet(0x001f01, 0x001f01, 0x001f09, 0x001f09, 0x001f81, 0x001f81, 0x001f89, 0x001f89),
                    rangeSet(0x001f02, 0x001f02, 0x001f0a, 0x001f0a, 0x001f82, 0x001f82, 0x001f8a, 0x001f8a),
                    rangeSet(0x001f03, 0x001f03, 0x001f0b, 0x001f0b, 0x001f83, 0x001f83, 0x001f8b, 0x001f8b),
                    rangeSet(0x001f04, 0x001f04, 0x001f0c, 0x001f0c, 0x001f84, 0x001f84, 0x001f8c, 0x001f8c),
                    rangeSet(0x001f05, 0x001f05, 0x001f0d, 0x001f0d, 0x001f85, 0x001f85, 0x001f8d, 0x001f8d),
                    rangeSet(0x001f06, 0x001f06, 0x001f0e, 0x001f0e, 0x001f86, 0x001f86, 0x001f8e, 0x001f8e),
                    rangeSet(0x001f07, 0x001f07, 0x001f0f, 0x001f0f, 0x001f87, 0x001f87, 0x001f8f, 0x001f8f),
                    rangeSet(0x001f20, 0x001f20, 0x001f28, 0x001f28, 0x001f90, 0x001f90, 0x001f98, 0x001f98),
                    rangeSet(0x001f21, 0x001f21, 0x001f29, 0x001f29, 0x001f91, 0x001f91, 0x001f99, 0x001f99),
                    rangeSet(0x001f22, 0x001f22, 0x001f2a, 0x001f2a, 0x001f92, 0x001f92, 0x001f9a, 0x001f9a),
                    rangeSet(0x001f23, 0x001f23, 0x001f2b, 0x001f2b, 0x001f93, 0x001f93, 0x001f9b, 0x001f9b),
                    rangeSet(0x001f24, 0x001f24, 0x001f2c, 0x001f2c, 0x001f94, 0x001f94, 0x001f9c, 0x001f9c),
                    rangeSet(0x001f25, 0x001f25, 0x001f2d, 0x001f2d, 0x001f95, 0x001f95, 0x001f9d, 0x001f9d),
                    rangeSet(0x001f26, 0x001f26, 0x001f2e, 0x001f2e, 0x001f96, 0x001f96, 0x001f9e, 0x001f9e),
                    rangeSet(0x001f27, 0x001f27, 0x001f2f, 0x001f2f, 0x001f97, 0x001f97, 0x001f9f, 0x001f9f),
                    rangeSet(0x001f60, 0x001f60, 0x001f68, 0x001f68, 0x001fa0, 0x001fa0, 0x001fa8, 0x001fa8),
                    rangeSet(0x001f61, 0x001f61, 0x001f69, 0x001f69, 0x001fa1, 0x001fa1, 0x001fa9, 0x001fa9),
                    rangeSet(0x001f62, 0x001f62, 0x001f6a, 0x001f6a, 0x001fa2, 0x001fa2, 0x001faa, 0x001faa),
                    rangeSet(0x001f63, 0x001f63, 0x001f6b, 0x001f6b, 0x001fa3, 0x001fa3, 0x001fab, 0x001fab),
                    rangeSet(0x001f64, 0x001f64, 0x001f6c, 0x001f6c, 0x001fa4, 0x001fa4, 0x001fac, 0x001fac),
                    rangeSet(0x001f65, 0x001f65, 0x001f6d, 0x001f6d, 0x001fa5, 0x001fa5, 0x001fad, 0x001fad),
                    rangeSet(0x001f66, 0x001f66, 0x001f6e, 0x001f6e, 0x001fa6, 0x001fa6, 0x001fae, 0x001fae),
                    rangeSet(0x001f67, 0x001f67, 0x001f6f, 0x001f6f, 0x001fa7, 0x001fa7, 0x001faf, 0x001faf),
                    rangeSet(0x001f70, 0x001f70, 0x001fb2, 0x001fb2, 0x001fba, 0x001fba),
                    rangeSet(0x001f74, 0x001f74, 0x001fc2, 0x001fc2, 0x001fca, 0x001fca),
                    rangeSet(0x001f7c, 0x001f7c, 0x001ff2, 0x001ff2, 0x001ffa, 0x001ffa)};

    private static final CaseFoldTableImpl NON_UNICODE_TABLE_ENTRIES = new CaseFoldTableImpl(new int[]{
                    0x000041, 0x00005a, INTEGER_OFFSET, 32,
                    0x000061, 0x00007a, INTEGER_OFFSET, -32,
                    0x0000b5, 0x0000b5, DIRECT_MAPPING, 0,
                    0x0000c0, 0x0000d6, INTEGER_OFFSET, 32,
                    0x0000d8, 0x0000de, INTEGER_OFFSET, 32,
                    0x0000e0, 0x0000f6, INTEGER_OFFSET, -32,
                    0x0000f8, 0x0000fe, INTEGER_OFFSET, -32,
                    0x0000ff, 0x0000ff, INTEGER_OFFSET, 121,
                    0x000100, 0x00012f, ALTERNATING_AL, 0,
                    0x000132, 0x000137, ALTERNATING_AL, 0,
                    0x000139, 0x000148, ALTERNATING_UL, 0,
                    0x00014a, 0x000177, ALTERNATING_AL, 0,
                    0x000178, 0x000178, INTEGER_OFFSET, -121,
                    0x000179, 0x00017e, ALTERNATING_UL, 0,
                    0x000180, 0x000180, INTEGER_OFFSET, 195,
                    0x000181, 0x000181, INTEGER_OFFSET, 210,
                    0x000182, 0x000185, ALTERNATING_AL, 0,
                    0x000186, 0x000186, INTEGER_OFFSET, 206,
                    0x000187, 0x000188, ALTERNATING_UL, 0,
                    0x000189, 0x00018a, INTEGER_OFFSET, 205,
                    0x00018b, 0x00018c, ALTERNATING_UL, 0,
                    0x00018e, 0x00018e, INTEGER_OFFSET, 79,
                    0x00018f, 0x00018f, INTEGER_OFFSET, 202,
                    0x000190, 0x000190, INTEGER_OFFSET, 203,
                    0x000191, 0x000192, ALTERNATING_UL, 0,
                    0x000193, 0x000193, INTEGER_OFFSET, 205,
                    0x000194, 0x000194, INTEGER_OFFSET, 207,
                    0x000195, 0x000195, INTEGER_OFFSET, 97,
                    0x000196, 0x000196, INTEGER_OFFSET, 211,
                    0x000197, 0x000197, INTEGER_OFFSET, 209,
                    0x000198, 0x000199, ALTERNATING_AL, 0,
                    0x00019a, 0x00019a, INTEGER_OFFSET, 163,
                    0x00019c, 0x00019c, INTEGER_OFFSET, 211,
                    0x00019d, 0x00019d, INTEGER_OFFSET, 213,
                    0x00019e, 0x00019e, INTEGER_OFFSET, 130,
                    0x00019f, 0x00019f, INTEGER_OFFSET, 214,
                    0x0001a0, 0x0001a5, ALTERNATING_AL, 0,
                    0x0001a6, 0x0001a6, INTEGER_OFFSET, 218,
                    0x0001a7, 0x0001a8, ALTERNATING_UL, 0,
                    0x0001a9, 0x0001a9, INTEGER_OFFSET, 218,
                    0x0001ac, 0x0001ad, ALTERNATING_AL, 0,
                    0x0001ae, 0x0001ae, INTEGER_OFFSET, 218,
                    0x0001af, 0x0001b0, ALTERNATING_UL, 0,
                    0x0001b1, 0x0001b2, INTEGER_OFFSET, 217,
                    0x0001b3, 0x0001b6, ALTERNATING_UL, 0,
                    0x0001b7, 0x0001b7, INTEGER_OFFSET, 219,
                    0x0001b8, 0x0001b9, ALTERNATING_AL, 0,
                    0x0001bc, 0x0001bd, ALTERNATING_AL, 0,
                    0x0001bf, 0x0001bf, INTEGER_OFFSET, 56,
                    0x0001c4, 0x0001c6, DIRECT_MAPPING, 1,
                    0x0001c7, 0x0001c9, DIRECT_MAPPING, 2,
                    0x0001ca, 0x0001cc, DIRECT_MAPPING, 3,
                    0x0001cd, 0x0001dc, ALTERNATING_UL, 0,
                    0x0001dd, 0x0001dd, INTEGER_OFFSET, -79,
                    0x0001de, 0x0001ef, ALTERNATING_AL, 0,
                    0x0001f1, 0x0001f3, DIRECT_MAPPING, 4,
                    0x0001f4, 0x0001f5, ALTERNATING_AL, 0,
                    0x0001f6, 0x0001f6, INTEGER_OFFSET, -97,
                    0x0001f7, 0x0001f7, INTEGER_OFFSET, -56,
                    0x0001f8, 0x00021f, ALTERNATING_AL, 0,
                    0x000220, 0x000220, INTEGER_OFFSET, -130,
                    0x000222, 0x000233, ALTERNATING_AL, 0,
                    0x00023a, 0x00023a, INTEGER_OFFSET, 10795,
                    0x00023b, 0x00023c, ALTERNATING_UL, 0,
                    0x00023d, 0x00023d, INTEGER_OFFSET, -163,
                    0x00023e, 0x00023e, INTEGER_OFFSET, 10792,
                    0x00023f, 0x000240, INTEGER_OFFSET, 10815,
                    0x000241, 0x000242, ALTERNATING_UL, 0,
                    0x000243, 0x000243, INTEGER_OFFSET, -195,
                    0x000244, 0x000244, INTEGER_OFFSET, 69,
                    0x000245, 0x000245, INTEGER_OFFSET, 71,
                    0x000246, 0x00024f, ALTERNATING_AL, 0,
                    0x000250, 0x000250, INTEGER_OFFSET, 10783,
                    0x000251, 0x000251, INTEGER_OFFSET, 10780,
                    0x000252, 0x000252, INTEGER_OFFSET, 10782,
                    0x000253, 0x000253, INTEGER_OFFSET, -210,
                    0x000254, 0x000254, INTEGER_OFFSET, -206,
                    0x000256, 0x000257, INTEGER_OFFSET, -205,
                    0x000259, 0x000259, INTEGER_OFFSET, -202,
                    0x00025b, 0x00025b, INTEGER_OFFSET, -203,
                    0x00025c, 0x00025c, INTEGER_OFFSET, 42319,
                    0x000260, 0x000260, INTEGER_OFFSET, -205,
                    0x000261, 0x000261, INTEGER_OFFSET, 42315,
                    0x000263, 0x000263, INTEGER_OFFSET, -207,
                    0x000265, 0x000265, INTEGER_OFFSET, 42280,
                    0x000266, 0x000266, INTEGER_OFFSET, 42308,
                    0x000268, 0x000268, INTEGER_OFFSET, -209,
                    0x000269, 0x000269, INTEGER_OFFSET, -211,
                    0x00026a, 0x00026a, INTEGER_OFFSET, 42308,
                    0x00026b, 0x00026b, INTEGER_OFFSET, 10743,
                    0x00026c, 0x00026c, INTEGER_OFFSET, 42305,
                    0x00026f, 0x00026f, INTEGER_OFFSET, -211,
                    0x000271, 0x000271, INTEGER_OFFSET, 10749,
                    0x000272, 0x000272, INTEGER_OFFSET, -213,
                    0x000275, 0x000275, INTEGER_OFFSET, -214,
                    0x00027d, 0x00027d, INTEGER_OFFSET, 10727,
                    0x000280, 0x000280, INTEGER_OFFSET, -218,
                    0x000282, 0x000282, INTEGER_OFFSET, 42307,
                    0x000283, 0x000283, INTEGER_OFFSET, -218,
                    0x000287, 0x000287, INTEGER_OFFSET, 42282,
                    0x000288, 0x000288, INTEGER_OFFSET, -218,
                    0x000289, 0x000289, INTEGER_OFFSET, -69,
                    0x00028a, 0x00028b, INTEGER_OFFSET, -217,
                    0x00028c, 0x00028c, INTEGER_OFFSET, -71,
                    0x000292, 0x000292, INTEGER_OFFSET, -219,
                    0x00029d, 0x00029d, INTEGER_OFFSET, 42261,
                    0x00029e, 0x00029e, INTEGER_OFFSET, 42258,
                    0x000345, 0x000345, DIRECT_MAPPING, 5,
                    0x000370, 0x000373, ALTERNATING_AL, 0,
                    0x000376, 0x000377, ALTERNATING_AL, 0,
                    0x00037b, 0x00037d, INTEGER_OFFSET, 130,
                    0x00037f, 0x00037f, INTEGER_OFFSET, 116,
                    0x000386, 0x000386, INTEGER_OFFSET, 38,
                    0x000388, 0x00038a, INTEGER_OFFSET, 37,
                    0x00038c, 0x00038c, INTEGER_OFFSET, 64,
                    0x00038e, 0x00038f, INTEGER_OFFSET, 63,
                    0x000391, 0x000391, INTEGER_OFFSET, 32,
                    0x000392, 0x000392, DIRECT_MAPPING, 6,
                    0x000393, 0x000394, INTEGER_OFFSET, 32,
                    0x000395, 0x000395, DIRECT_MAPPING, 7,
                    0x000396, 0x000397, INTEGER_OFFSET, 32,
                    0x000398, 0x000398, DIRECT_MAPPING, 8,
                    0x000399, 0x000399, DIRECT_MAPPING, 5,
                    0x00039a, 0x00039a, DIRECT_MAPPING, 9,
                    0x00039b, 0x00039b, INTEGER_OFFSET, 32,
                    0x00039c, 0x00039c, DIRECT_MAPPING, 0,
                    0x00039d, 0x00039f, INTEGER_OFFSET, 32,
                    0x0003a0, 0x0003a0, DIRECT_MAPPING, 10,
                    0x0003a1, 0x0003a1, DIRECT_MAPPING, 11,
                    0x0003a3, 0x0003a3, DIRECT_MAPPING, 12,
                    0x0003a4, 0x0003a5, INTEGER_OFFSET, 32,
                    0x0003a6, 0x0003a6, DIRECT_MAPPING, 13,
                    0x0003a7, 0x0003ab, INTEGER_OFFSET, 32,
                    0x0003ac, 0x0003ac, INTEGER_OFFSET, -38,
                    0x0003ad, 0x0003af, INTEGER_OFFSET, -37,
                    0x0003b1, 0x0003b1, INTEGER_OFFSET, -32,
                    0x0003b2, 0x0003b2, DIRECT_MAPPING, 6,
                    0x0003b3, 0x0003b4, INTEGER_OFFSET, -32,
                    0x0003b5, 0x0003b5, DIRECT_MAPPING, 7,
                    0x0003b6, 0x0003b7, INTEGER_OFFSET, -32,
                    0x0003b8, 0x0003b8, DIRECT_MAPPING, 8,
                    0x0003b9, 0x0003b9, DIRECT_MAPPING, 5,
                    0x0003ba, 0x0003ba, DIRECT_MAPPING, 9,
                    0x0003bb, 0x0003bb, INTEGER_OFFSET, -32,
                    0x0003bc, 0x0003bc, DIRECT_MAPPING, 0,
                    0x0003bd, 0x0003bf, INTEGER_OFFSET, -32,
                    0x0003c0, 0x0003c0, DIRECT_MAPPING, 10,
                    0x0003c1, 0x0003c1, DIRECT_MAPPING, 11,
                    0x0003c2, 0x0003c3, DIRECT_MAPPING, 12,
                    0x0003c4, 0x0003c5, INTEGER_OFFSET, -32,
                    0x0003c6, 0x0003c6, DIRECT_MAPPING, 13,
                    0x0003c7, 0x0003cb, INTEGER_OFFSET, -32,
                    0x0003cc, 0x0003cc, INTEGER_OFFSET, -64,
                    0x0003cd, 0x0003ce, INTEGER_OFFSET, -63,
                    0x0003cf, 0x0003cf, INTEGER_OFFSET, 8,
                    0x0003d0, 0x0003d0, DIRECT_MAPPING, 6,
                    0x0003d1, 0x0003d1, DIRECT_MAPPING, 8,
                    0x0003d5, 0x0003d5, DIRECT_MAPPING, 13,
                    0x0003d6, 0x0003d6, DIRECT_MAPPING, 10,
                    0x0003d7, 0x0003d7, INTEGER_OFFSET, -8,
                    0x0003d8, 0x0003ef, ALTERNATING_AL, 0,
                    0x0003f0, 0x0003f0, DIRECT_MAPPING, 9,
                    0x0003f1, 0x0003f1, DIRECT_MAPPING, 11,
                    0x0003f2, 0x0003f2, INTEGER_OFFSET, 7,
                    0x0003f3, 0x0003f3, INTEGER_OFFSET, -116,
                    0x0003f5, 0x0003f5, DIRECT_MAPPING, 7,
                    0x0003f7, 0x0003f8, ALTERNATING_UL, 0,
                    0x0003f9, 0x0003f9, INTEGER_OFFSET, -7,
                    0x0003fa, 0x0003fb, ALTERNATING_AL, 0,
                    0x0003fd, 0x0003ff, INTEGER_OFFSET, -130,
                    0x000400, 0x00040f, INTEGER_OFFSET, 80,
                    0x000410, 0x000411, INTEGER_OFFSET, 32,
                    0x000412, 0x000412, DIRECT_MAPPING, 14,
                    0x000413, 0x000413, INTEGER_OFFSET, 32,
                    0x000414, 0x000414, DIRECT_MAPPING, 15,
                    0x000415, 0x00041d, INTEGER_OFFSET, 32,
                    0x00041e, 0x00041e, DIRECT_MAPPING, 16,
                    0x00041f, 0x000420, INTEGER_OFFSET, 32,
                    0x000421, 0x000421, DIRECT_MAPPING, 17,
                    0x000422, 0x000422, DIRECT_MAPPING, 18,
                    0x000423, 0x000429, INTEGER_OFFSET, 32,
                    0x00042a, 0x00042a, DIRECT_MAPPING, 19,
                    0x00042b, 0x00042f, INTEGER_OFFSET, 32,
                    0x000430, 0x000431, INTEGER_OFFSET, -32,
                    0x000432, 0x000432, DIRECT_MAPPING, 14,
                    0x000433, 0x000433, INTEGER_OFFSET, -32,
                    0x000434, 0x000434, DIRECT_MAPPING, 15,
                    0x000435, 0x00043d, INTEGER_OFFSET, -32,
                    0x00043e, 0x00043e, DIRECT_MAPPING, 16,
                    0x00043f, 0x000440, INTEGER_OFFSET, -32,
                    0x000441, 0x000441, DIRECT_MAPPING, 17,
                    0x000442, 0x000442, DIRECT_MAPPING, 18,
                    0x000443, 0x000449, INTEGER_OFFSET, -32,
                    0x00044a, 0x00044a, DIRECT_MAPPING, 19,
                    0x00044b, 0x00044f, INTEGER_OFFSET, -32,
                    0x000450, 0x00045f, INTEGER_OFFSET, -80,
                    0x000460, 0x000461, ALTERNATING_AL, 0,
                    0x000462, 0x000463, DIRECT_MAPPING, 20,
                    0x000464, 0x000481, ALTERNATING_AL, 0,
                    0x00048a, 0x0004bf, ALTERNATING_AL, 0,
                    0x0004c0, 0x0004c0, INTEGER_OFFSET, 15,
                    0x0004c1, 0x0004ce, ALTERNATING_UL, 0,
                    0x0004cf, 0x0004cf, INTEGER_OFFSET, -15,
                    0x0004d0, 0x00052f, ALTERNATING_AL, 0,
                    0x000531, 0x000556, INTEGER_OFFSET, 48,
                    0x000561, 0x000586, INTEGER_OFFSET, -48,
                    0x0010a0, 0x0010c5, INTEGER_OFFSET, 7264,
                    0x0010c7, 0x0010c7, INTEGER_OFFSET, 7264,
                    0x0010cd, 0x0010cd, INTEGER_OFFSET, 7264,
                    0x0010d0, 0x0010fa, INTEGER_OFFSET, 3008,
                    0x0010fd, 0x0010ff, INTEGER_OFFSET, 3008,
                    0x0013a0, 0x0013ef, INTEGER_OFFSET, 38864,
                    0x0013f0, 0x0013f5, INTEGER_OFFSET, 8,
                    0x0013f8, 0x0013fd, INTEGER_OFFSET, -8,
                    0x001c80, 0x001c80, DIRECT_MAPPING, 14,
                    0x001c81, 0x001c81, DIRECT_MAPPING, 15,
                    0x001c82, 0x001c82, DIRECT_MAPPING, 16,
                    0x001c83, 0x001c83, DIRECT_MAPPING, 17,
                    0x001c84, 0x001c85, DIRECT_MAPPING, 18,
                    0x001c86, 0x001c86, DIRECT_MAPPING, 19,
                    0x001c87, 0x001c87, DIRECT_MAPPING, 20,
                    0x001c88, 0x001c88, DIRECT_MAPPING, 21,
                    0x001c90, 0x001cba, INTEGER_OFFSET, -3008,
                    0x001cbd, 0x001cbf, INTEGER_OFFSET, -3008,
                    0x001d79, 0x001d79, INTEGER_OFFSET, 35332,
                    0x001d7d, 0x001d7d, INTEGER_OFFSET, 3814,
                    0x001d8e, 0x001d8e, INTEGER_OFFSET, 35384,
                    0x001e00, 0x001e5f, ALTERNATING_AL, 0,
                    0x001e60, 0x001e61, DIRECT_MAPPING, 22,
                    0x001e62, 0x001e95, ALTERNATING_AL, 0,
                    0x001e9b, 0x001e9b, DIRECT_MAPPING, 22,
                    0x001ea0, 0x001eff, ALTERNATING_AL, 0,
                    0x001f00, 0x001f07, INTEGER_OFFSET, 8,
                    0x001f08, 0x001f0f, INTEGER_OFFSET, -8,
                    0x001f10, 0x001f15, INTEGER_OFFSET, 8,
                    0x001f18, 0x001f1d, INTEGER_OFFSET, -8,
                    0x001f20, 0x001f27, INTEGER_OFFSET, 8,
                    0x001f28, 0x001f2f, INTEGER_OFFSET, -8,
                    0x001f30, 0x001f37, INTEGER_OFFSET, 8,
                    0x001f38, 0x001f3f, INTEGER_OFFSET, -8,
                    0x001f40, 0x001f45, INTEGER_OFFSET, 8,
                    0x001f48, 0x001f4d, INTEGER_OFFSET, -8,
                    0x001f51, 0x001f51, INTEGER_OFFSET, 8,
                    0x001f53, 0x001f53, INTEGER_OFFSET, 8,
                    0x001f55, 0x001f55, INTEGER_OFFSET, 8,
                    0x001f57, 0x001f57, INTEGER_OFFSET, 8,
                    0x001f59, 0x001f59, INTEGER_OFFSET, -8,
                    0x001f5b, 0x001f5b, INTEGER_OFFSET, -8,
                    0x001f5d, 0x001f5d, INTEGER_OFFSET, -8,
                    0x001f5f, 0x001f5f, INTEGER_OFFSET, -8,
                    0x001f60, 0x001f67, INTEGER_OFFSET, 8,
                    0x001f68, 0x001f6f, INTEGER_OFFSET, -8,
                    0x001f70, 0x001f71, INTEGER_OFFSET, 74,
                    0x001f72, 0x001f75, INTEGER_OFFSET, 86,
                    0x001f76, 0x001f77, INTEGER_OFFSET, 100,
                    0x001f78, 0x001f79, INTEGER_OFFSET, 128,
                    0x001f7a, 0x001f7b, INTEGER_OFFSET, 112,
                    0x001f7c, 0x001f7d, INTEGER_OFFSET, 126,
                    0x001f80, 0x001f87, INTEGER_OFFSET, 8,
                    0x001f88, 0x001f8f, INTEGER_OFFSET, -8,
                    0x001f90, 0x001f97, INTEGER_OFFSET, 8,
                    0x001f98, 0x001f9f, INTEGER_OFFSET, -8,
                    0x001fa0, 0x001fa7, INTEGER_OFFSET, 8,
                    0x001fa8, 0x001faf, INTEGER_OFFSET, -8,
                    0x001fb0, 0x001fb1, INTEGER_OFFSET, 8,
                    0x001fb3, 0x001fb3, INTEGER_OFFSET, 9,
                    0x001fb8, 0x001fb9, INTEGER_OFFSET, -8,
                    0x001fba, 0x001fbb, INTEGER_OFFSET, -74,
                    0x001fbc, 0x001fbc, INTEGER_OFFSET, -9,
                    0x001fbe, 0x001fbe, DIRECT_MAPPING, 5,
                    0x001fc3, 0x001fc3, INTEGER_OFFSET, 9,
                    0x001fc8, 0x001fcb, INTEGER_OFFSET, -86,
                    0x001fcc, 0x001fcc, INTEGER_OFFSET, -9,
                    0x001fd0, 0x001fd1, INTEGER_OFFSET, 8,
                    0x001fd8, 0x001fd9, INTEGER_OFFSET, -8,
                    0x001fda, 0x001fdb, INTEGER_OFFSET, -100,
                    0x001fe0, 0x001fe1, INTEGER_OFFSET, 8,
                    0x001fe5, 0x001fe5, INTEGER_OFFSET, 7,
                    0x001fe8, 0x001fe9, INTEGER_OFFSET, -8,
                    0x001fea, 0x001feb, INTEGER_OFFSET, -112,
                    0x001fec, 0x001fec, INTEGER_OFFSET, -7,
                    0x001ff3, 0x001ff3, INTEGER_OFFSET, 9,
                    0x001ff8, 0x001ff9, INTEGER_OFFSET, -128,
                    0x001ffa, 0x001ffb, INTEGER_OFFSET, -126,
                    0x001ffc, 0x001ffc, INTEGER_OFFSET, -9,
                    0x002132, 0x002132, INTEGER_OFFSET, 28,
                    0x00214e, 0x00214e, INTEGER_OFFSET, -28,
                    0x002160, 0x00216f, INTEGER_OFFSET, 16,
                    0x002170, 0x00217f, INTEGER_OFFSET, -16,
                    0x002183, 0x002184, ALTERNATING_UL, 0,
                    0x0024b6, 0x0024cf, INTEGER_OFFSET, 26,
                    0x0024d0, 0x0024e9, INTEGER_OFFSET, -26,
                    0x002c00, 0x002c2e, INTEGER_OFFSET, 48,
                    0x002c30, 0x002c5e, INTEGER_OFFSET, -48,
                    0x002c60, 0x002c61, ALTERNATING_AL, 0,
                    0x002c62, 0x002c62, INTEGER_OFFSET, -10743,
                    0x002c63, 0x002c63, INTEGER_OFFSET, -3814,
                    0x002c64, 0x002c64, INTEGER_OFFSET, -10727,
                    0x002c65, 0x002c65, INTEGER_OFFSET, -10795,
                    0x002c66, 0x002c66, INTEGER_OFFSET, -10792,
                    0x002c67, 0x002c6c, ALTERNATING_UL, 0,
                    0x002c6d, 0x002c6d, INTEGER_OFFSET, -10780,
                    0x002c6e, 0x002c6e, INTEGER_OFFSET, -10749,
                    0x002c6f, 0x002c6f, INTEGER_OFFSET, -10783,
                    0x002c70, 0x002c70, INTEGER_OFFSET, -10782,
                    0x002c72, 0x002c73, ALTERNATING_AL, 0,
                    0x002c75, 0x002c76, ALTERNATING_UL, 0,
                    0x002c7e, 0x002c7f, INTEGER_OFFSET, -10815,
                    0x002c80, 0x002ce3, ALTERNATING_AL, 0,
                    0x002ceb, 0x002cee, ALTERNATING_UL, 0,
                    0x002cf2, 0x002cf3, ALTERNATING_AL, 0,
                    0x002d00, 0x002d25, INTEGER_OFFSET, -7264,
                    0x002d27, 0x002d27, INTEGER_OFFSET, -7264,
                    0x002d2d, 0x002d2d, INTEGER_OFFSET, -7264,
                    0x00a640, 0x00a649, ALTERNATING_AL, 0,
                    0x00a64a, 0x00a64b, DIRECT_MAPPING, 21,
                    0x00a64c, 0x00a66d, ALTERNATING_AL, 0,
                    0x00a680, 0x00a69b, ALTERNATING_AL, 0,
                    0x00a722, 0x00a72f, ALTERNATING_AL, 0,
                    0x00a732, 0x00a76f, ALTERNATING_AL, 0,
                    0x00a779, 0x00a77c, ALTERNATING_UL, 0,
                    0x00a77d, 0x00a77d, INTEGER_OFFSET, -35332,
                    0x00a77e, 0x00a787, ALTERNATING_AL, 0,
                    0x00a78b, 0x00a78c, ALTERNATING_UL, 0,
                    0x00a78d, 0x00a78d, INTEGER_OFFSET, -42280,
                    0x00a790, 0x00a793, ALTERNATING_AL, 0,
                    0x00a794, 0x00a794, INTEGER_OFFSET, 48,
                    0x00a796, 0x00a7a9, ALTERNATING_AL, 0,
                    0x00a7aa, 0x00a7aa, INTEGER_OFFSET, -42308,
                    0x00a7ab, 0x00a7ab, INTEGER_OFFSET, -42319,
                    0x00a7ac, 0x00a7ac, INTEGER_OFFSET, -42315,
                    0x00a7ad, 0x00a7ad, INTEGER_OFFSET, -42305,
                    0x00a7ae, 0x00a7ae, INTEGER_OFFSET, -42308,
                    0x00a7b0, 0x00a7b0, INTEGER_OFFSET, -42258,
                    0x00a7b1, 0x00a7b1, INTEGER_OFFSET, -42282,
                    0x00a7b2, 0x00a7b2, INTEGER_OFFSET, -42261,
                    0x00a7b3, 0x00a7b3, INTEGER_OFFSET, 928,
                    0x00a7b4, 0x00a7bf, ALTERNATING_AL, 0,
                    0x00a7c2, 0x00a7c3, ALTERNATING_AL, 0,
                    0x00a7c4, 0x00a7c4, INTEGER_OFFSET, -48,
                    0x00a7c5, 0x00a7c5, INTEGER_OFFSET, -42307,
                    0x00a7c6, 0x00a7c6, INTEGER_OFFSET, -35384,
                    0x00a7c7, 0x00a7ca, ALTERNATING_UL, 0,
                    0x00a7f5, 0x00a7f6, ALTERNATING_UL, 0,
                    0x00ab53, 0x00ab53, INTEGER_OFFSET, -928,
                    0x00ab70, 0x00abbf, INTEGER_OFFSET, -38864,
                    0x00ff21, 0x00ff3a, INTEGER_OFFSET, 32,
                    0x00ff41, 0x00ff5a, INTEGER_OFFSET, -32,
                    0x010400, 0x010427, INTEGER_OFFSET, 40,
                    0x010428, 0x01044f, INTEGER_OFFSET, -40,
                    0x0104b0, 0x0104d3, INTEGER_OFFSET, 40,
                    0x0104d8, 0x0104fb, INTEGER_OFFSET, -40,
                    0x010c80, 0x010cb2, INTEGER_OFFSET, 64,
                    0x010cc0, 0x010cf2, INTEGER_OFFSET, -64,
                    0x0118a0, 0x0118bf, INTEGER_OFFSET, 32,
                    0x0118c0, 0x0118df, INTEGER_OFFSET, -32,
                    0x016e40, 0x016e5f, INTEGER_OFFSET, 32,
                    0x016e60, 0x016e7f, INTEGER_OFFSET, -32,
                    0x01e900, 0x01e921, INTEGER_OFFSET, 34,
                    0x01e922, 0x01e943, INTEGER_OFFSET, -34
    });

    private static final CaseFoldTableImpl UNICODE_TABLE_ENTRIES = new CaseFoldTableImpl(new int[]{
                    0x000041, 0x00004a, INTEGER_OFFSET, 32,
                    0x00004b, 0x00004b, DIRECT_MAPPING, 23,
                    0x00004c, 0x000052, INTEGER_OFFSET, 32,
                    0x000053, 0x000053, DIRECT_MAPPING, 24,
                    0x000054, 0x00005a, INTEGER_OFFSET, 32,
                    0x000061, 0x00006a, INTEGER_OFFSET, -32,
                    0x00006b, 0x00006b, DIRECT_MAPPING, 23,
                    0x00006c, 0x000072, INTEGER_OFFSET, -32,
                    0x000073, 0x000073, DIRECT_MAPPING, 24,
                    0x000074, 0x00007a, INTEGER_OFFSET, -32,
                    0x0000b5, 0x0000b5, DIRECT_MAPPING, 0,
                    0x0000c0, 0x0000c4, INTEGER_OFFSET, 32,
                    0x0000c5, 0x0000c5, DIRECT_MAPPING, 25,
                    0x0000c6, 0x0000d6, INTEGER_OFFSET, 32,
                    0x0000d8, 0x0000de, INTEGER_OFFSET, 32,
                    0x0000df, 0x0000df, INTEGER_OFFSET, 7615,
                    0x0000e0, 0x0000e4, INTEGER_OFFSET, -32,
                    0x0000e5, 0x0000e5, DIRECT_MAPPING, 25,
                    0x0000e6, 0x0000f6, INTEGER_OFFSET, -32,
                    0x0000f8, 0x0000fe, INTEGER_OFFSET, -32,
                    0x0000ff, 0x0000ff, INTEGER_OFFSET, 121,
                    0x000100, 0x00012f, ALTERNATING_AL, 0,
                    0x000132, 0x000137, ALTERNATING_AL, 0,
                    0x000139, 0x000148, ALTERNATING_UL, 0,
                    0x00014a, 0x000177, ALTERNATING_AL, 0,
                    0x000178, 0x000178, INTEGER_OFFSET, -121,
                    0x000179, 0x00017e, ALTERNATING_UL, 0,
                    0x00017f, 0x00017f, DIRECT_MAPPING, 24,
                    0x000180, 0x000180, INTEGER_OFFSET, 195,
                    0x000181, 0x000181, INTEGER_OFFSET, 210,
                    0x000182, 0x000185, ALTERNATING_AL, 0,
                    0x000186, 0x000186, INTEGER_OFFSET, 206,
                    0x000187, 0x000188, ALTERNATING_UL, 0,
                    0x000189, 0x00018a, INTEGER_OFFSET, 205,
                    0x00018b, 0x00018c, ALTERNATING_UL, 0,
                    0x00018e, 0x00018e, INTEGER_OFFSET, 79,
                    0x00018f, 0x00018f, INTEGER_OFFSET, 202,
                    0x000190, 0x000190, INTEGER_OFFSET, 203,
                    0x000191, 0x000192, ALTERNATING_UL, 0,
                    0x000193, 0x000193, INTEGER_OFFSET, 205,
                    0x000194, 0x000194, INTEGER_OFFSET, 207,
                    0x000195, 0x000195, INTEGER_OFFSET, 97,
                    0x000196, 0x000196, INTEGER_OFFSET, 211,
                    0x000197, 0x000197, INTEGER_OFFSET, 209,
                    0x000198, 0x000199, ALTERNATING_AL, 0,
                    0x00019a, 0x00019a, INTEGER_OFFSET, 163,
                    0x00019c, 0x00019c, INTEGER_OFFSET, 211,
                    0x00019d, 0x00019d, INTEGER_OFFSET, 213,
                    0x00019e, 0x00019e, INTEGER_OFFSET, 130,
                    0x00019f, 0x00019f, INTEGER_OFFSET, 214,
                    0x0001a0, 0x0001a5, ALTERNATING_AL, 0,
                    0x0001a6, 0x0001a6, INTEGER_OFFSET, 218,
                    0x0001a7, 0x0001a8, ALTERNATING_UL, 0,
                    0x0001a9, 0x0001a9, INTEGER_OFFSET, 218,
                    0x0001ac, 0x0001ad, ALTERNATING_AL, 0,
                    0x0001ae, 0x0001ae, INTEGER_OFFSET, 218,
                    0x0001af, 0x0001b0, ALTERNATING_UL, 0,
                    0x0001b1, 0x0001b2, INTEGER_OFFSET, 217,
                    0x0001b3, 0x0001b6, ALTERNATING_UL, 0,
                    0x0001b7, 0x0001b7, INTEGER_OFFSET, 219,
                    0x0001b8, 0x0001b9, ALTERNATING_AL, 0,
                    0x0001bc, 0x0001bd, ALTERNATING_AL, 0,
                    0x0001bf, 0x0001bf, INTEGER_OFFSET, 56,
                    0x0001c4, 0x0001c6, DIRECT_MAPPING, 1,
                    0x0001c7, 0x0001c9, DIRECT_MAPPING, 2,
                    0x0001ca, 0x0001cc, DIRECT_MAPPING, 3,
                    0x0001cd, 0x0001dc, ALTERNATING_UL, 0,
                    0x0001dd, 0x0001dd, INTEGER_OFFSET, -79,
                    0x0001de, 0x0001ef, ALTERNATING_AL, 0,
                    0x0001f1, 0x0001f3, DIRECT_MAPPING, 4,
                    0x0001f4, 0x0001f5, ALTERNATING_AL, 0,
                    0x0001f6, 0x0001f6, INTEGER_OFFSET, -97,
                    0x0001f7, 0x0001f7, INTEGER_OFFSET, -56,
                    0x0001f8, 0x00021f, ALTERNATING_AL, 0,
                    0x000220, 0x000220, INTEGER_OFFSET, -130,
                    0x000222, 0x000233, ALTERNATING_AL, 0,
                    0x00023a, 0x00023a, INTEGER_OFFSET, 10795,
                    0x00023b, 0x00023c, ALTERNATING_UL, 0,
                    0x00023d, 0x00023d, INTEGER_OFFSET, -163,
                    0x00023e, 0x00023e, INTEGER_OFFSET, 10792,
                    0x00023f, 0x000240, INTEGER_OFFSET, 10815,
                    0x000241, 0x000242, ALTERNATING_UL, 0,
                    0x000243, 0x000243, INTEGER_OFFSET, -195,
                    0x000244, 0x000244, INTEGER_OFFSET, 69,
                    0x000245, 0x000245, INTEGER_OFFSET, 71,
                    0x000246, 0x00024f, ALTERNATING_AL, 0,
                    0x000250, 0x000250, INTEGER_OFFSET, 10783,
                    0x000251, 0x000251, INTEGER_OFFSET, 10780,
                    0x000252, 0x000252, INTEGER_OFFSET, 10782,
                    0x000253, 0x000253, INTEGER_OFFSET, -210,
                    0x000254, 0x000254, INTEGER_OFFSET, -206,
                    0x000256, 0x000257, INTEGER_OFFSET, -205,
                    0x000259, 0x000259, INTEGER_OFFSET, -202,
                    0x00025b, 0x00025b, INTEGER_OFFSET, -203,
                    0x00025c, 0x00025c, INTEGER_OFFSET, 42319,
                    0x000260, 0x000260, INTEGER_OFFSET, -205,
                    0x000261, 0x000261, INTEGER_OFFSET, 42315,
                    0x000263, 0x000263, INTEGER_OFFSET, -207,
                    0x000265, 0x000265, INTEGER_OFFSET, 42280,
                    0x000266, 0x000266, INTEGER_OFFSET, 42308,
                    0x000268, 0x000268, INTEGER_OFFSET, -209,
                    0x000269, 0x000269, INTEGER_OFFSET, -211,
                    0x00026a, 0x00026a, INTEGER_OFFSET, 42308,
                    0x00026b, 0x00026b, INTEGER_OFFSET, 10743,
                    0x00026c, 0x00026c, INTEGER_OFFSET, 42305,
                    0x00026f, 0x00026f, INTEGER_OFFSET, -211,
                    0x000271, 0x000271, INTEGER_OFFSET, 10749,
                    0x000272, 0x000272, INTEGER_OFFSET, -213,
                    0x000275, 0x000275, INTEGER_OFFSET, -214,
                    0x00027d, 0x00027d, INTEGER_OFFSET, 10727,
                    0x000280, 0x000280, INTEGER_OFFSET, -218,
                    0x000282, 0x000282, INTEGER_OFFSET, 42307,
                    0x000283, 0x000283, INTEGER_OFFSET, -218,
                    0x000287, 0x000287, INTEGER_OFFSET, 42282,
                    0x000288, 0x000288, INTEGER_OFFSET, -218,
                    0x000289, 0x000289, INTEGER_OFFSET, -69,
                    0x00028a, 0x00028b, INTEGER_OFFSET, -217,
                    0x00028c, 0x00028c, INTEGER_OFFSET, -71,
                    0x000292, 0x000292, INTEGER_OFFSET, -219,
                    0x00029d, 0x00029d, INTEGER_OFFSET, 42261,
                    0x00029e, 0x00029e, INTEGER_OFFSET, 42258,
                    0x000345, 0x000345, DIRECT_MAPPING, 5,
                    0x000370, 0x000373, ALTERNATING_AL, 0,
                    0x000376, 0x000377, ALTERNATING_AL, 0,
                    0x00037b, 0x00037d, INTEGER_OFFSET, 130,
                    0x00037f, 0x00037f, INTEGER_OFFSET, 116,
                    0x000386, 0x000386, INTEGER_OFFSET, 38,
                    0x000388, 0x00038a, INTEGER_OFFSET, 37,
                    0x00038c, 0x00038c, INTEGER_OFFSET, 64,
                    0x00038e, 0x00038f, INTEGER_OFFSET, 63,
                    0x000391, 0x000391, INTEGER_OFFSET, 32,
                    0x000392, 0x000392, DIRECT_MAPPING, 6,
                    0x000393, 0x000394, INTEGER_OFFSET, 32,
                    0x000395, 0x000395, DIRECT_MAPPING, 7,
                    0x000396, 0x000397, INTEGER_OFFSET, 32,
                    0x000398, 0x000398, DIRECT_MAPPING, 26,
                    0x000399, 0x000399, DIRECT_MAPPING, 5,
                    0x00039a, 0x00039a, DIRECT_MAPPING, 9,
                    0x00039b, 0x00039b, INTEGER_OFFSET, 32,
                    0x00039c, 0x00039c, DIRECT_MAPPING, 0,
                    0x00039d, 0x00039f, INTEGER_OFFSET, 32,
                    0x0003a0, 0x0003a0, DIRECT_MAPPING, 10,
                    0x0003a1, 0x0003a1, DIRECT_MAPPING, 11,
                    0x0003a3, 0x0003a3, DIRECT_MAPPING, 12,
                    0x0003a4, 0x0003a5, INTEGER_OFFSET, 32,
                    0x0003a6, 0x0003a6, DIRECT_MAPPING, 13,
                    0x0003a7, 0x0003a8, INTEGER_OFFSET, 32,
                    0x0003a9, 0x0003a9, DIRECT_MAPPING, 27,
                    0x0003aa, 0x0003ab, INTEGER_OFFSET, 32,
                    0x0003ac, 0x0003ac, INTEGER_OFFSET, -38,
                    0x0003ad, 0x0003af, INTEGER_OFFSET, -37,
                    0x0003b1, 0x0003b1, INTEGER_OFFSET, -32,
                    0x0003b2, 0x0003b2, DIRECT_MAPPING, 6,
                    0x0003b3, 0x0003b4, INTEGER_OFFSET, -32,
                    0x0003b5, 0x0003b5, DIRECT_MAPPING, 7,
                    0x0003b6, 0x0003b7, INTEGER_OFFSET, -32,
                    0x0003b8, 0x0003b8, DIRECT_MAPPING, 26,
                    0x0003b9, 0x0003b9, DIRECT_MAPPING, 5,
                    0x0003ba, 0x0003ba, DIRECT_MAPPING, 9,
                    0x0003bb, 0x0003bb, INTEGER_OFFSET, -32,
                    0x0003bc, 0x0003bc, DIRECT_MAPPING, 0,
                    0x0003bd, 0x0003bf, INTEGER_OFFSET, -32,
                    0x0003c0, 0x0003c0, DIRECT_MAPPING, 10,
                    0x0003c1, 0x0003c1, DIRECT_MAPPING, 11,
                    0x0003c2, 0x0003c3, DIRECT_MAPPING, 12,
                    0x0003c4, 0x0003c5, INTEGER_OFFSET, -32,
                    0x0003c6, 0x0003c6, DIRECT_MAPPING, 13,
                    0x0003c7, 0x0003c8, INTEGER_OFFSET, -32,
                    0x0003c9, 0x0003c9, DIRECT_MAPPING, 27,
                    0x0003ca, 0x0003cb, INTEGER_OFFSET, -32,
                    0x0003cc, 0x0003cc, INTEGER_OFFSET, -64,
                    0x0003cd, 0x0003ce, INTEGER_OFFSET, -63,
                    0x0003cf, 0x0003cf, INTEGER_OFFSET, 8,
                    0x0003d0, 0x0003d0, DIRECT_MAPPING, 6,
                    0x0003d1, 0x0003d1, DIRECT_MAPPING, 26,
                    0x0003d5, 0x0003d5, DIRECT_MAPPING, 13,
                    0x0003d6, 0x0003d6, DIRECT_MAPPING, 10,
                    0x0003d7, 0x0003d7, INTEGER_OFFSET, -8,
                    0x0003d8, 0x0003ef, ALTERNATING_AL, 0,
                    0x0003f0, 0x0003f0, DIRECT_MAPPING, 9,
                    0x0003f1, 0x0003f1, DIRECT_MAPPING, 11,
                    0x0003f2, 0x0003f2, INTEGER_OFFSET, 7,
                    0x0003f3, 0x0003f3, INTEGER_OFFSET, -116,
                    0x0003f4, 0x0003f4, DIRECT_MAPPING, 26,
                    0x0003f5, 0x0003f5, DIRECT_MAPPING, 7,
                    0x0003f7, 0x0003f8, ALTERNATING_UL, 0,
                    0x0003f9, 0x0003f9, INTEGER_OFFSET, -7,
                    0x0003fa, 0x0003fb, ALTERNATING_AL, 0,
                    0x0003fd, 0x0003ff, INTEGER_OFFSET, -130,
                    0x000400, 0x00040f, INTEGER_OFFSET, 80,
                    0x000410, 0x000411, INTEGER_OFFSET, 32,
                    0x000412, 0x000412, DIRECT_MAPPING, 14,
                    0x000413, 0x000413, INTEGER_OFFSET, 32,
                    0x000414, 0x000414, DIRECT_MAPPING, 15,
                    0x000415, 0x00041d, INTEGER_OFFSET, 32,
                    0x00041e, 0x00041e, DIRECT_MAPPING, 16,
                    0x00041f, 0x000420, INTEGER_OFFSET, 32,
                    0x000421, 0x000421, DIRECT_MAPPING, 17,
                    0x000422, 0x000422, DIRECT_MAPPING, 18,
                    0x000423, 0x000429, INTEGER_OFFSET, 32,
                    0x00042a, 0x00042a, DIRECT_MAPPING, 19,
                    0x00042b, 0x00042f, INTEGER_OFFSET, 32,
                    0x000430, 0x000431, INTEGER_OFFSET, -32,
                    0x000432, 0x000432, DIRECT_MAPPING, 14,
                    0x000433, 0x000433, INTEGER_OFFSET, -32,
                    0x000434, 0x000434, DIRECT_MAPPING, 15,
                    0x000435, 0x00043d, INTEGER_OFFSET, -32,
                    0x00043e, 0x00043e, DIRECT_MAPPING, 16,
                    0x00043f, 0x000440, INTEGER_OFFSET, -32,
                    0x000441, 0x000441, DIRECT_MAPPING, 17,
                    0x000442, 0x000442, DIRECT_MAPPING, 18,
                    0x000443, 0x000449, INTEGER_OFFSET, -32,
                    0x00044a, 0x00044a, DIRECT_MAPPING, 19,
                    0x00044b, 0x00044f, INTEGER_OFFSET, -32,
                    0x000450, 0x00045f, INTEGER_OFFSET, -80,
                    0x000460, 0x000461, ALTERNATING_AL, 0,
                    0x000462, 0x000463, DIRECT_MAPPING, 20,
                    0x000464, 0x000481, ALTERNATING_AL, 0,
                    0x00048a, 0x0004bf, ALTERNATING_AL, 0,
                    0x0004c0, 0x0004c0, INTEGER_OFFSET, 15,
                    0x0004c1, 0x0004ce, ALTERNATING_UL, 0,
                    0x0004cf, 0x0004cf, INTEGER_OFFSET, -15,
                    0x0004d0, 0x00052f, ALTERNATING_AL, 0,
                    0x000531, 0x000556, INTEGER_OFFSET, 48,
                    0x000561, 0x000586, INTEGER_OFFSET, -48,
                    0x0010a0, 0x0010c5, INTEGER_OFFSET, 7264,
                    0x0010c7, 0x0010c7, INTEGER_OFFSET, 7264,
                    0x0010cd, 0x0010cd, INTEGER_OFFSET, 7264,
                    0x0010d0, 0x0010fa, INTEGER_OFFSET, 3008,
                    0x0010fd, 0x0010ff, INTEGER_OFFSET, 3008,
                    0x0013a0, 0x0013ef, INTEGER_OFFSET, 38864,
                    0x0013f0, 0x0013f5, INTEGER_OFFSET, 8,
                    0x0013f8, 0x0013fd, INTEGER_OFFSET, -8,
                    0x001c80, 0x001c80, DIRECT_MAPPING, 14,
                    0x001c81, 0x001c81, DIRECT_MAPPING, 15,
                    0x001c82, 0x001c82, DIRECT_MAPPING, 16,
                    0x001c83, 0x001c83, DIRECT_MAPPING, 17,
                    0x001c84, 0x001c85, DIRECT_MAPPING, 18,
                    0x001c86, 0x001c86, DIRECT_MAPPING, 19,
                    0x001c87, 0x001c87, DIRECT_MAPPING, 20,
                    0x001c88, 0x001c88, DIRECT_MAPPING, 21,
                    0x001c90, 0x001cba, INTEGER_OFFSET, -3008,
                    0x001cbd, 0x001cbf, INTEGER_OFFSET, -3008,
                    0x001d79, 0x001d79, INTEGER_OFFSET, 35332,
                    0x001d7d, 0x001d7d, INTEGER_OFFSET, 3814,
                    0x001d8e, 0x001d8e, INTEGER_OFFSET, 35384,
                    0x001e00, 0x001e5f, ALTERNATING_AL, 0,
                    0x001e60, 0x001e61, DIRECT_MAPPING, 22,
                    0x001e62, 0x001e95, ALTERNATING_AL, 0,
                    0x001e9b, 0x001e9b, DIRECT_MAPPING, 22,
                    0x001e9e, 0x001e9e, INTEGER_OFFSET, -7615,
                    0x001ea0, 0x001eff, ALTERNATING_AL, 0,
                    0x001f00, 0x001f07, INTEGER_OFFSET, 8,
                    0x001f08, 0x001f0f, INTEGER_OFFSET, -8,
                    0x001f10, 0x001f15, INTEGER_OFFSET, 8,
                    0x001f18, 0x001f1d, INTEGER_OFFSET, -8,
                    0x001f20, 0x001f27, INTEGER_OFFSET, 8,
                    0x001f28, 0x001f2f, INTEGER_OFFSET, -8,
                    0x001f30, 0x001f37, INTEGER_OFFSET, 8,
                    0x001f38, 0x001f3f, INTEGER_OFFSET, -8,
                    0x001f40, 0x001f45, INTEGER_OFFSET, 8,
                    0x001f48, 0x001f4d, INTEGER_OFFSET, -8,
                    0x001f51, 0x001f51, INTEGER_OFFSET, 8,
                    0x001f53, 0x001f53, INTEGER_OFFSET, 8,
                    0x001f55, 0x001f55, INTEGER_OFFSET, 8,
                    0x001f57, 0x001f57, INTEGER_OFFSET, 8,
                    0x001f59, 0x001f59, INTEGER_OFFSET, -8,
                    0x001f5b, 0x001f5b, INTEGER_OFFSET, -8,
                    0x001f5d, 0x001f5d, INTEGER_OFFSET, -8,
                    0x001f5f, 0x001f5f, INTEGER_OFFSET, -8,
                    0x001f60, 0x001f67, INTEGER_OFFSET, 8,
                    0x001f68, 0x001f6f, INTEGER_OFFSET, -8,
                    0x001f70, 0x001f71, INTEGER_OFFSET, 74,
                    0x001f72, 0x001f75, INTEGER_OFFSET, 86,
                    0x001f76, 0x001f77, INTEGER_OFFSET, 100,
                    0x001f78, 0x001f79, INTEGER_OFFSET, 128,
                    0x001f7a, 0x001f7b, INTEGER_OFFSET, 112,
                    0x001f7c, 0x001f7d, INTEGER_OFFSET, 126,
                    0x001f80, 0x001f87, INTEGER_OFFSET, 8,
                    0x001f88, 0x001f8f, INTEGER_OFFSET, -8,
                    0x001f90, 0x001f97, INTEGER_OFFSET, 8,
                    0x001f98, 0x001f9f, INTEGER_OFFSET, -8,
                    0x001fa0, 0x001fa7, INTEGER_OFFSET, 8,
                    0x001fa8, 0x001faf, INTEGER_OFFSET, -8,
                    0x001fb0, 0x001fb1, INTEGER_OFFSET, 8,
                    0x001fb3, 0x001fb3, INTEGER_OFFSET, 9,
                    0x001fb8, 0x001fb9, INTEGER_OFFSET, -8,
                    0x001fba, 0x001fbb, INTEGER_OFFSET, -74,
                    0x001fbc, 0x001fbc, INTEGER_OFFSET, -9,
                    0x001fbe, 0x001fbe, DIRECT_MAPPING, 5,
                    0x001fc3, 0x001fc3, INTEGER_OFFSET, 9,
                    0x001fc8, 0x001fcb, INTEGER_OFFSET, -86,
                    0x001fcc, 0x001fcc, INTEGER_OFFSET, -9,
                    0x001fd0, 0x001fd1, INTEGER_OFFSET, 8,
                    0x001fd8, 0x001fd9, INTEGER_OFFSET, -8,
                    0x001fda, 0x001fdb, INTEGER_OFFSET, -100,
                    0x001fe0, 0x001fe1, INTEGER_OFFSET, 8,
                    0x001fe5, 0x001fe5, INTEGER_OFFSET, 7,
                    0x001fe8, 0x001fe9, INTEGER_OFFSET, -8,
                    0x001fea, 0x001feb, INTEGER_OFFSET, -112,
                    0x001fec, 0x001fec, INTEGER_OFFSET, -7,
                    0x001ff3, 0x001ff3, INTEGER_OFFSET, 9,
                    0x001ff8, 0x001ff9, INTEGER_OFFSET, -128,
                    0x001ffa, 0x001ffb, INTEGER_OFFSET, -126,
                    0x001ffc, 0x001ffc, INTEGER_OFFSET, -9,
                    0x002126, 0x002126, DIRECT_MAPPING, 27,
                    0x00212a, 0x00212a, DIRECT_MAPPING, 23,
                    0x00212b, 0x00212b, DIRECT_MAPPING, 25,
                    0x002132, 0x002132, INTEGER_OFFSET, 28,
                    0x00214e, 0x00214e, INTEGER_OFFSET, -28,
                    0x002160, 0x00216f, INTEGER_OFFSET, 16,
                    0x002170, 0x00217f, INTEGER_OFFSET, -16,
                    0x002183, 0x002184, ALTERNATING_UL, 0,
                    0x0024b6, 0x0024cf, INTEGER_OFFSET, 26,
                    0x0024d0, 0x0024e9, INTEGER_OFFSET, -26,
                    0x002c00, 0x002c2e, INTEGER_OFFSET, 48,
                    0x002c30, 0x002c5e, INTEGER_OFFSET, -48,
                    0x002c60, 0x002c61, ALTERNATING_AL, 0,
                    0x002c62, 0x002c62, INTEGER_OFFSET, -10743,
                    0x002c63, 0x002c63, INTEGER_OFFSET, -3814,
                    0x002c64, 0x002c64, INTEGER_OFFSET, -10727,
                    0x002c65, 0x002c65, INTEGER_OFFSET, -10795,
                    0x002c66, 0x002c66, INTEGER_OFFSET, -10792,
                    0x002c67, 0x002c6c, ALTERNATING_UL, 0,
                    0x002c6d, 0x002c6d, INTEGER_OFFSET, -10780,
                    0x002c6e, 0x002c6e, INTEGER_OFFSET, -10749,
                    0x002c6f, 0x002c6f, INTEGER_OFFSET, -10783,
                    0x002c70, 0x002c70, INTEGER_OFFSET, -10782,
                    0x002c72, 0x002c73, ALTERNATING_AL, 0,
                    0x002c75, 0x002c76, ALTERNATING_UL, 0,
                    0x002c7e, 0x002c7f, INTEGER_OFFSET, -10815,
                    0x002c80, 0x002ce3, ALTERNATING_AL, 0,
                    0x002ceb, 0x002cee, ALTERNATING_UL, 0,
                    0x002cf2, 0x002cf3, ALTERNATING_AL, 0,
                    0x002d00, 0x002d25, INTEGER_OFFSET, -7264,
                    0x002d27, 0x002d27, INTEGER_OFFSET, -7264,
                    0x002d2d, 0x002d2d, INTEGER_OFFSET, -7264,
                    0x00a640, 0x00a649, ALTERNATING_AL, 0,
                    0x00a64a, 0x00a64b, DIRECT_MAPPING, 21,
                    0x00a64c, 0x00a66d, ALTERNATING_AL, 0,
                    0x00a680, 0x00a69b, ALTERNATING_AL, 0,
                    0x00a722, 0x00a72f, ALTERNATING_AL, 0,
                    0x00a732, 0x00a76f, ALTERNATING_AL, 0,
                    0x00a779, 0x00a77c, ALTERNATING_UL, 0,
                    0x00a77d, 0x00a77d, INTEGER_OFFSET, -35332,
                    0x00a77e, 0x00a787, ALTERNATING_AL, 0,
                    0x00a78b, 0x00a78c, ALTERNATING_UL, 0,
                    0x00a78d, 0x00a78d, INTEGER_OFFSET, -42280,
                    0x00a790, 0x00a793, ALTERNATING_AL, 0,
                    0x00a794, 0x00a794, INTEGER_OFFSET, 48,
                    0x00a796, 0x00a7a9, ALTERNATING_AL, 0,
                    0x00a7aa, 0x00a7aa, INTEGER_OFFSET, -42308,
                    0x00a7ab, 0x00a7ab, INTEGER_OFFSET, -42319,
                    0x00a7ac, 0x00a7ac, INTEGER_OFFSET, -42315,
                    0x00a7ad, 0x00a7ad, INTEGER_OFFSET, -42305,
                    0x00a7ae, 0x00a7ae, INTEGER_OFFSET, -42308,
                    0x00a7b0, 0x00a7b0, INTEGER_OFFSET, -42258,
                    0x00a7b1, 0x00a7b1, INTEGER_OFFSET, -42282,
                    0x00a7b2, 0x00a7b2, INTEGER_OFFSET, -42261,
                    0x00a7b3, 0x00a7b3, INTEGER_OFFSET, 928,
                    0x00a7b4, 0x00a7bf, ALTERNATING_AL, 0,
                    0x00a7c2, 0x00a7c3, ALTERNATING_AL, 0,
                    0x00a7c4, 0x00a7c4, INTEGER_OFFSET, -48,
                    0x00a7c5, 0x00a7c5, INTEGER_OFFSET, -42307,
                    0x00a7c6, 0x00a7c6, INTEGER_OFFSET, -35384,
                    0x00a7c7, 0x00a7ca, ALTERNATING_UL, 0,
                    0x00a7f5, 0x00a7f6, ALTERNATING_UL, 0,
                    0x00ab53, 0x00ab53, INTEGER_OFFSET, -928,
                    0x00ab70, 0x00abbf, INTEGER_OFFSET, -38864,
                    0x00ff21, 0x00ff3a, INTEGER_OFFSET, 32,
                    0x00ff41, 0x00ff5a, INTEGER_OFFSET, -32,
                    0x010400, 0x010427, INTEGER_OFFSET, 40,
                    0x010428, 0x01044f, INTEGER_OFFSET, -40,
                    0x0104b0, 0x0104d3, INTEGER_OFFSET, 40,
                    0x0104d8, 0x0104fb, INTEGER_OFFSET, -40,
                    0x010c80, 0x010cb2, INTEGER_OFFSET, 64,
                    0x010cc0, 0x010cf2, INTEGER_OFFSET, -64,
                    0x0118a0, 0x0118bf, INTEGER_OFFSET, 32,
                    0x0118c0, 0x0118df, INTEGER_OFFSET, -32,
                    0x016e40, 0x016e5f, INTEGER_OFFSET, 32,
                    0x016e60, 0x016e7f, INTEGER_OFFSET, -32,
                    0x01e900, 0x01e921, INTEGER_OFFSET, 34,
                    0x01e922, 0x01e943, INTEGER_OFFSET, -34
    });

    private static final CaseFoldTableImpl PYTHON_ASCII_TABLE_ENTRIES = new CaseFoldTableImpl(new int[]{
                    0x000041, 0x00005a, INTEGER_OFFSET, 32,
                    0x000061, 0x00007a, INTEGER_OFFSET, -32
    });

    private static final CaseFoldTableImpl PYTHON_UNICODE_TABLE_ENTRIES = new CaseFoldTableImpl(new int[]{
                    0x000041, 0x000041, DIRECT_MAPPING, 28,
                    0x000042, 0x000045, INTEGER_OFFSET, 32,
                    0x000046, 0x000046, DIRECT_MAPPING, 29,
                    0x000047, 0x000047, INTEGER_OFFSET, 32,
                    0x000048, 0x000048, DIRECT_MAPPING, 30,
                    0x000049, 0x000049, DIRECT_MAPPING, 31,
                    0x00004a, 0x00004a, DIRECT_MAPPING, 32,
                    0x00004b, 0x00004b, DIRECT_MAPPING, 23,
                    0x00004c, 0x000052, INTEGER_OFFSET, 32,
                    0x000053, 0x000053, DIRECT_MAPPING, 33,
                    0x000054, 0x000054, DIRECT_MAPPING, 34,
                    0x000055, 0x000056, INTEGER_OFFSET, 32,
                    0x000057, 0x000057, DIRECT_MAPPING, 35,
                    0x000058, 0x000058, INTEGER_OFFSET, 32,
                    0x000059, 0x000059, DIRECT_MAPPING, 36,
                    0x00005a, 0x00005a, INTEGER_OFFSET, 32,
                    0x000061, 0x000061, DIRECT_MAPPING, 28,
                    0x000062, 0x000065, INTEGER_OFFSET, -32,
                    0x000066, 0x000066, DIRECT_MAPPING, 29,
                    0x000067, 0x000067, INTEGER_OFFSET, -32,
                    0x000068, 0x000068, DIRECT_MAPPING, 30,
                    0x000069, 0x000069, DIRECT_MAPPING, 31,
                    0x00006a, 0x00006a, DIRECT_MAPPING, 32,
                    0x00006b, 0x00006b, DIRECT_MAPPING, 23,
                    0x00006c, 0x000072, INTEGER_OFFSET, -32,
                    0x000073, 0x000073, DIRECT_MAPPING, 33,
                    0x000074, 0x000074, DIRECT_MAPPING, 34,
                    0x000075, 0x000076, INTEGER_OFFSET, -32,
                    0x000077, 0x000077, DIRECT_MAPPING, 35,
                    0x000078, 0x000078, INTEGER_OFFSET, -32,
                    0x000079, 0x000079, DIRECT_MAPPING, 36,
                    0x00007a, 0x00007a, INTEGER_OFFSET, -32,
                    0x0000b5, 0x0000b5, DIRECT_MAPPING, 0,
                    0x0000c0, 0x0000c4, INTEGER_OFFSET, 32,
                    0x0000c5, 0x0000c5, DIRECT_MAPPING, 25,
                    0x0000c6, 0x0000cb, INTEGER_OFFSET, 32,
                    0x0000cc, 0x0000cd, DIRECT_MAPPING, 31,
                    0x0000ce, 0x0000d6, INTEGER_OFFSET, 32,
                    0x0000d8, 0x0000de, INTEGER_OFFSET, 32,
                    0x0000df, 0x0000df, DIRECT_MAPPING, 33,
                    0x0000e0, 0x0000e4, INTEGER_OFFSET, -32,
                    0x0000e5, 0x0000e5, DIRECT_MAPPING, 25,
                    0x0000e6, 0x0000eb, INTEGER_OFFSET, -32,
                    0x0000ec, 0x0000ed, DIRECT_MAPPING, 31,
                    0x0000ee, 0x0000f6, INTEGER_OFFSET, -32,
                    0x0000f8, 0x0000fe, INTEGER_OFFSET, -32,
                    0x0000ff, 0x0000ff, INTEGER_OFFSET, 121,
                    0x000100, 0x000127, ALTERNATING_AL, 0,
                    0x000128, 0x000129, DIRECT_MAPPING, 31,
                    0x00012a, 0x00012f, ALTERNATING_AL, 0,
                    0x000130, 0x000131, DIRECT_MAPPING, 31,
                    0x000132, 0x000137, ALTERNATING_AL, 0,
                    0x000139, 0x000148, ALTERNATING_UL, 0,
                    0x000149, 0x000149, INTEGER_OFFSET, 371,
                    0x00014a, 0x000177, ALTERNATING_AL, 0,
                    0x000178, 0x000178, INTEGER_OFFSET, -121,
                    0x000179, 0x00017e, ALTERNATING_UL, 0,
                    0x00017f, 0x00017f, DIRECT_MAPPING, 33,
                    0x000180, 0x000180, INTEGER_OFFSET, 195,
                    0x000181, 0x000181, INTEGER_OFFSET, 210,
                    0x000182, 0x000185, ALTERNATING_AL, 0,
                    0x000186, 0x000186, INTEGER_OFFSET, 206,
                    0x000187, 0x000188, ALTERNATING_UL, 0,
                    0x000189, 0x00018a, INTEGER_OFFSET, 205,
                    0x00018b, 0x00018c, ALTERNATING_UL, 0,
                    0x00018e, 0x00018e, INTEGER_OFFSET, 79,
                    0x00018f, 0x00018f, INTEGER_OFFSET, 202,
                    0x000190, 0x000190, INTEGER_OFFSET, 203,
                    0x000191, 0x000192, ALTERNATING_UL, 0,
                    0x000193, 0x000193, INTEGER_OFFSET, 205,
                    0x000194, 0x000194, INTEGER_OFFSET, 207,
                    0x000195, 0x000195, INTEGER_OFFSET, 97,
                    0x000196, 0x000196, INTEGER_OFFSET, 211,
                    0x000197, 0x000197, INTEGER_OFFSET, 209,
                    0x000198, 0x000199, ALTERNATING_AL, 0,
                    0x00019a, 0x00019a, INTEGER_OFFSET, 163,
                    0x00019c, 0x00019c, INTEGER_OFFSET, 211,
                    0x00019d, 0x00019d, INTEGER_OFFSET, 213,
                    0x00019e, 0x00019e, INTEGER_OFFSET, 130,
                    0x00019f, 0x00019f, INTEGER_OFFSET, 214,
                    0x0001a0, 0x0001a5, ALTERNATING_AL, 0,
                    0x0001a6, 0x0001a6, INTEGER_OFFSET, 218,
                    0x0001a7, 0x0001a8, ALTERNATING_UL, 0,
                    0x0001a9, 0x0001a9, INTEGER_OFFSET, 218,
                    0x0001ac, 0x0001ad, ALTERNATING_AL, 0,
                    0x0001ae, 0x0001ae, INTEGER_OFFSET, 218,
                    0x0001af, 0x0001b0, ALTERNATING_UL, 0,
                    0x0001b1, 0x0001b2, INTEGER_OFFSET, 217,
                    0x0001b3, 0x0001b6, ALTERNATING_UL, 0,
                    0x0001b7, 0x0001b7, INTEGER_OFFSET, 219,
                    0x0001b8, 0x0001b9, ALTERNATING_AL, 0,
                    0x0001bc, 0x0001bd, ALTERNATING_AL, 0,
                    0x0001bf, 0x0001bf, INTEGER_OFFSET, 56,
                    0x0001c4, 0x0001c6, DIRECT_MAPPING, 1,
                    0x0001c7, 0x0001c9, DIRECT_MAPPING, 2,
                    0x0001ca, 0x0001cc, DIRECT_MAPPING, 3,
                    0x0001cd, 0x0001dc, ALTERNATING_UL, 0,
                    0x0001dd, 0x0001dd, INTEGER_OFFSET, -79,
                    0x0001de, 0x0001ef, ALTERNATING_AL, 0,
                    0x0001f0, 0x0001f0, DIRECT_MAPPING, 32,
                    0x0001f1, 0x0001f3, DIRECT_MAPPING, 4,
                    0x0001f4, 0x0001f5, ALTERNATING_AL, 0,
                    0x0001f6, 0x0001f6, INTEGER_OFFSET, -97,
                    0x0001f7, 0x0001f7, INTEGER_OFFSET, -56,
                    0x0001f8, 0x00021f, ALTERNATING_AL, 0,
                    0x000220, 0x000220, INTEGER_OFFSET, -130,
                    0x000222, 0x000233, ALTERNATING_AL, 0,
                    0x00023a, 0x00023a, INTEGER_OFFSET, 10795,
                    0x00023b, 0x00023c, ALTERNATING_UL, 0,
                    0x00023d, 0x00023d, INTEGER_OFFSET, -163,
                    0x00023e, 0x00023e, INTEGER_OFFSET, 10792,
                    0x00023f, 0x000240, INTEGER_OFFSET, 10815,
                    0x000241, 0x000242, ALTERNATING_UL, 0,
                    0x000243, 0x000243, INTEGER_OFFSET, -195,
                    0x000244, 0x000244, INTEGER_OFFSET, 69,
                    0x000245, 0x000245, INTEGER_OFFSET, 71,
                    0x000246, 0x00024f, ALTERNATING_AL, 0,
                    0x000250, 0x000250, INTEGER_OFFSET, 10783,
                    0x000251, 0x000251, INTEGER_OFFSET, 10780,
                    0x000252, 0x000252, INTEGER_OFFSET, 10782,
                    0x000253, 0x000253, INTEGER_OFFSET, -210,
                    0x000254, 0x000254, INTEGER_OFFSET, -206,
                    0x000256, 0x000257, INTEGER_OFFSET, -205,
                    0x000259, 0x000259, INTEGER_OFFSET, -202,
                    0x00025b, 0x00025b, INTEGER_OFFSET, -203,
                    0x00025c, 0x00025c, INTEGER_OFFSET, 42319,
                    0x000260, 0x000260, INTEGER_OFFSET, -205,
                    0x000261, 0x000261, INTEGER_OFFSET, 42315,
                    0x000263, 0x000263, INTEGER_OFFSET, -207,
                    0x000265, 0x000265, INTEGER_OFFSET, 42280,
                    0x000266, 0x000266, INTEGER_OFFSET, 42308,
                    0x000268, 0x000268, INTEGER_OFFSET, -209,
                    0x000269, 0x000269, INTEGER_OFFSET, -211,
                    0x00026a, 0x00026a, INTEGER_OFFSET, 42308,
                    0x00026b, 0x00026b, INTEGER_OFFSET, 10743,
                    0x00026c, 0x00026c, INTEGER_OFFSET, 42305,
                    0x00026f, 0x00026f, INTEGER_OFFSET, -211,
                    0x000271, 0x000271, INTEGER_OFFSET, 10749,
                    0x000272, 0x000272, INTEGER_OFFSET, -213,
                    0x000275, 0x000275, INTEGER_OFFSET, -214,
                    0x00027d, 0x00027d, INTEGER_OFFSET, 10727,
                    0x000280, 0x000280, INTEGER_OFFSET, -218,
                    0x000282, 0x000282, INTEGER_OFFSET, 42307,
                    0x000283, 0x000283, INTEGER_OFFSET, -218,
                    0x000287, 0x000287, INTEGER_OFFSET, 42282,
                    0x000288, 0x000288, INTEGER_OFFSET, -218,
                    0x000289, 0x000289, INTEGER_OFFSET, -69,
                    0x00028a, 0x00028b, INTEGER_OFFSET, -217,
                    0x00028c, 0x00028c, INTEGER_OFFSET, -71,
                    0x000292, 0x000292, INTEGER_OFFSET, -219,
                    0x00029d, 0x00029d, INTEGER_OFFSET, 42261,
                    0x00029e, 0x00029e, INTEGER_OFFSET, 42258,
                    0x0002bc, 0x0002bc, INTEGER_OFFSET, -371,
                    0x000345, 0x000345, DIRECT_MAPPING, 37,
                    0x000370, 0x000373, ALTERNATING_AL, 0,
                    0x000376, 0x000377, ALTERNATING_AL, 0,
                    0x00037b, 0x00037d, INTEGER_OFFSET, 130,
                    0x00037f, 0x00037f, INTEGER_OFFSET, 116,
                    0x000386, 0x000386, DIRECT_MAPPING, 38,
                    0x000388, 0x000388, INTEGER_OFFSET, 37,
                    0x000389, 0x000389, DIRECT_MAPPING, 39,
                    0x00038a, 0x00038a, INTEGER_OFFSET, 37,
                    0x00038c, 0x00038c, INTEGER_OFFSET, 64,
                    0x00038e, 0x00038e, INTEGER_OFFSET, 63,
                    0x00038f, 0x00038f, DIRECT_MAPPING, 40,
                    0x000390, 0x000390, DIRECT_MAPPING, 37,
                    0x000391, 0x000391, DIRECT_MAPPING, 41,
                    0x000392, 0x000392, DIRECT_MAPPING, 6,
                    0x000393, 0x000394, INTEGER_OFFSET, 32,
                    0x000395, 0x000395, DIRECT_MAPPING, 7,
                    0x000396, 0x000396, INTEGER_OFFSET, 32,
                    0x000397, 0x000397, DIRECT_MAPPING, 42,
                    0x000398, 0x000398, DIRECT_MAPPING, 26,
                    0x000399, 0x000399, DIRECT_MAPPING, 37,
                    0x00039a, 0x00039a, DIRECT_MAPPING, 9,
                    0x00039b, 0x00039b, INTEGER_OFFSET, 32,
                    0x00039c, 0x00039c, DIRECT_MAPPING, 0,
                    0x00039d, 0x00039f, INTEGER_OFFSET, 32,
                    0x0003a0, 0x0003a0, DIRECT_MAPPING, 10,
                    0x0003a1, 0x0003a1, DIRECT_MAPPING, 43,
                    0x0003a3, 0x0003a3, DIRECT_MAPPING, 12,
                    0x0003a4, 0x0003a4, INTEGER_OFFSET, 32,
                    0x0003a5, 0x0003a5, DIRECT_MAPPING, 44,
                    0x0003a6, 0x0003a6, DIRECT_MAPPING, 13,
                    0x0003a7, 0x0003a8, INTEGER_OFFSET, 32,
                    0x0003a9, 0x0003a9, DIRECT_MAPPING, 45,
                    0x0003aa, 0x0003ab, INTEGER_OFFSET, 32,
                    0x0003ac, 0x0003ac, DIRECT_MAPPING, 38,
                    0x0003ad, 0x0003ad, INTEGER_OFFSET, -37,
                    0x0003ae, 0x0003ae, DIRECT_MAPPING, 39,
                    0x0003af, 0x0003af, INTEGER_OFFSET, -37,
                    0x0003b0, 0x0003b0, DIRECT_MAPPING, 44,
                    0x0003b1, 0x0003b1, DIRECT_MAPPING, 41,
                    0x0003b2, 0x0003b2, DIRECT_MAPPING, 6,
                    0x0003b3, 0x0003b4, INTEGER_OFFSET, -32,
                    0x0003b5, 0x0003b5, DIRECT_MAPPING, 7,
                    0x0003b6, 0x0003b6, INTEGER_OFFSET, -32,
                    0x0003b7, 0x0003b7, DIRECT_MAPPING, 42,
                    0x0003b8, 0x0003b8, DIRECT_MAPPING, 26,
                    0x0003b9, 0x0003b9, DIRECT_MAPPING, 37,
                    0x0003ba, 0x0003ba, DIRECT_MAPPING, 9,
                    0x0003bb, 0x0003bb, INTEGER_OFFSET, -32,
                    0x0003bc, 0x0003bc, DIRECT_MAPPING, 0,
                    0x0003bd, 0x0003bf, INTEGER_OFFSET, -32,
                    0x0003c0, 0x0003c0, DIRECT_MAPPING, 10,
                    0x0003c1, 0x0003c1, DIRECT_MAPPING, 43,
                    0x0003c2, 0x0003c3, DIRECT_MAPPING, 12,
                    0x0003c4, 0x0003c4, INTEGER_OFFSET, -32,
                    0x0003c5, 0x0003c5, DIRECT_MAPPING, 44,
                    0x0003c6, 0x0003c6, DIRECT_MAPPING, 13,
                    0x0003c7, 0x0003c8, INTEGER_OFFSET, -32,
                    0x0003c9, 0x0003c9, DIRECT_MAPPING, 45,
                    0x0003ca, 0x0003cb, INTEGER_OFFSET, -32,
                    0x0003cc, 0x0003cc, INTEGER_OFFSET, -64,
                    0x0003cd, 0x0003cd, INTEGER_OFFSET, -63,
                    0x0003ce, 0x0003ce, DIRECT_MAPPING, 40,
                    0x0003cf, 0x0003cf, INTEGER_OFFSET, 8,
                    0x0003d0, 0x0003d0, DIRECT_MAPPING, 6,
                    0x0003d1, 0x0003d1, DIRECT_MAPPING, 26,
                    0x0003d5, 0x0003d5, DIRECT_MAPPING, 13,
                    0x0003d6, 0x0003d6, DIRECT_MAPPING, 10,
                    0x0003d7, 0x0003d7, INTEGER_OFFSET, -8,
                    0x0003d8, 0x0003ef, ALTERNATING_AL, 0,
                    0x0003f0, 0x0003f0, DIRECT_MAPPING, 9,
                    0x0003f1, 0x0003f1, DIRECT_MAPPING, 43,
                    0x0003f2, 0x0003f2, INTEGER_OFFSET, 7,
                    0x0003f3, 0x0003f3, INTEGER_OFFSET, -116,
                    0x0003f4, 0x0003f4, DIRECT_MAPPING, 26,
                    0x0003f5, 0x0003f5, DIRECT_MAPPING, 7,
                    0x0003f7, 0x0003f8, ALTERNATING_UL, 0,
                    0x0003f9, 0x0003f9, INTEGER_OFFSET, -7,
                    0x0003fa, 0x0003fb, ALTERNATING_AL, 0,
                    0x0003fd, 0x0003ff, INTEGER_OFFSET, -130,
                    0x000400, 0x00040f, INTEGER_OFFSET, 80,
                    0x000410, 0x000411, INTEGER_OFFSET, 32,
                    0x000412, 0x000412, DIRECT_MAPPING, 14,
                    0x000413, 0x000413, INTEGER_OFFSET, 32,
                    0x000414, 0x000414, DIRECT_MAPPING, 15,
                    0x000415, 0x00041d, INTEGER_OFFSET, 32,
                    0x00041e, 0x00041e, DIRECT_MAPPING, 16,
                    0x00041f, 0x000420, INTEGER_OFFSET, 32,
                    0x000421, 0x000421, DIRECT_MAPPING, 17,
                    0x000422, 0x000422, DIRECT_MAPPING, 18,
                    0x000423, 0x000429, INTEGER_OFFSET, 32,
                    0x00042a, 0x00042a, DIRECT_MAPPING, 19,
                    0x00042b, 0x00042f, INTEGER_OFFSET, 32,
                    0x000430, 0x000431, INTEGER_OFFSET, -32,
                    0x000432, 0x000432, DIRECT_MAPPING, 14,
                    0x000433, 0x000433, INTEGER_OFFSET, -32,
                    0x000434, 0x000434, DIRECT_MAPPING, 15,
                    0x000435, 0x00043d, INTEGER_OFFSET, -32,
                    0x00043e, 0x00043e, DIRECT_MAPPING, 16,
                    0x00043f, 0x000440, INTEGER_OFFSET, -32,
                    0x000441, 0x000441, DIRECT_MAPPING, 17,
                    0x000442, 0x000442, DIRECT_MAPPING, 18,
                    0x000443, 0x000449, INTEGER_OFFSET, -32,
                    0x00044a, 0x00044a, DIRECT_MAPPING, 19,
                    0x00044b, 0x00044f, INTEGER_OFFSET, -32,
                    0x000450, 0x00045f, INTEGER_OFFSET, -80,
                    0x000460, 0x000461, ALTERNATING_AL, 0,
                    0x000462, 0x000463, DIRECT_MAPPING, 20,
                    0x000464, 0x000481, ALTERNATING_AL, 0,
                    0x00048a, 0x0004bf, ALTERNATING_AL, 0,
                    0x0004c0, 0x0004c0, INTEGER_OFFSET, 15,
                    0x0004c1, 0x0004ce, ALTERNATING_UL, 0,
                    0x0004cf, 0x0004cf, INTEGER_OFFSET, -15,
                    0x0004d0, 0x00052f, ALTERNATING_AL, 0,
                    0x000531, 0x000534, INTEGER_OFFSET, 48,
                    0x000535, 0x000535, DIRECT_MAPPING, 46,
                    0x000536, 0x000543, INTEGER_OFFSET, 48,
                    0x000544, 0x000544, DIRECT_MAPPING, 47,
                    0x000545, 0x00054d, INTEGER_OFFSET, 48,
                    0x00054e, 0x00054e, DIRECT_MAPPING, 48,
                    0x00054f, 0x000556, INTEGER_OFFSET, 48,
                    0x000561, 0x000564, INTEGER_OFFSET, -48,
                    0x000565, 0x000565, DIRECT_MAPPING, 46,
                    0x000566, 0x000573, INTEGER_OFFSET, -48,
                    0x000574, 0x000574, DIRECT_MAPPING, 47,
                    0x000575, 0x00057d, INTEGER_OFFSET, -48,
                    0x00057e, 0x00057e, DIRECT_MAPPING, 48,
                    0x00057f, 0x000586, INTEGER_OFFSET, -48,
                    0x000587, 0x000587, DIRECT_MAPPING, 46,
                    0x0010a0, 0x0010c5, INTEGER_OFFSET, 7264,
                    0x0010c7, 0x0010c7, INTEGER_OFFSET, 7264,
                    0x0010cd, 0x0010cd, INTEGER_OFFSET, 7264,
                    0x0010d0, 0x0010fa, INTEGER_OFFSET, 3008,
                    0x0010fd, 0x0010ff, INTEGER_OFFSET, 3008,
                    0x0013a0, 0x0013ef, INTEGER_OFFSET, 38864,
                    0x0013f0, 0x0013f5, INTEGER_OFFSET, 8,
                    0x0013f8, 0x0013fd, INTEGER_OFFSET, -8,
                    0x001c80, 0x001c80, DIRECT_MAPPING, 14,
                    0x001c81, 0x001c81, DIRECT_MAPPING, 15,
                    0x001c82, 0x001c82, DIRECT_MAPPING, 16,
                    0x001c83, 0x001c83, DIRECT_MAPPING, 17,
                    0x001c84, 0x001c85, DIRECT_MAPPING, 18,
                    0x001c86, 0x001c86, DIRECT_MAPPING, 19,
                    0x001c87, 0x001c87, DIRECT_MAPPING, 20,
                    0x001c88, 0x001c88, DIRECT_MAPPING, 21,
                    0x001c90, 0x001cba, INTEGER_OFFSET, -3008,
                    0x001cbd, 0x001cbf, INTEGER_OFFSET, -3008,
                    0x001d79, 0x001d79, INTEGER_OFFSET, 35332,
                    0x001d7d, 0x001d7d, INTEGER_OFFSET, 3814,
                    0x001d8e, 0x001d8e, INTEGER_OFFSET, 35384,
                    0x001e00, 0x001e5f, ALTERNATING_AL, 0,
                    0x001e60, 0x001e61, DIRECT_MAPPING, 22,
                    0x001e62, 0x001e95, ALTERNATING_AL, 0,
                    0x001e96, 0x001e96, DIRECT_MAPPING, 30,
                    0x001e97, 0x001e97, DIRECT_MAPPING, 34,
                    0x001e98, 0x001e98, DIRECT_MAPPING, 35,
                    0x001e99, 0x001e99, DIRECT_MAPPING, 36,
                    0x001e9a, 0x001e9a, DIRECT_MAPPING, 28,
                    0x001e9b, 0x001e9b, DIRECT_MAPPING, 22,
                    0x001e9e, 0x001e9e, DIRECT_MAPPING, 33,
                    0x001ea0, 0x001eff, ALTERNATING_AL, 0,
                    0x001f00, 0x001f00, DIRECT_MAPPING, 49,
                    0x001f01, 0x001f01, DIRECT_MAPPING, 50,
                    0x001f02, 0x001f02, DIRECT_MAPPING, 51,
                    0x001f03, 0x001f03, DIRECT_MAPPING, 52,
                    0x001f04, 0x001f04, DIRECT_MAPPING, 53,
                    0x001f05, 0x001f05, DIRECT_MAPPING, 54,
                    0x001f06, 0x001f06, DIRECT_MAPPING, 55,
                    0x001f07, 0x001f07, DIRECT_MAPPING, 56,
                    0x001f08, 0x001f08, DIRECT_MAPPING, 49,
                    0x001f09, 0x001f09, DIRECT_MAPPING, 50,
                    0x001f0a, 0x001f0a, DIRECT_MAPPING, 51,
                    0x001f0b, 0x001f0b, DIRECT_MAPPING, 52,
                    0x001f0c, 0x001f0c, DIRECT_MAPPING, 53,
                    0x001f0d, 0x001f0d, DIRECT_MAPPING, 54,
                    0x001f0e, 0x001f0e, DIRECT_MAPPING, 55,
                    0x001f0f, 0x001f0f, DIRECT_MAPPING, 56,
                    0x001f10, 0x001f15, INTEGER_OFFSET, 8,
                    0x001f18, 0x001f1d, INTEGER_OFFSET, -8,
                    0x001f20, 0x001f20, DIRECT_MAPPING, 57,
                    0x001f21, 0x001f21, DIRECT_MAPPING, 58,
                    0x001f22, 0x001f22, DIRECT_MAPPING, 59,
                    0x001f23, 0x001f23, DIRECT_MAPPING, 60,
                    0x001f24, 0x001f24, DIRECT_MAPPING, 61,
                    0x001f25, 0x001f25, DIRECT_MAPPING, 62,
                    0x001f26, 0x001f26, DIRECT_MAPPING, 63,
                    0x001f27, 0x001f27, DIRECT_MAPPING, 64,
                    0x001f28, 0x001f28, DIRECT_MAPPING, 57,
                    0x001f29, 0x001f29, DIRECT_MAPPING, 58,
                    0x001f2a, 0x001f2a, DIRECT_MAPPING, 59,
                    0x001f2b, 0x001f2b, DIRECT_MAPPING, 60,
                    0x001f2c, 0x001f2c, DIRECT_MAPPING, 61,
                    0x001f2d, 0x001f2d, DIRECT_MAPPING, 62,
                    0x001f2e, 0x001f2e, DIRECT_MAPPING, 63,
                    0x001f2f, 0x001f2f, DIRECT_MAPPING, 64,
                    0x001f30, 0x001f37, INTEGER_OFFSET, 8,
                    0x001f38, 0x001f3f, INTEGER_OFFSET, -8,
                    0x001f40, 0x001f45, INTEGER_OFFSET, 8,
                    0x001f48, 0x001f4d, INTEGER_OFFSET, -8,
                    0x001f50, 0x001f50, DIRECT_MAPPING, 44,
                    0x001f51, 0x001f51, INTEGER_OFFSET, 8,
                    0x001f52, 0x001f52, DIRECT_MAPPING, 44,
                    0x001f53, 0x001f53, INTEGER_OFFSET, 8,
                    0x001f54, 0x001f54, DIRECT_MAPPING, 44,
                    0x001f55, 0x001f55, INTEGER_OFFSET, 8,
                    0x001f56, 0x001f56, DIRECT_MAPPING, 44,
                    0x001f57, 0x001f57, INTEGER_OFFSET, 8,
                    0x001f59, 0x001f59, INTEGER_OFFSET, -8,
                    0x001f5b, 0x001f5b, INTEGER_OFFSET, -8,
                    0x001f5d, 0x001f5d, INTEGER_OFFSET, -8,
                    0x001f5f, 0x001f5f, INTEGER_OFFSET, -8,
                    0x001f60, 0x001f60, DIRECT_MAPPING, 65,
                    0x001f61, 0x001f61, DIRECT_MAPPING, 66,
                    0x001f62, 0x001f62, DIRECT_MAPPING, 67,
                    0x001f63, 0x001f63, DIRECT_MAPPING, 68,
                    0x001f64, 0x001f64, DIRECT_MAPPING, 69,
                    0x001f65, 0x001f65, DIRECT_MAPPING, 70,
                    0x001f66, 0x001f66, DIRECT_MAPPING, 71,
                    0x001f67, 0x001f67, DIRECT_MAPPING, 72,
                    0x001f68, 0x001f68, DIRECT_MAPPING, 65,
                    0x001f69, 0x001f69, DIRECT_MAPPING, 66,
                    0x001f6a, 0x001f6a, DIRECT_MAPPING, 67,
                    0x001f6b, 0x001f6b, DIRECT_MAPPING, 68,
                    0x001f6c, 0x001f6c, DIRECT_MAPPING, 69,
                    0x001f6d, 0x001f6d, DIRECT_MAPPING, 70,
                    0x001f6e, 0x001f6e, DIRECT_MAPPING, 71,
                    0x001f6f, 0x001f6f, DIRECT_MAPPING, 72,
                    0x001f70, 0x001f70, DIRECT_MAPPING, 73,
                    0x001f71, 0x001f71, INTEGER_OFFSET, 74,
                    0x001f72, 0x001f73, INTEGER_OFFSET, 86,
                    0x001f74, 0x001f74, DIRECT_MAPPING, 74,
                    0x001f75, 0x001f75, INTEGER_OFFSET, 86,
                    0x001f76, 0x001f77, INTEGER_OFFSET, 100,
                    0x001f78, 0x001f79, INTEGER_OFFSET, 128,
                    0x001f7a, 0x001f7b, INTEGER_OFFSET, 112,
                    0x001f7c, 0x001f7c, DIRECT_MAPPING, 75,
                    0x001f7d, 0x001f7d, INTEGER_OFFSET, 126,
                    0x001f80, 0x001f80, DIRECT_MAPPING, 49,
                    0x001f81, 0x001f81, DIRECT_MAPPING, 50,
                    0x001f82, 0x001f82, DIRECT_MAPPING, 51,
                    0x001f83, 0x001f83, DIRECT_MAPPING, 52,
                    0x001f84, 0x001f84, DIRECT_MAPPING, 53,
                    0x001f85, 0x001f85, DIRECT_MAPPING, 54,
                    0x001f86, 0x001f86, DIRECT_MAPPING, 55,
                    0x001f87, 0x001f87, DIRECT_MAPPING, 56,
                    0x001f88, 0x001f88, DIRECT_MAPPING, 49,
                    0x001f89, 0x001f89, DIRECT_MAPPING, 50,
                    0x001f8a, 0x001f8a, DIRECT_MAPPING, 51,
                    0x001f8b, 0x001f8b, DIRECT_MAPPING, 52,
                    0x001f8c, 0x001f8c, DIRECT_MAPPING, 53,
                    0x001f8d, 0x001f8d, DIRECT_MAPPING, 54,
                    0x001f8e, 0x001f8e, DIRECT_MAPPING, 55,
                    0x001f8f, 0x001f8f, DIRECT_MAPPING, 56,
                    0x001f90, 0x001f90, DIRECT_MAPPING, 57,
                    0x001f91, 0x001f91, DIRECT_MAPPING, 58,
                    0x001f92, 0x001f92, DIRECT_MAPPING, 59,
                    0x001f93, 0x001f93, DIRECT_MAPPING, 60,
                    0x001f94, 0x001f94, DIRECT_MAPPING, 61,
                    0x001f95, 0x001f95, DIRECT_MAPPING, 62,
                    0x001f96, 0x001f96, DIRECT_MAPPING, 63,
                    0x001f97, 0x001f97, DIRECT_MAPPING, 64,
                    0x001f98, 0x001f98, DIRECT_MAPPING, 57,
                    0x001f99, 0x001f99, DIRECT_MAPPING, 58,
                    0x001f9a, 0x001f9a, DIRECT_MAPPING, 59,
                    0x001f9b, 0x001f9b, DIRECT_MAPPING, 60,
                    0x001f9c, 0x001f9c, DIRECT_MAPPING, 61,
                    0x001f9d, 0x001f9d, DIRECT_MAPPING, 62,
                    0x001f9e, 0x001f9e, DIRECT_MAPPING, 63,
                    0x001f9f, 0x001f9f, DIRECT_MAPPING, 64,
                    0x001fa0, 0x001fa0, DIRECT_MAPPING, 65,
                    0x001fa1, 0x001fa1, DIRECT_MAPPING, 66,
                    0x001fa2, 0x001fa2, DIRECT_MAPPING, 67,
                    0x001fa3, 0x001fa3, DIRECT_MAPPING, 68,
                    0x001fa4, 0x001fa4, DIRECT_MAPPING, 69,
                    0x001fa5, 0x001fa5, DIRECT_MAPPING, 70,
                    0x001fa6, 0x001fa6, DIRECT_MAPPING, 71,
                    0x001fa7, 0x001fa7, DIRECT_MAPPING, 72,
                    0x001fa8, 0x001fa8, DIRECT_MAPPING, 65,
                    0x001fa9, 0x001fa9, DIRECT_MAPPING, 66,
                    0x001faa, 0x001faa, DIRECT_MAPPING, 67,
                    0x001fab, 0x001fab, DIRECT_MAPPING, 68,
                    0x001fac, 0x001fac, DIRECT_MAPPING, 69,
                    0x001fad, 0x001fad, DIRECT_MAPPING, 70,
                    0x001fae, 0x001fae, DIRECT_MAPPING, 71,
                    0x001faf, 0x001faf, DIRECT_MAPPING, 72,
                    0x001fb0, 0x001fb1, INTEGER_OFFSET, 8,
                    0x001fb2, 0x001fb2, DIRECT_MAPPING, 73,
                    0x001fb3, 0x001fb3, DIRECT_MAPPING, 41,
                    0x001fb4, 0x001fb4, DIRECT_MAPPING, 38,
                    0x001fb6, 0x001fb7, DIRECT_MAPPING, 41,
                    0x001fb8, 0x001fb9, INTEGER_OFFSET, -8,
                    0x001fba, 0x001fba, DIRECT_MAPPING, 73,
                    0x001fbb, 0x001fbb, INTEGER_OFFSET, -74,
                    0x001fbc, 0x001fbc, DIRECT_MAPPING, 41,
                    0x001fbe, 0x001fbe, DIRECT_MAPPING, 37,
                    0x001fc2, 0x001fc2, DIRECT_MAPPING, 74,
                    0x001fc3, 0x001fc3, DIRECT_MAPPING, 42,
                    0x001fc4, 0x001fc4, DIRECT_MAPPING, 39,
                    0x001fc6, 0x001fc7, DIRECT_MAPPING, 42,
                    0x001fc8, 0x001fc9, INTEGER_OFFSET, -86,
                    0x001fca, 0x001fca, DIRECT_MAPPING, 74,
                    0x001fcb, 0x001fcb, INTEGER_OFFSET, -86,
                    0x001fcc, 0x001fcc, DIRECT_MAPPING, 42,
                    0x001fd0, 0x001fd1, INTEGER_OFFSET, 8,
                    0x001fd2, 0x001fd3, DIRECT_MAPPING, 37,
                    0x001fd6, 0x001fd7, DIRECT_MAPPING, 37,
                    0x001fd8, 0x001fd9, INTEGER_OFFSET, -8,
                    0x001fda, 0x001fdb, INTEGER_OFFSET, -100,
                    0x001fe0, 0x001fe1, INTEGER_OFFSET, 8,
                    0x001fe2, 0x001fe3, DIRECT_MAPPING, 44,
                    0x001fe4, 0x001fe4, DIRECT_MAPPING, 43,
                    0x001fe5, 0x001fe5, INTEGER_OFFSET, 7,
                    0x001fe6, 0x001fe7, DIRECT_MAPPING, 44,
                    0x001fe8, 0x001fe9, INTEGER_OFFSET, -8,
                    0x001fea, 0x001feb, INTEGER_OFFSET, -112,
                    0x001fec, 0x001fec, INTEGER_OFFSET, -7,
                    0x001ff2, 0x001ff2, DIRECT_MAPPING, 75,
                    0x001ff3, 0x001ff3, DIRECT_MAPPING, 45,
                    0x001ff4, 0x001ff4, DIRECT_MAPPING, 40,
                    0x001ff6, 0x001ff7, DIRECT_MAPPING, 45,
                    0x001ff8, 0x001ff9, INTEGER_OFFSET, -128,
                    0x001ffa, 0x001ffa, DIRECT_MAPPING, 75,
                    0x001ffb, 0x001ffb, INTEGER_OFFSET, -126,
                    0x001ffc, 0x001ffc, DIRECT_MAPPING, 45,
                    0x002126, 0x002126, DIRECT_MAPPING, 45,
                    0x00212a, 0x00212a, DIRECT_MAPPING, 23,
                    0x00212b, 0x00212b, DIRECT_MAPPING, 25,
                    0x002132, 0x002132, INTEGER_OFFSET, 28,
                    0x00214e, 0x00214e, INTEGER_OFFSET, -28,
                    0x002160, 0x00216f, INTEGER_OFFSET, 16,
                    0x002170, 0x00217f, INTEGER_OFFSET, -16,
                    0x002183, 0x002184, ALTERNATING_UL, 0,
                    0x0024b6, 0x0024cf, INTEGER_OFFSET, 26,
                    0x0024d0, 0x0024e9, INTEGER_OFFSET, -26,
                    0x002c00, 0x002c2e, INTEGER_OFFSET, 48,
                    0x002c30, 0x002c5e, INTEGER_OFFSET, -48,
                    0x002c60, 0x002c61, ALTERNATING_AL, 0,
                    0x002c62, 0x002c62, INTEGER_OFFSET, -10743,
                    0x002c63, 0x002c63, INTEGER_OFFSET, -3814,
                    0x002c64, 0x002c64, INTEGER_OFFSET, -10727,
                    0x002c65, 0x002c65, INTEGER_OFFSET, -10795,
                    0x002c66, 0x002c66, INTEGER_OFFSET, -10792,
                    0x002c67, 0x002c6c, ALTERNATING_UL, 0,
                    0x002c6d, 0x002c6d, INTEGER_OFFSET, -10780,
                    0x002c6e, 0x002c6e, INTEGER_OFFSET, -10749,
                    0x002c6f, 0x002c6f, INTEGER_OFFSET, -10783,
                    0x002c70, 0x002c70, INTEGER_OFFSET, -10782,
                    0x002c72, 0x002c73, ALTERNATING_AL, 0,
                    0x002c75, 0x002c76, ALTERNATING_UL, 0,
                    0x002c7e, 0x002c7f, INTEGER_OFFSET, -10815,
                    0x002c80, 0x002ce3, ALTERNATING_AL, 0,
                    0x002ceb, 0x002cee, ALTERNATING_UL, 0,
                    0x002cf2, 0x002cf3, ALTERNATING_AL, 0,
                    0x002d00, 0x002d25, INTEGER_OFFSET, -7264,
                    0x002d27, 0x002d27, INTEGER_OFFSET, -7264,
                    0x002d2d, 0x002d2d, INTEGER_OFFSET, -7264,
                    0x00a640, 0x00a649, ALTERNATING_AL, 0,
                    0x00a64a, 0x00a64b, DIRECT_MAPPING, 21,
                    0x00a64c, 0x00a66d, ALTERNATING_AL, 0,
                    0x00a680, 0x00a69b, ALTERNATING_AL, 0,
                    0x00a722, 0x00a72f, ALTERNATING_AL, 0,
                    0x00a732, 0x00a76f, ALTERNATING_AL, 0,
                    0x00a779, 0x00a77c, ALTERNATING_UL, 0,
                    0x00a77d, 0x00a77d, INTEGER_OFFSET, -35332,
                    0x00a77e, 0x00a787, ALTERNATING_AL, 0,
                    0x00a78b, 0x00a78c, ALTERNATING_UL, 0,
                    0x00a78d, 0x00a78d, INTEGER_OFFSET, -42280,
                    0x00a790, 0x00a793, ALTERNATING_AL, 0,
                    0x00a794, 0x00a794, INTEGER_OFFSET, 48,
                    0x00a796, 0x00a7a9, ALTERNATING_AL, 0,
                    0x00a7aa, 0x00a7aa, INTEGER_OFFSET, -42308,
                    0x00a7ab, 0x00a7ab, INTEGER_OFFSET, -42319,
                    0x00a7ac, 0x00a7ac, INTEGER_OFFSET, -42315,
                    0x00a7ad, 0x00a7ad, INTEGER_OFFSET, -42305,
                    0x00a7ae, 0x00a7ae, INTEGER_OFFSET, -42308,
                    0x00a7b0, 0x00a7b0, INTEGER_OFFSET, -42258,
                    0x00a7b1, 0x00a7b1, INTEGER_OFFSET, -42282,
                    0x00a7b2, 0x00a7b2, INTEGER_OFFSET, -42261,
                    0x00a7b3, 0x00a7b3, INTEGER_OFFSET, 928,
                    0x00a7b4, 0x00a7bf, ALTERNATING_AL, 0,
                    0x00a7c2, 0x00a7c3, ALTERNATING_AL, 0,
                    0x00a7c4, 0x00a7c4, INTEGER_OFFSET, -48,
                    0x00a7c5, 0x00a7c5, INTEGER_OFFSET, -42307,
                    0x00a7c6, 0x00a7c6, INTEGER_OFFSET, -35384,
                    0x00a7c7, 0x00a7ca, ALTERNATING_UL, 0,
                    0x00a7f5, 0x00a7f6, ALTERNATING_UL, 0,
                    0x00ab53, 0x00ab53, INTEGER_OFFSET, -928,
                    0x00ab70, 0x00abbf, INTEGER_OFFSET, -38864,
                    0x00fb00, 0x00fb04, DIRECT_MAPPING, 29,
                    0x00fb05, 0x00fb06, DIRECT_MAPPING, 33,
                    0x00fb13, 0x00fb15, DIRECT_MAPPING, 47,
                    0x00fb16, 0x00fb16, DIRECT_MAPPING, 48,
                    0x00fb17, 0x00fb17, DIRECT_MAPPING, 47,
                    0x00ff21, 0x00ff3a, INTEGER_OFFSET, 32,
                    0x00ff41, 0x00ff5a, INTEGER_OFFSET, -32,
                    0x010400, 0x010427, INTEGER_OFFSET, 40,
                    0x010428, 0x01044f, INTEGER_OFFSET, -40,
                    0x0104b0, 0x0104d3, INTEGER_OFFSET, 40,
                    0x0104d8, 0x0104fb, INTEGER_OFFSET, -40,
                    0x010c80, 0x010cb2, INTEGER_OFFSET, 64,
                    0x010cc0, 0x010cf2, INTEGER_OFFSET, -64,
                    0x0118a0, 0x0118bf, INTEGER_OFFSET, 32,
                    0x0118c0, 0x0118df, INTEGER_OFFSET, -32,
                    0x016e40, 0x016e5f, INTEGER_OFFSET, 32,
                    0x016e60, 0x016e7f, INTEGER_OFFSET, -32,
                    0x01e900, 0x01e921, INTEGER_OFFSET, 34,
                    0x01e922, 0x01e943, INTEGER_OFFSET, -34
    });

    /* GENERATED CODE END - KEEP THIS MARKER FOR AUTOMATIC UPDATES */
}
