/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later.
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.annotations;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.hibernate.Incubating;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;

/**
 * Specifies the mapped column for storing the time zone information.
 * The annotation can be used in conjunction with the <code>TimeZoneStorageType.AUTO</code> and
 * <code>TimeZoneStorageType.COLUMN</code>. The column is simply ignored if <code>TimeZoneStorageType.AUTO</code>
 * is used and the database supports native time zone storage.
 *
 * @author Christian Beikov
 * @author Steve Ebersole
 * @author Andrea Boriero
 * @see TimeZoneStorage
 * @see TimeZoneStorageType#COLUMN
 * @see TimeZoneStorageType#AUTO
 */
@Incubating
@Retention(RetentionPolicy.RUNTIME)
@Target({ FIELD, METHOD })
public @interface TimeZoneColumn {

	/**
	 * (Optional) The name of the column. Defaults to
	 * the property or field name, suffixed by <code>_tz</code>.
	 */
	String name() default "";

	/**
	 * (Optional) Whether the column is included in SQL INSERT
	 * statements generated by the persistence provider.
	 */
	boolean insertable() default true;

	/**
	 * (Optional) Whether the column is included in SQL UPDATE
	 * statements generated by the persistence provider.
	 */
	boolean updatable() default true;

	/**
	 * (Optional) The SQL fragment that is used when
	 * generating the DDL for the column.
	 * <p> Defaults to the generated SQL to create a
	 * column of the inferred type.
	 */
	String columnDefinition() default "";

	/**
	 * (Optional) The name of the table that contains the column.
	 * If absent the column is assumed to be in the primary table.
	 */
	String table() default "";

}
