
/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later.
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.grammars.ordering;

/**
 * Grammar for parsing order-by fragments.
 *
 * @implNote While we could re-use the HQL lexer/parser for order fragment parsing, both the HQL lexer and parser
 * are way "heavier" than needed here.  So we use a simplified lexer and parser that defione just what is needed
 * to parse the order fragment
 */

import org.antlr.v4.runtime.tree.ParseTreeVisitor;

/**
 * This interface defines a complete generic visitor for a parse tree produced
 * by {@link OrderingParser}.
 *
 * @param <T> The return type of the visit operation. Use {@link Void} for
 * operations with no return type.
 */
public interface OrderingParserVisitor<T> extends ParseTreeVisitor<T> {
	/**
	 * Visit a parse tree produced by {@link OrderingParser#orderByFragment}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitOrderByFragment(OrderingParser.OrderByFragmentContext ctx);
	/**
	 * Visit a parse tree produced by {@link OrderingParser#sortSpecification}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitSortSpecification(OrderingParser.SortSpecificationContext ctx);
	/**
	 * Visit a parse tree produced by the {@code FunctionExpression}
	 * labeled alternative in {@link OrderingParser#expression}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitFunctionExpression(OrderingParser.FunctionExpressionContext ctx);
	/**
	 * Visit a parse tree produced by the {@code IdentifierExpression}
	 * labeled alternative in {@link OrderingParser#expression}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitIdentifierExpression(OrderingParser.IdentifierExpressionContext ctx);
	/**
	 * Visit a parse tree produced by the {@code DotIdentifierExpression}
	 * labeled alternative in {@link OrderingParser#expression}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitDotIdentifierExpression(OrderingParser.DotIdentifierExpressionContext ctx);
	/**
	 * Visit a parse tree produced by {@link OrderingParser#function}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitFunction(OrderingParser.FunctionContext ctx);
	/**
	 * Visit a parse tree produced by {@link OrderingParser#simpleFunction}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitSimpleFunction(OrderingParser.SimpleFunctionContext ctx);
	/**
	 * Visit a parse tree produced by {@link OrderingParser#packagedFunction}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitPackagedFunction(OrderingParser.PackagedFunctionContext ctx);
	/**
	 * Visit a parse tree produced by {@link OrderingParser#functionArguments}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitFunctionArguments(OrderingParser.FunctionArgumentsContext ctx);
	/**
	 * Visit a parse tree produced by {@link OrderingParser#functionArgument}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitFunctionArgument(OrderingParser.FunctionArgumentContext ctx);
	/**
	 * Visit a parse tree produced by {@link OrderingParser#literal}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitLiteral(OrderingParser.LiteralContext ctx);
	/**
	 * Visit a parse tree produced by {@link OrderingParser#collationSpecification}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitCollationSpecification(OrderingParser.CollationSpecificationContext ctx);
	/**
	 * Visit a parse tree produced by {@link OrderingParser#direction}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitDirection(OrderingParser.DirectionContext ctx);
	/**
	 * Visit a parse tree produced by {@link OrderingParser#nullsPrecedence}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitNullsPrecedence(OrderingParser.NullsPrecedenceContext ctx);
	/**
	 * Visit a parse tree produced by {@link OrderingParser#identifier}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitIdentifier(OrderingParser.IdentifierContext ctx);
	/**
	 * Visit a parse tree produced by {@link OrderingParser#dotIdentifier}.
	 * @param ctx the parse tree
	 * @return the visitor result
	 */
	T visitDotIdentifier(OrderingParser.DotIdentifierContext ctx);
}
