/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later
 * See the lgpl.txt file in the root directory or http://www.gnu.org/licenses/lgpl-2.1.html
 */
package org.hibernate.boot.model.process.internal;

import java.util.function.Function;

import org.hibernate.boot.model.convert.internal.ClassBasedConverterDescriptor;
import org.hibernate.boot.model.convert.spi.ConverterDescriptor;
import org.hibernate.boot.model.convert.spi.JpaAttributeConverterCreationContext;
import org.hibernate.boot.registry.classloading.spi.ClassLoaderService;
import org.hibernate.boot.spi.MetadataBuildingContext;
import org.hibernate.mapping.BasicValue;
import org.hibernate.metamodel.mapping.JdbcMapping;
import org.hibernate.metamodel.model.convert.spi.JpaAttributeConverter;
import org.hibernate.type.BasicType;
import org.hibernate.type.descriptor.converter.AttributeConverterMutabilityPlanImpl;
import org.hibernate.type.descriptor.java.BasicJavaType;
import org.hibernate.type.descriptor.java.ImmutableMutabilityPlan;
import org.hibernate.type.descriptor.java.JavaType;
import org.hibernate.type.descriptor.java.MutabilityPlan;
import org.hibernate.type.descriptor.jdbc.JdbcType;
import org.hibernate.type.descriptor.jdbc.JdbcTypeIndicators;
import org.hibernate.type.internal.CustomMutabilityConvertedBasicTypeImpl;
import org.hibernate.type.spi.TypeConfiguration;

/**
 * @author Steve Ebersole
 */
public class NamedConverterResolution<J> implements BasicValue.Resolution<J> {

	public static <T> NamedConverterResolution<T> from(
			ConverterDescriptor converterDescriptor,
			Function<TypeConfiguration, BasicJavaType> explicitJtdAccess,
			Function<TypeConfiguration, JdbcType> explicitStdAccess,
			Function<TypeConfiguration, MutabilityPlan> explicitMutabilityPlanAccess,
			JdbcTypeIndicators sqlTypeIndicators,
			JpaAttributeConverterCreationContext converterCreationContext,
			MetadataBuildingContext context) {
		return fromInternal(
				explicitJtdAccess,
				explicitStdAccess,
				explicitMutabilityPlanAccess,
				converter( converterCreationContext, converterDescriptor ),
				sqlTypeIndicators,
				context
		);
	}

	public static <T> NamedConverterResolution<T> from(
			String name,
			Function<TypeConfiguration, BasicJavaType> explicitJtdAccess,
			Function<TypeConfiguration, JdbcType> explicitStdAccess,
			Function<TypeConfiguration, MutabilityPlan> explicitMutabilityPlanAccess,
			JdbcTypeIndicators sqlTypeIndicators,
			JpaAttributeConverterCreationContext converterCreationContext,
			MetadataBuildingContext context) {
		assert name.startsWith( ConverterDescriptor.TYPE_NAME_PREFIX );
		final String converterClassName = name.substring( ConverterDescriptor.TYPE_NAME_PREFIX.length() );

		final ClassBasedConverterDescriptor converterDescriptor = new ClassBasedConverterDescriptor(
				context.getBootstrapContext().getServiceRegistry()
						.getService( ClassLoaderService.class )
						.classForName( converterClassName ),
				context.getBootstrapContext().getClassmateContext()
		);

		return fromInternal(
				explicitJtdAccess,
				explicitStdAccess,
				explicitMutabilityPlanAccess,
				converter( converterCreationContext, converterDescriptor ),
				sqlTypeIndicators,
				context
		);
	}

	private static <T> JpaAttributeConverter<T, ?> converter(
			JpaAttributeConverterCreationContext converterCreationContext,
			ConverterDescriptor converterDescriptor) {
		//noinspection unchecked
		return (JpaAttributeConverter<T,?>) converterDescriptor.createJpaAttributeConverter(converterCreationContext);
	}

	private static <T> NamedConverterResolution<T> fromInternal(
			Function<TypeConfiguration, BasicJavaType> explicitJtdAccess,
			Function<TypeConfiguration, JdbcType> explicitStdAccess,
			Function<TypeConfiguration, MutabilityPlan> explicitMutabilityPlanAccess,
			JpaAttributeConverter<T,?> converter,
			JdbcTypeIndicators sqlTypeIndicators,
			MetadataBuildingContext context) {
		final TypeConfiguration typeConfiguration = context.getBootstrapContext().getTypeConfiguration();

		final JavaType<T> explicitJtd = explicitJtdAccess != null
				? explicitJtdAccess.apply( typeConfiguration )
				: null;

		final JavaType<T> domainJtd = explicitJtd != null
				? explicitJtd
				: converter.getDomainJavaType();

		final JdbcType explicitJdbcType = explicitStdAccess != null
				? explicitStdAccess.apply( typeConfiguration )
				: null;

		final JavaType<?> relationalJtd = converter.getRelationalJavaType();

		final JdbcType jdbcType = explicitJdbcType != null
				? explicitJdbcType
				: relationalJtd.getRecommendedJdbcType( sqlTypeIndicators );

		final MutabilityPlan<T> explicitMutabilityPlan = explicitMutabilityPlanAccess != null
				? explicitMutabilityPlanAccess.apply( typeConfiguration )
				: null;


		final MutabilityPlan<T> mutabilityPlan;
		if ( explicitMutabilityPlan != null ) {
			mutabilityPlan = explicitMutabilityPlan;
		}
		else if ( ! domainJtd.getMutabilityPlan().isMutable() ) {
			mutabilityPlan = ImmutableMutabilityPlan.instance();
		}
		else {
			mutabilityPlan = new AttributeConverterMutabilityPlanImpl<>( converter, true );
		}

		return new NamedConverterResolution<T>(
				domainJtd,
				relationalJtd,
				jdbcType,
				converter,
				mutabilityPlan,
				context.getBootstrapContext().getTypeConfiguration()
		);
	}


	private final JavaType<J> domainJtd;
	private final JavaType<?> relationalJtd;
	private final JdbcType jdbcType;

	private final JpaAttributeConverter<J,?> valueConverter;
	private final MutabilityPlan<J> mutabilityPlan;

	private final JdbcMapping jdbcMapping;

	private final BasicType<J> legacyResolvedType;

	public NamedConverterResolution(
			JavaType<J> domainJtd,
			JavaType<?> relationalJtd,
			JdbcType jdbcType,
			JpaAttributeConverter<J,?> valueConverter,
			MutabilityPlan<J> mutabilityPlan,
			TypeConfiguration typeConfiguration) {
		assert domainJtd != null;
		this.domainJtd = domainJtd;

		assert relationalJtd != null;
		this.relationalJtd = relationalJtd;

		assert jdbcType != null;
		this.jdbcType = jdbcType;

		assert valueConverter != null;
		this.valueConverter = valueConverter;

		assert mutabilityPlan != null;
		this.mutabilityPlan = mutabilityPlan;

		this.legacyResolvedType = new CustomMutabilityConvertedBasicTypeImpl<>(
				ConverterDescriptor.TYPE_NAME_PREFIX
						+ valueConverter.getConverterJavaType().getJavaType().getTypeName(),
				String.format(
						"BasicType adapter for AttributeConverter<%s,%s>",
						domainJtd.getJavaType().getTypeName(),
						relationalJtd.getJavaType().getTypeName()
				),
				jdbcType,
				valueConverter,
				mutabilityPlan
		);
		this.jdbcMapping = legacyResolvedType;
	}

	@Override
	public BasicType<J> getLegacyResolvedBasicType() {
		return legacyResolvedType;
	}

	@Override
	public JavaType<J> getDomainJavaType() {
		return domainJtd;
	}

	@Override
	public JavaType<?> getRelationalJavaType() {
		return relationalJtd;
	}

	@Override
	public JdbcType getJdbcType() {
		return jdbcType;
	}

	@Override
	public JdbcMapping getJdbcMapping() {
		return jdbcMapping;
	}

	@Override
	public JpaAttributeConverter<J,?> getValueConverter() {
		return valueConverter;
	}

	@Override
	public MutabilityPlan<J> getMutabilityPlan() {
		return mutabilityPlan;
	}

	@Override
	public String toString() {
		return "NamedConverterResolution(" + valueConverter.getConverterBean().getBeanClass().getName() + ')';
	}

}
