/*
 * Decompiled with CFR 0.152.
 */
package org.hibernate.engine.jdbc.batch.internal;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.LinkedHashSet;
import org.hibernate.HibernateException;
import org.hibernate.engine.jdbc.JdbcLogging;
import org.hibernate.engine.jdbc.batch.JdbcBatchLogging;
import org.hibernate.engine.jdbc.batch.spi.Batch;
import org.hibernate.engine.jdbc.batch.spi.BatchKey;
import org.hibernate.engine.jdbc.batch.spi.BatchObserver;
import org.hibernate.engine.jdbc.mutation.JdbcValueBindings;
import org.hibernate.engine.jdbc.mutation.TableInclusionChecker;
import org.hibernate.engine.jdbc.mutation.group.PreparedStatementDetails;
import org.hibernate.engine.jdbc.mutation.group.PreparedStatementGroup;
import org.hibernate.engine.jdbc.spi.JdbcCoordinator;
import org.hibernate.engine.jdbc.spi.JdbcServices;
import org.hibernate.engine.jdbc.spi.SqlExceptionHelper;
import org.hibernate.engine.jdbc.spi.SqlStatementLogger;
import org.hibernate.engine.spi.SharedSessionContractImplementor;
import org.hibernate.resource.jdbc.spi.JdbcObserver;
import org.hibernate.sql.model.ModelMutationLogging;

public class BatchImpl
implements Batch {
    private final BatchKey key;
    private final int batchSizeToUse;
    private final PreparedStatementGroup statementGroup;
    private final JdbcCoordinator jdbcCoordinator;
    private final SqlStatementLogger sqlStatementLogger;
    private final SqlExceptionHelper sqlExceptionHelper;
    private final LinkedHashSet<BatchObserver> observers = new LinkedHashSet();
    private int batchPosition;
    private boolean batchExecuted;

    public BatchImpl(BatchKey key, PreparedStatementGroup statementGroup, int batchSizeToUse, JdbcCoordinator jdbcCoordinator) {
        if (key == null) {
            throw new IllegalArgumentException("Batch key cannot be null");
        }
        if (jdbcCoordinator == null) {
            throw new IllegalArgumentException("JDBC coordinator cannot be null");
        }
        this.key = key;
        this.jdbcCoordinator = jdbcCoordinator;
        this.statementGroup = statementGroup;
        JdbcServices jdbcServices = jdbcCoordinator.getJdbcSessionOwner().getJdbcSessionContext().getSessionFactory().getFastSessionServices().jdbcServices;
        this.sqlStatementLogger = jdbcServices.getSqlStatementLogger();
        this.sqlExceptionHelper = jdbcServices.getSqlExceptionHelper();
        this.batchSizeToUse = batchSizeToUse;
        if (JdbcBatchLogging.BATCH_TRACE_ENABLED) {
            JdbcBatchLogging.BATCH_LOGGER.tracef("Created Batch (%s) - `%s`", batchSizeToUse, (Object)key.toLoggableString());
        }
    }

    @Override
    public final BatchKey getKey() {
        return this.key;
    }

    @Override
    public PreparedStatementGroup getStatementGroup() {
        return this.statementGroup;
    }

    @Override
    public void addObserver(BatchObserver observer) {
        this.observers.add(observer);
    }

    @Override
    public void addToBatch(JdbcValueBindings jdbcValueBindings, TableInclusionChecker inclusionChecker) {
        if (JdbcBatchLogging.BATCH_TRACE_ENABLED) {
            JdbcBatchLogging.BATCH_LOGGER.tracef("Adding to JDBC batch (%s) - `%s`", this.batchPosition + 1, (Object)this.getKey().toLoggableString());
        }
        SharedSessionContractImplementor session = (SharedSessionContractImplementor)this.jdbcCoordinator.getJdbcSessionOwner();
        try {
            this.getStatementGroup().forEachStatement((tableName, statementDetails) -> {
                if (inclusionChecker != null && !inclusionChecker.include(statementDetails.getMutatingTableDetails())) {
                    if (ModelMutationLogging.MODEL_MUTATION_LOGGER_TRACE_ENABLED) {
                        ModelMutationLogging.MODEL_MUTATION_LOGGER.tracef("Skipping addBatch for table : %s (batch-position=%s)", (Object)statementDetails.getMutatingTableDetails().getTableName(), (Object)(this.batchPosition + 1));
                    }
                    return;
                }
                PreparedStatement statement = statementDetails.resolveStatement();
                this.sqlStatementLogger.logStatement(statementDetails.getSqlString());
                jdbcValueBindings.beforeStatement((PreparedStatementDetails)statementDetails, session);
                try {
                    statement.addBatch();
                }
                catch (SQLException e) {
                    JdbcBatchLogging.BATCH_LOGGER.debug((Object)"SQLException escaped proxy", (Throwable)e);
                    throw this.sqlExceptionHelper.convert(e, "Could not perform addBatch", statementDetails.getSqlString());
                }
                finally {
                    jdbcValueBindings.afterStatement(statementDetails.getMutatingTableDetails(), session);
                }
            });
        }
        catch (RuntimeException e) {
            this.abortBatch(e);
            throw e;
        }
        ++this.batchPosition;
        if (this.batchPosition == this.batchSizeToUse) {
            this.notifyObserversImplicitExecution();
            this.performExecution();
            this.batchPosition = 0;
            this.batchExecuted = true;
        }
    }

    protected void releaseStatements() {
        this.statementGroup.forEachStatement((tableName, statementDetails) -> {
            if (statementDetails.getStatement() == null) {
                JdbcBatchLogging.BATCH_LOGGER.debugf("PreparedStatementDetails did not contain PreparedStatement on #releaseStatements : %s", (Object)statementDetails.getSqlString());
                return;
            }
            this.clearBatch((PreparedStatementDetails)statementDetails);
        });
        this.statementGroup.release();
        this.jdbcCoordinator.afterStatementExecution();
    }

    protected void clearBatch(PreparedStatementDetails statementDetails) {
        PreparedStatement statement = statementDetails.getStatement();
        assert (statement != null);
        try {
            if (!statement.isClosed()) {
                statement.clearBatch();
            }
        }
        catch (SQLException e) {
            JdbcBatchLogging.BATCH_MESSAGE_LOGGER.unableToReleaseBatchStatement();
        }
    }

    protected final void notifyObserversExplicitExecution() {
        for (BatchObserver observer : this.observers) {
            observer.batchExplicitlyExecuted();
        }
    }

    protected final void notifyObserversImplicitExecution() {
        for (BatchObserver observer : this.observers) {
            observer.batchImplicitlyExecuted();
        }
    }

    protected void abortBatch(Exception cause) {
        try {
            this.jdbcCoordinator.abortBatch();
        }
        catch (RuntimeException e) {
            cause.addSuppressed(e);
        }
    }

    @Override
    public void execute() {
        this.notifyObserversExplicitExecution();
        if (this.getStatementGroup().getNumberOfStatements() == 0) {
            return;
        }
        try {
            if (this.batchPosition == 0) {
                if (!this.batchExecuted && JdbcBatchLogging.BATCH_DEBUG_ENABLED) {
                    JdbcBatchLogging.BATCH_LOGGER.debugf("No batched statements to execute - %s", (Object)this.getKey().toLoggableString());
                }
            } else {
                this.performExecution();
            }
        }
        finally {
            this.releaseStatements();
        }
    }

    protected void performExecution() {
        if (JdbcBatchLogging.BATCH_TRACE_ENABLED) {
            JdbcBatchLogging.BATCH_LOGGER.tracef("Executing JDBC batch (%s / %s) - `%s`", this.batchPosition, this.batchSizeToUse, (Object)this.getKey().toLoggableString());
        }
        JdbcObserver observer = this.jdbcCoordinator.getJdbcSessionOwner().getJdbcSessionContext().getObserver();
        try {
            this.getStatementGroup().forEachStatement((tableName, statementDetails) -> {
                block9: {
                    String sql = statementDetails.getSqlString();
                    PreparedStatement statement = statementDetails.getStatement();
                    if (statement == null) {
                        return;
                    }
                    try {
                        if (statementDetails.getMutatingTableDetails().isIdentifierTable()) {
                            int[] rowCounts;
                            try {
                                observer.jdbcExecuteBatchStart();
                                rowCounts = statement.executeBatch();
                            }
                            finally {
                                observer.jdbcExecuteBatchEnd();
                            }
                            this.checkRowCounts(rowCounts, (PreparedStatementDetails)statementDetails);
                            break block9;
                        }
                        statement.executeBatch();
                    }
                    catch (SQLException e) {
                        this.abortBatch(e);
                        JdbcBatchLogging.BATCH_MESSAGE_LOGGER.unableToExecuteBatch(e, sql);
                        throw this.sqlExceptionHelper.convert(e, "could not execute batch", sql);
                    }
                    catch (RuntimeException re) {
                        this.abortBatch(re);
                        JdbcBatchLogging.BATCH_MESSAGE_LOGGER.unableToExecuteBatch(re, sql);
                        throw re;
                    }
                }
            });
        }
        finally {
            this.batchPosition = 0;
        }
    }

    private void checkRowCounts(int[] rowCounts, PreparedStatementDetails statementDetails) throws SQLException, HibernateException {
        int expectedNumberOfCounts;
        int numberOfRowCounts = rowCounts.length;
        if (this.batchPosition != 0 && numberOfRowCounts != (expectedNumberOfCounts = this.batchPosition / this.getStatementGroup().getNumberOfStatements())) {
            JdbcLogging.JDBC_MESSAGE_LOGGER.unexpectedRowCounts(statementDetails.getMutatingTableDetails().getTableName(), numberOfRowCounts, expectedNumberOfCounts);
        }
        for (int i = 0; i < numberOfRowCounts; ++i) {
            statementDetails.getExpectation().verifyOutcome(rowCounts[i], statementDetails.getStatement(), i, statementDetails.getSqlString());
        }
    }

    @Override
    public void release() {
        PreparedStatementGroup statementGroup;
        if (JdbcBatchLogging.BATCH_MESSAGE_LOGGER.isInfoEnabled() && (statementGroup = this.getStatementGroup()).getNumberOfStatements() != 0 && statementGroup.hasMatching(statementDetails -> statementDetails.getStatement() != null)) {
            JdbcBatchLogging.BATCH_MESSAGE_LOGGER.batchContainedStatementsOnRelease();
        }
        this.releaseStatements();
        this.observers.clear();
    }

    public String toString() {
        return "BatchImpl(" + this.getKey().toLoggableString() + ")";
    }
}

