/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later.
 * See the lgpl.txt file in the root directory or http://www.gnu.org/licenses/lgpl-2.1.html.
 */
package org.hibernate.boot.spi;

import org.hibernate.AssertionFailure;
import org.hibernate.HibernateException;

/**
 * Enumerates various access strategies for accessing entity values.
 *
 * @author Hardy Ferentschik
 */
public enum AccessType {
	/**
	 * Default access strategy is property
	 */
	DEFAULT( "property" ),

	/**
	 * Access to value via property
	 */
	PROPERTY( "property" ),

	/**
	 * Access to value via field
	 */
	FIELD( "field" ),

	/**
	 * Access to value via record component accessor
	 */
	RECORD( "record" );

	private final String accessType;

	AccessType(String type) {
		this.accessType = type;
	}

	/**
	 * Retrieves the external name for this access type
	 *
	 * @return The external name
	 */
	public String getType() {
		return accessType;
	}

	/**
	 * Resolve an externalized name to the {@code AccessType} value it names.
	 *
	 * @param externalName The external name
	 *
	 * @return The matching {@code AccessType};
	 *         {@link #DEFAULT} is returned rather than {@code null}
	 */
	public static AccessType getAccessStrategy(String externalName) {
		if ( externalName == null ) {
			return DEFAULT;
		}
		for ( AccessType value : values() ) {
			if ( value.accessType.equals( externalName ) ) {
				return value;
			}
		}
		return DEFAULT; // because sometimes we get passed magic values like "unsupported"
	}

	/**
	 * Convert the JPA access type to the corresponding {@link AccessType} value.
	 *
	 * @param type The JPA access type
	 *
	 * @return The Hibernate {@link AccessType}
	 */
	public static AccessType getAccessStrategy(jakarta.persistence.AccessType type) {
		if ( type == null ) {
			return DEFAULT;
		}
		switch ( type ) {
			case FIELD:
				return FIELD;
			case PROPERTY:
				return PROPERTY;
			default:
				throw new AssertionFailure( "unrecognized AccessType" );
		}
	}
}
