/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later
 * See the lgpl.txt file in the root directory or http://www.gnu.org/licenses/lgpl-2.1.html
 */
package org.hibernate.annotations;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import org.hibernate.Remove;
import org.hibernate.generator.EventType;
import org.hibernate.generator.internal.CurrentTimestampGeneration;
import org.hibernate.tuple.GenerationTiming;

import static java.lang.annotation.ElementType.ANNOTATION_TYPE;
import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;
import static org.hibernate.generator.EventType.INSERT;
import static org.hibernate.generator.EventType.UPDATE;

/**
 * Specifies that the annotated field of property is a generated timestamp,
 * and also specifies the {@linkplain #timing() timing} of the timestamp
 * generation, and whether it is generated in Java or by the database:
 * <ul>
 * <li>{@link SourceType#VM source = VM} indicates that the virtual machine
 *     {@linkplain java.time.Clock#instant() current instant}
 *     is used, and
 * <li>{@link SourceType#DB source = DB} indicates that the database
 *     {@code current_timestamp} function should be used.
 * </ul>
 * <p>
 * By default, the timestamp is generated by the database, which requires
 * an extra round trip to the database to fetch the generated value.
 * <p>
 * This annotation may be used in combination with the JPA-defined
 * {@link jakarta.persistence.Version} annotation.
 * <p>
 * The annotated property may be of any one of the following types:
 * {@link java.util.Date},
 * {@link java.util.Calendar},
 * {@link java.sql.Date},
 * {@link java.sql.Time},
 * {@link java.sql.Timestamp},
 * {@link java.time.Instant},
 * {@link java.time.LocalDate},
 * {@link java.time.LocalDateTime},
 * {@link java.time.LocalTime},
 * {@link java.time.MonthDay},
 * {@link java.time.OffsetDateTime},
 * {@link java.time.OffsetTime},
 * {@link java.time.Year},
 * {@link java.time.YearMonth}, or
 * {@link java.time.ZonedDateTime}.
 *
 * @see UpdateTimestamp
 * @see CreationTimestamp
 * @see CurrentTimestampGeneration
 *
 * @since 6.0
 *
 * @author Steve Ebersole
 */
@ValueGenerationType(generatedBy = CurrentTimestampGeneration.class)
@Retention(RUNTIME)
@Target({ FIELD, METHOD, ANNOTATION_TYPE })
public @interface CurrentTimestamp {
	/**
	 * Determines when the timestamp is generated. But default, it is updated
	 * when any SQL {@code insert} or {@code update} statement is executed.
	 * If it should be generated just once, on the initial SQL {@code insert},
	 * explicitly specify {@link EventType#INSERT event = INSERT}.
	 */
	EventType[] event() default {INSERT, UPDATE};

	/**
	 * Determines when the timestamp is generated. But default, it is updated
	 * when any SQL {@code insert} or {@code update} statement is executed.
	 * If it should be generated just once, on the initial SQL {@code insert},
	 * explicitly specify {@link GenerationTiming#INSERT timing = INSERT}.
	 *
	 * @deprecated This was introduced in error
	 */
	@Deprecated(since = "6.2") @Remove
	GenerationTiming timing() default GenerationTiming.ALWAYS;

	/**
	 * Specifies how the timestamp is generated. By default, it is generated
	 * by the database, and fetched using a subsequent {@code select} statement.
	 */
	SourceType source() default SourceType.DB;
}
