/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later.
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.annotations;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import org.hibernate.Session;
import org.hibernate.tuple.AnnotationValueGeneration;
import org.hibernate.tuple.ValueGenerator;
import org.hibernate.tuple.VmValueGeneration;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * Marks a field or property of an entity as automatically generated by
 * code written in Java, before any SQL statement to {@code insert} or
 * {@code update} the entity is executed, specifying an implementation
 * of {@link ValueGenerator} used for generating its values.
 * <p>
 * It is the responsibility of the client to ensure that a specified
 * {@linkplain #type generator type} produces values which are assignable
 * to the annotated property.
 * <p>
 * This annotation is only useful for values generated in the Java code,
 * and it is not used for generating the values of entity identifiers:
 * <ul>
 * <li>For identifier generators, use {@link GenericGenerator} or
 *     {@link IdGeneratorType}.
 * <li>If the value of a field or property is generated by the database
 *     when an {@code insert} or {@code update} statement is executed,
 *     use the {@link Generated} annotation.
 * </ul>
 *
 * @author Gunnar Morling
 *
 * @deprecated {@link ValueGenerationType} and {@link AnnotationValueGeneration}
 *             now provide a much more powerful and typesafe alternative
 */
@ValueGenerationType( generatedBy = VmValueGeneration.class )
@Retention(RUNTIME)
@Target({FIELD, METHOD})
@Deprecated(since = "6.0")
public @interface GeneratorType {
	/**
	 * A class that implements {@link ValueGenerator}, which will be called to
	 * {@linkplain ValueGenerator#generateValue(Session, Object) generate values}
	 * of the annotated field or property.
	 *
	 * @return the value generator type
	 */
	Class<? extends ValueGenerator<?>> type();

	/**
	 * Specifies when values should be generated:
	 * <ul>
	 * <li>If {@link GenerationTime#INSERT}, the value will be generated before
	 *     each SQL {@code insert} statement is executed.
	 * <li>If {@link GenerationTime#UPDATE}, the value will be generated before
	 *     each SQL {@code update} statement is executed.
	 * <li>If {@link GenerationTime#ALWAYS}, the value will be generated before
	 *     each SQL {@code insert} or {@code update} statement is executed.
	 * </ul>
	 */
	GenerationTime when() default GenerationTime.ALWAYS;
}
