/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later.
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.annotations;

import java.lang.annotation.Repeatable;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.ElementType.TYPE;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * Specifies a custom SQL DML statement to be used in place of the default SQL
 * generated by Hibernate when an entity or collection row is deleted from the
 * database.
 * <p>
 * The given {@linkplain #sql SQL statement} must have exactly the number of JDBC
 * {@code ?} parameters that Hibernate expects, in the exact order Hibernate
 * expects. The primary key columns come before the version column if the
 * entity is versioned.
 *
 * @author Laszlo Benke
 */
@Target({TYPE, FIELD, METHOD})
@Retention(RUNTIME)
@Repeatable(SQLDeletes.class)
public @interface SQLDelete {
	/**
	 * Procedure name or SQL {@code DELETE} statement.
	 */
	String sql();

	/**
	 * Is the statement callable (aka a {@link java.sql.CallableStatement})?
	 */
	boolean callable() default false;

	/**
	 * For persistence operation what style of determining results (success/failure) is to be used.
	 */
	ResultCheckStyle check() default ResultCheckStyle.NONE;

	/**
	 * The name of the table in the case of an entity with {@link jakarta.persistence.SecondaryTable
	 * secondary tables}, defaults to the primary table.
	 *
	 * @return the name of the table
	 *
	 * @since 6.2
	 */
	String table() default "";
}
