/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later.
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.annotations;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.ElementType.TYPE;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * Specifies a custom HQL/JPQL query to be used in place of the default SQL
 * generated by Hibernate when an entity or collection is fetched from the
 * database by id. This occurs when:
 * <ul>
 * <li>an association to an entity is fetched lazily,
 * <li>a collection is fetched lazily, or
 * <li>when an entity is retrieved using {@link org.hibernate.Session#get}
 *     or {@link org.hibernate.Session#find}.
 * </ul>
 * <p>
 * The given {@linkplain #query HQL query} must have exactly one parameter
 * which accepts the {@linkplain jakarta.persistence.Id id} of the entity
 * or of the entity which owns the collection. It must return a single item
 * in the select list, and that item must be the correct type of entity or
 * collection element.
 *
 * @see SQLSelect
 *
 * @author Gavin King
 *
 * @since 6.2
 *
 * @implNote This annotation is just an abbreviation for {@link Loader}
 *           together with {@link NamedQuery}.
 */
@Target({TYPE, FIELD, METHOD})
@Retention(RUNTIME)
public @interface HQLSelect {
	/**
	 * The HQL {@code SELECT} statement.
	 */
	String query();
}
