/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later.
 * See the lgpl.txt file in the root directory or http://www.gnu.org/licenses/lgpl-2.1.html.
 */
package org.hibernate.sql.model.ast.builder;

import java.util.ArrayList;
import java.util.List;

import org.hibernate.engine.jdbc.mutation.ParameterUsage;
import org.hibernate.engine.jdbc.spi.JdbcServices;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.metamodel.mapping.JdbcMapping;
import org.hibernate.sql.model.MutationTarget;
import org.hibernate.sql.model.MutationType;
import org.hibernate.sql.model.TableMapping;
import org.hibernate.sql.model.ast.ColumnValueBinding;
import org.hibernate.sql.model.ast.ColumnValueParameterList;
import org.hibernate.sql.model.ast.MutatingTableReference;
import org.hibernate.sql.model.ast.TableMutation;

/**
 * Base support for TableMutationBuilder implementations
 *
 * @author Steve Ebersole
 */
public abstract class AbstractTableMutationBuilder<M extends TableMutation<?>> implements TableMutationBuilder<M> {
	private final SessionFactoryImplementor sessionFactory;

	private final MutationType mutationType;
	private final MutationTarget<?> mutationTarget;

	private final MutatingTableReference mutatingTable;
	private final ColumnValueParameterList parameters;

	public AbstractTableMutationBuilder(
			MutationType mutationType,
			MutationTarget<?> mutationTarget,
			TableMapping table,
			SessionFactoryImplementor sessionFactory) {
		this( mutationType, mutationTarget, new MutatingTableReference( table ), sessionFactory );
	}

	public AbstractTableMutationBuilder(
			MutationType mutationType,
			MutationTarget<?> mutationTarget,
			MutatingTableReference mutatingTable,
			SessionFactoryImplementor sessionFactory) {
		this.mutationType = mutationType;
		this.mutationTarget = mutationTarget;
		this.sessionFactory = sessionFactory;

		this.mutatingTable = mutatingTable;
		this.parameters = new ColumnValueParameterList( mutatingTable, null, 0 );
	}

	protected MutationTarget<?> getMutationTarget() {
		return mutationTarget;
	}

	@Override
	public MutatingTableReference getMutatingTable() {
		return mutatingTable;
	}

	protected ColumnValueParameterList getParameters() {
		return parameters;
	}

	protected SessionFactoryImplementor getSessionFactory() {
		return sessionFactory;
	}

	protected JdbcServices getJdbcServices() {
		return sessionFactory.getJdbcServices();
	}

	protected void addColumn(
			String columnName,
			String columnWriteFragment,
			JdbcMapping jdbcMapping,
			List<ColumnValueBinding> list) {
		final ColumnValueBinding valueBinding = createValueBinding( columnName, columnWriteFragment, jdbcMapping );
		list.add( valueBinding );
	}

	protected void addColumn(
			String columnName,
			String columnWriteFragment,
			JdbcMapping jdbcMapping,
			ParameterUsage parameterUsage,
			List<ColumnValueBinding> list) {
		final ColumnValueBinding valueBinding = createValueBinding( columnName, columnWriteFragment, jdbcMapping, parameterUsage );
		list.add( valueBinding );
	}

	protected ColumnValueBinding createValueBinding(
			String columnName,
			String columnWriteFragment,
			JdbcMapping jdbcMapping) {
		return createValueBinding( columnName, columnWriteFragment, jdbcMapping, ParameterUsage.SET );
	}
	protected ColumnValueBinding createValueBinding(
			String columnName,
			String customWriteExpression,
			JdbcMapping jdbcMapping,
			ParameterUsage parameterUsage) {
		return ColumnValueBindingBuilder.createValueBinding(
				columnName,
				customWriteExpression,
				jdbcMapping,
				getMutatingTable(),
				parameterUsage,
				parameters::apply
		);
	}

	@SafeVarargs
	protected final <T> List<T> combine(List<T> list1, List<T>... additionalLists) {
		final ArrayList<T> combined = list1 == null
				? new ArrayList<>()
				: new ArrayList<>( list1 );

		if ( additionalLists != null ) {
			for ( int i = 0; i < additionalLists.length; i++ ) {
				if ( additionalLists[i] == null ) {
					continue;
				}
				combined.addAll( additionalLists[i] );
			}
		}

		return combined;
	}

	@Override
	public String toString() {
		return "TableMutationBuilder( " + mutationType + " - `" + mutatingTable.getTableName() + "`)";
	}
}
