/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later
 * See the lgpl.txt file in the root directory or http://www.gnu.org/licenses/lgpl-2.1.html
 */
package org.hibernate.engine.spi;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Map.Entry;

import org.hibernate.cache.spi.access.CollectionDataAccess;
import org.hibernate.cache.spi.access.EntityDataAccess;
import org.hibernate.collection.spi.PersistentCollection;
import org.hibernate.engine.internal.CacheHelper;
import org.hibernate.internal.CoreLogging;
import org.hibernate.internal.util.IndexedConsumer;
import org.hibernate.internal.util.collections.CollectionHelper;
import org.hibernate.metamodel.mapping.EntityIdentifierMapping;
import org.hibernate.metamodel.mapping.EntityMappingType;
import org.hibernate.metamodel.mapping.PluralAttributeMapping;
import org.hibernate.persister.collection.CollectionPersister;
import org.hibernate.persister.entity.EntityPersister;
import org.hibernate.type.Type;

import org.jboss.logging.Logger;

import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;

/**
 * Keeps track of:<ul>
 *     <li>entity and collection keys that are available for batch fetching</li>
 *     <li>details related to queries which load entities with sub-select-fetchable collections</li>
 * </ul>
 *
 * @author Gavin King
 * @author Steve Ebersole
 * @author Guenther Demetz
 */
public class BatchFetchQueue {
	private static final Logger LOG = CoreLogging.logger( BatchFetchQueue.class );

	private final PersistenceContext context;

	/**
	 * A map of {@link SubselectFetch subselect-fetch descriptors} keyed by the
	 * {@link EntityKey} against which the descriptor is registered.
	 */
	private @Nullable Map<EntityKey, SubselectFetch> subselectsByEntityKey;

	/**
	 * Used to hold information about the entities that are currently eligible for batch-fetching. Ultimately
	 * used by {@link #getBatchLoadableEntityIds} to build entity load batches.
	 * <p>
	 * A Map structure is used to segment the keys by entity type since loading can only be done for a particular entity
	 * type at a time.
	 */
	private @Nullable Map <String,LinkedHashSet<EntityKey>> batchLoadableEntityKeys;

	/**
	 * Used to hold information about the collections that are currently eligible for batch-fetching. Ultimately
	 * used by {@link #getCollectionBatch} to build collection load batches.
	 */
	private @Nullable Map<String, LinkedHashMap<CollectionEntry, PersistentCollection<?>>> batchLoadableCollections;

	/**
	 * Constructs a queue for the given context.
	 *
	 * @param context The owning context.
	 */
	public BatchFetchQueue(PersistenceContext context) {
		this.context = context;
	}

	/**
	 * Clears all entries from this fetch queue.
	 * <p>
	 * Called after flushing or clearing the session.
	 */
	public void clear() {
		batchLoadableEntityKeys = null;
		batchLoadableCollections = null;
		subselectsByEntityKey = null;
	}


	// sub-select support ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Retrieve the fetch descriptor associated with the given entity key.
	 *
	 * @param key The entity key for which to locate any defined subselect fetch.
	 * @return The fetch descriptor; may return null if no subselect fetch queued for
	 * this entity key.
	 */
	public @Nullable SubselectFetch getSubselect(EntityKey key) {
		if ( subselectsByEntityKey == null ) {
			return null;
		}
		return subselectsByEntityKey.get( key );
	}

	/**
	 * Adds a subselect fetch descriptor for the given entity key.
	 *
	 * @param key The entity for which to register the subselect fetch.
	 * @param subquery The fetch descriptor.
	 */
	public void addSubselect(EntityKey key, SubselectFetch subquery) {
		if ( subselectsByEntityKey == null ) {
			subselectsByEntityKey = CollectionHelper.mapOfSize( 12 );
		}

		final SubselectFetch previous = subselectsByEntityKey.put( key, subquery );
		if ( previous != null && LOG.isDebugEnabled() ) {
			LOG.debugf(
					"SubselectFetch previously registered with BatchFetchQueue for `%s#s`",
					key.getEntityName(),
					key.getIdentifier()
			);
		}
	}

	/**
	 * After evicting or deleting an entity, we don't need to
	 * know the query that was used to load it anymore (don't
	 * call this after loading the entity, since we might still
	 * need to load its collections)
	 */
	public void removeSubselect(EntityKey key) {
		if ( subselectsByEntityKey != null ) {
			subselectsByEntityKey.remove( key );
		}
	}

	// entity batch support ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * If an EntityKey represents a batch loadable entity, add
	 * it to the queue.
	 * <p>
	 * Note that the contract here is such that any key passed in should
	 * previously have been been checked for existence within the
	 * {@link PersistenceContext}; failure to do so may cause the
	 * referenced entity to be included in a batch even though it is
	 * already associated with the {@link PersistenceContext}.
	 */
	public void addBatchLoadableEntityKey(EntityKey key) {
		if ( key.isBatchLoadable( context.getSession().getLoadQueryInfluencers() ) ) {
			if ( batchLoadableEntityKeys == null ) {
				batchLoadableEntityKeys = CollectionHelper.mapOfSize( 12 );
			}
			final LinkedHashSet<EntityKey> keysForEntity = batchLoadableEntityKeys.computeIfAbsent(
					key.getEntityName(),
					k -> CollectionHelper.linkedSetOfSize( 8 )
			);

			keysForEntity.add( key );
		}
	}


	/**
	 * After evicting or deleting or loading an entity, we don't
	 * need to batch fetch it anymore, remove it from the queue
	 * if necessary
	 */
	public void removeBatchLoadableEntityKey(EntityKey key) {
		if ( key.isBatchLoadable( context.getSession().getLoadQueryInfluencers() )
				&& batchLoadableEntityKeys != null ) {
			final LinkedHashSet<EntityKey> set = batchLoadableEntityKeys.get( key.getEntityName() );
			if ( set != null ) {
				set.remove( key );
			}
		}
	}

	/**
	 * Intended for test usage. Really has no use-case in Hibernate proper.
	 */
	public boolean containsEntityKey(EntityKey key) {
		if ( key.isBatchLoadable( context.getSession().getLoadQueryInfluencers() ) && batchLoadableEntityKeys != null ) {
			LinkedHashSet<EntityKey> set = batchLoadableEntityKeys.get( key.getEntityName() );
			if ( set != null ) {
				return set.contains( key );
			}
		}
		return false;
	}

	/**
	 * A "collector" form of {@link #getBatchLoadableEntityIds}. Useful
	 * in cases where we want a specially created array/container - allows
	 * creation of concretely typed array for ARRAY param binding to ensure
	 * the driver does not need to cast/copy the values array.
	 */
	public <T> void collectBatchLoadableEntityIds(
			final int domainBatchSize,
			IndexedConsumer<T> collector,
			final @NonNull T loadingId,
			final EntityMappingType entityDescriptor) {
		// make sure we load the id being loaded in the batch!
		collector.accept( 0, loadingId );

		if ( batchLoadableEntityKeys == null ) {
			return;
		}

		final LinkedHashSet<EntityKey> set = batchLoadableEntityKeys.get( entityDescriptor.getEntityName() );
		if ( set == null ) {
			return;
		}

		final EntityIdentifierMapping identifierMapping = entityDescriptor.getIdentifierMapping();

		int batchPosition = 1;
		int end = -1;
		boolean checkForEnd = false;

		for ( EntityKey key : set ) {
			if ( checkForEnd && batchPosition == end ) {
				// the first id found after the given id
				return;
			}

			if ( identifierMapping.areEqual( loadingId, key.getIdentifier(), context.getSession() ) ) {
				end = batchPosition;
			}
			else {
				if ( !isCached( key, entityDescriptor.getEntityPersister() ) ) {
					//noinspection unchecked
					collector.accept( batchPosition++, (T) key.getIdentifier() );
				}
			}

			if ( batchPosition == domainBatchSize ) {
				// end of array, start filling again from start
				batchPosition = 1;
				if ( end != -1 ) {
					checkForEnd = true;
				}
			}
		}
	}

	/**
	 * Get a batch of unloaded identifiers for this class, using a slightly
	 * complex algorithm that tries to grab keys registered immediately after
	 * the given key.
	 */
	public Object [] getBatchLoadableEntityIds(
			final EntityMappingType entityDescriptor,
			final Object loadingId,
			final int maxBatchSize) {

		final Object[] ids = new Object[maxBatchSize];
		// make sure we load the id being loaded in the batch!
		ids[0] = loadingId;

		if ( batchLoadableEntityKeys == null ) {
			return ids;
		}

		int i = 1;
		int end = -1;
		boolean checkForEnd = false;

		// TODO: this needn't exclude subclasses...

		final LinkedHashSet<EntityKey> set =
				batchLoadableEntityKeys.get( entityDescriptor.getEntityName() );
		final EntityPersister entityPersister = entityDescriptor.getEntityPersister();
		final Type identifierType = entityPersister.getIdentifierType();
		if ( set != null ) {
			for ( EntityKey key : set ) {
				if ( checkForEnd && i == end ) {
					// the first id found after the given id
					return ids;
				}

				if ( identifierType.isEqual( loadingId, key.getIdentifier() ) ) {
					end = i;
				}
				else {
					if ( !isCached( key, entityPersister ) ) {
						ids[i++] = key.getIdentifier();
					}
				}

				if ( i == maxBatchSize ) {
					i = 1; // end of array, start filling again from start
					if ( end != -1 ) {
						checkForEnd = true;
					}
				}
			}
		}

		//we ran out of ids to try
		return ids;
	}

	private boolean isCached(EntityKey entityKey, EntityPersister persister) {
		final SharedSessionContractImplementor session = context.getSession();
		if ( context.getSession().getCacheMode().isGetEnabled() && persister.canReadFromCache() ) {
			final EntityDataAccess cache = persister.getCacheAccessStrategy();
			final Object key = cache.generateCacheKey(
					entityKey.getIdentifier(),
					persister,
					session.getFactory(),
					session.getTenantIdentifier()
			);
			return CacheHelper.fromSharedCache( session, key, persister, cache ) != null;
		}
		return false;
	}


	// collection batch support ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * If a CollectionEntry represents a batch loadable collection, add
	 * it to the queue.
	 */
	public void addBatchLoadableCollection(PersistentCollection<?> collection, CollectionEntry ce) {
		final CollectionPersister persister = ce.getLoadedPersister();

		if ( batchLoadableCollections == null ) {
			batchLoadableCollections = CollectionHelper.mapOfSize( 12 );
		}

		assert persister != null : "@AssumeAssertion(nullness)";
		final LinkedHashMap<CollectionEntry, PersistentCollection<?>> map =
				batchLoadableCollections.computeIfAbsent(
						persister.getRole(),
						k -> CollectionHelper.linkedMapOfSize( 16 )
				);

		map.put( ce, collection );
	}

	/**
	 * After a collection was initialized or evicted, we don't
	 * need to batch fetch it anymore, remove it from the queue
	 * if necessary
	 */
	public void removeBatchLoadableCollection(CollectionEntry ce) {
		final CollectionPersister persister = ce.getLoadedPersister();
		if ( batchLoadableCollections == null ) {
			return;
		}
		assert persister != null : "@AssumeAssertion(nullness)";
		LinkedHashMap<CollectionEntry, PersistentCollection<?>> map =
				batchLoadableCollections.get( persister.getRole() );
		if ( map != null ) {
			map.remove( ce );
		}
	}


	/**
	 * A "collector" form of {@link #getCollectionBatch}. Useful
	 * in cases where we want a specially created array/container - allows
	 * creation of concretely typed array for ARRAY param binding to ensure
	 * the driver does not need to cast/copy the values array.
	 */
	public <T> void collectBatchLoadableCollectionKeys(
			int batchSize,
			IndexedConsumer<T> collector,
			@NonNull T keyBeingLoaded,
			PluralAttributeMapping pluralAttributeMapping) {
		collector.accept( 0, keyBeingLoaded );

		if ( batchLoadableCollections == null ) {
			return;
		}

		final LinkedHashMap<CollectionEntry, PersistentCollection<?>> map =
				batchLoadableCollections.get( pluralAttributeMapping.getNavigableRole().getFullPath() );
		if ( map == null ) {
			return;
		}

		int i = 1;
		int end = -1;
		boolean checkForEnd = false;

		for ( Entry<CollectionEntry, PersistentCollection<?>> me : map.entrySet() ) {
			final CollectionEntry ce = me.getKey();
			final Object loadedKey = ce.getLoadedKey();
			final PersistentCollection<?> collection = me.getValue();

			if ( loadedKey == null ) {
				// the loadedKey of the collectionEntry might be null as it might have been reset to null
				// (see for example Collections.processDereferencedCollection()
				// and CollectionEntry.afterAction())
				// though we clear the queue on flush, it seems like a good idea to guard
				// against potentially null loadedKeys (which leads to various NPEs as demonstrated in HHH-7821).
				continue;
			}

			if ( collection.wasInitialized() ) {
				// should never happen
				LOG.warn( "Encountered initialized collection in BatchFetchQueue, this should not happen." );
				continue;
			}

			if ( checkForEnd && i == end ) {
				// the first key found after the given key
				return;
			}

			final boolean isEqual = pluralAttributeMapping.getKeyDescriptor().areEqual(
					keyBeingLoaded,
					loadedKey,
					context.getSession()
			);
//			final boolean isEqual = collectionPersister.getKeyType().isEqual(
//					id,
//					loadedKey,
//					collectionPersister.getFactory()
//			);

			if ( isEqual ) {
				end = i;
			}
			else if ( !isCached( loadedKey, pluralAttributeMapping.getCollectionDescriptor() ) ) {
				//noinspection unchecked
				collector.accept( i++, (T) loadedKey );
			}

			if ( i == batchSize ) {
				//end of array, start filling again from start
				i = 1;
				if ( end != -1 ) {
					checkForEnd = true;
				}
			}
		}

		//we ran out of keys to try
	}

	/**
	 * Get a batch of uninitialized collection keys for a given role
	 *
	 * @param collectionPersister The persister for the collection role.
	 * @param id A key that must be included in the batch fetch
	 * @param batchSize the maximum number of keys to return
	 * @return an array of collection keys, of length batchSize (padded with nulls)
	 */
	public Object [] getCollectionBatch(
			final CollectionPersister collectionPersister,
			final Object id,
			final int batchSize) {

		final Object[] keys = new Object[batchSize];
		keys[0] = id;

		if ( batchLoadableCollections == null ) {
			return keys;
		}

		int i = 1;
		int end = -1;
		boolean checkForEnd = false;

		final LinkedHashMap<CollectionEntry, PersistentCollection<?>> map =
				batchLoadableCollections.get( collectionPersister.getRole() );
		if ( map != null ) {
			for ( Entry<CollectionEntry, PersistentCollection<?>> me : map.entrySet() ) {
				final CollectionEntry ce = me.getKey();
				final Object loadedKey = ce.getLoadedKey();
				final PersistentCollection<?> collection = me.getValue();

				if ( loadedKey == null ) {
					// the loadedKey of the collectionEntry might be null as it might have been reset to null
					// (see for example Collections.processDereferencedCollection()
					// and CollectionEntry.afterAction())
					// though we clear the queue on flush, it seems like a good idea to guard
					// against potentially null loadedKeys (which leads to various NPEs as demonstrated in HHH-7821).
					continue;
				}

				if ( collection.wasInitialized() ) {
					// should never happen
					LOG.warn( "Encountered initialized collection in BatchFetchQueue, this should not happen." );
					continue;
				}

				if ( checkForEnd && i == end ) {
					return keys; //the first key found after the given key
				}

				final boolean isEqual = collectionPersister.getKeyType().isEqual(
						id,
						loadedKey,
						collectionPersister.getFactory()
				);

				if ( isEqual ) {
					end = i;
					//checkForEnd = false;
				}
				else if ( !isCached( loadedKey, collectionPersister ) ) {
					keys[i++] = loadedKey;
					//count++;
				}

				if ( i == batchSize ) {
					i = 1; //end of array, start filling again from start
					if ( end != -1 ) {
						checkForEnd = true;
					}
				}
			}
		}
		return keys; //we ran out of keys to try
	}

	private boolean isCached(Object collectionKey, CollectionPersister persister) {
		SharedSessionContractImplementor session = context.getSession();
		if ( session.getCacheMode().isGetEnabled() && persister.hasCache() ) {
			CollectionDataAccess cache = persister.getCacheAccessStrategy();
			Object cacheKey = cache.generateCacheKey(
					collectionKey,
					persister,
					session.getFactory(),
					session.getTenantIdentifier()
			);
			return CacheHelper.fromSharedCache( session, cacheKey, persister, cache ) != null;
		}
		return false;
	}

	public SharedSessionContractImplementor getSession() {
		return context.getSession();
	}
}
