/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later.
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.annotations;

import org.hibernate.jdbc.Expectation;

import java.lang.annotation.Repeatable;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.ElementType.TYPE;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * Specifies a custom SQL DML statement to be used in place of the default SQL
 * generated by Hibernate when an entity or collection row is deleted from the
 * database.
 * <p>
 * The given {@linkplain #sql SQL statement} must have exactly the number of JDBC
 * {@code ?} parameters that Hibernate expects, in the exact order Hibernate
 * expects. The primary key columns come before the version column if the
 * entity is versioned.
 * <p>
 * If an entity has {@linkplain jakarta.persistence.SecondaryTable secondary
 * tables}, it may have a {@code @SQLDelete} annotation for each secondary table.
 * The {@link #table} member must specify the name of the secondary table.
 *
 * @author Laszlo Benke
 */
@Target({TYPE, FIELD, METHOD})
@Retention(RUNTIME)
@Repeatable(SQLDeletes.class)
public @interface SQLDelete {
	/**
	 * Procedure name or SQL {@code DELETE} statement.
	 */
	String sql();

	/**
	 * Is the statement callable (aka a {@link java.sql.CallableStatement})?
	 */
	boolean callable() default false;

	/**
	 * An {@link Expectation} class used to verify that the operation was successful.
	 *
	 * @see Expectation.None
	 * @see Expectation.RowCount
	 * @see Expectation.OutParameter
	 */
	Class<? extends Expectation> verify() default Expectation.class;

	/**
	 * A {@link ResultCheckStyle} used to verify that the operation was successful.
	 *
	 * @deprecated use {@link #verify()} with an {@link Expectation} class
	 */
	@Deprecated(since = "6.5")
	ResultCheckStyle check() default ResultCheckStyle.NONE;

	/**
	 * The name of the table affected by the delete statement. Required when the
	 * statement affects a {@linkplain jakarta.persistence.SecondaryTable secondary
	 * table} of an entity. Not required for collections nor when the insert statement
	 * affects the primary table of an entity.
	 *
	 * @return the name of the secondary table
	 *
	 * @since 6.2
	 */
	String table() default "";
}
