/*
 * Decompiled with CFR 0.152.
 */
package org.hornetq.core.journal.impl;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.hornetq.core.journal.SequentialFile;
import org.hornetq.core.journal.SequentialFileFactory;
import org.hornetq.core.journal.impl.JournalFile;
import org.hornetq.core.journal.impl.JournalFileImpl;
import org.hornetq.core.journal.impl.JournalImpl;
import org.hornetq.core.logging.Logger;

public class JournalFilesRepository {
    private static final Logger log = Logger.getLogger(JournalFilesRepository.class);
    private static final boolean trace = log.isTraceEnabled();
    private final SequentialFileFactory fileFactory;
    private final BlockingDeque<JournalFile> dataFiles = new LinkedBlockingDeque<JournalFile>();
    private final BlockingQueue<JournalFile> pendingCloseFiles = new LinkedBlockingDeque<JournalFile>();
    private final ConcurrentLinkedQueue<JournalFile> freeFiles = new ConcurrentLinkedQueue();
    private final BlockingQueue<JournalFile> openedFiles = new LinkedBlockingQueue<JournalFile>();
    private final AtomicLong nextFileID = new AtomicLong(0L);
    private final int maxAIO;
    private final int minFiles;
    private final int fileSize;
    private final String filePrefix;
    private final String fileExtension;
    private final int userVersion;
    private Executor filesExecutor;

    private static final void trace(String message) {
        log.trace(message);
    }

    public JournalFilesRepository(SequentialFileFactory fileFactory, String filePrefix, String fileExtension, int userVersion, int maxAIO, int fileSize, int minFiles) {
        this.fileFactory = fileFactory;
        this.maxAIO = maxAIO;
        this.filePrefix = filePrefix;
        this.fileExtension = fileExtension;
        this.minFiles = minFiles;
        this.fileSize = fileSize;
        this.userVersion = userVersion;
    }

    public void setExecutor(Executor executor) {
        this.filesExecutor = executor;
    }

    public void clear() {
        this.dataFiles.clear();
        this.drainClosedFiles();
        this.freeFiles.clear();
        for (JournalFile file : this.openedFiles) {
            try {
                file.getFile().close();
            }
            catch (Exception e) {
                log.warn(e.getMessage(), e);
            }
        }
        this.openedFiles.clear();
    }

    public int getMaxAIO() {
        return this.maxAIO;
    }

    public String getFileExtension() {
        return this.fileExtension;
    }

    public String getFilePrefix() {
        return this.filePrefix;
    }

    public void calculateNextfileID(List<JournalFile> files) {
        for (JournalFile file : files) {
            long fileID = file.getFileID();
            if (this.nextFileID.get() < fileID) {
                this.nextFileID.set(fileID);
            }
            long fileNameID = this.getFileNameID(file.getFile().getFileName());
            if (this.nextFileID.get() >= fileNameID) continue;
            this.nextFileID.set(fileNameID);
        }
    }

    public void ensureMinFiles() throws Exception {
        int filesToCreate = this.minFiles - (this.dataFiles.size() + this.freeFiles.size());
        if (filesToCreate > 0) {
            for (int i = 0; i < filesToCreate; ++i) {
                this.freeFiles.add(this.createFile(false, false, true, false));
            }
        }
    }

    public void openFile(JournalFile file, boolean multiAIO) throws Exception {
        if (multiAIO) {
            file.getFile().open();
        } else {
            file.getFile().open(1, false);
        }
        file.getFile().position(file.getFile().calculateBlockStart(16));
    }

    public JournalFile[] getDataFilesArray() {
        return this.dataFiles.toArray(new JournalFile[this.dataFiles.size()]);
    }

    public JournalFile pollLastDataFile() {
        return (JournalFile)this.dataFiles.pollLast();
    }

    public void removeDataFile(JournalFile file) {
        if (!this.dataFiles.remove(file)) {
            log.warn("Could not remove file " + file + " from the list of data files");
        }
    }

    public int getDataFilesCount() {
        return this.dataFiles.size();
    }

    public Collection<JournalFile> getDataFiles() {
        return this.dataFiles;
    }

    public void clearDataFiles() {
        this.dataFiles.clear();
    }

    public void addDataFileOnTop(JournalFile file) {
        this.dataFiles.addFirst(file);
    }

    public void addDataFileOnBottom(JournalFile file) {
        this.dataFiles.add(file);
    }

    public int getFreeFilesCount() {
        return this.freeFiles.size();
    }

    public void addFreeFileNoInit(JournalFile file) {
        this.freeFiles.add(file);
    }

    public synchronized void addFreeFile(JournalFile file, boolean renameTmp) throws Exception {
        if (file.getFile().size() != (long)this.fileSize) {
            log.warn("Deleting " + file + ".. as it doesn't have the configured size");
            file.getFile().delete();
        } else if (this.freeFiles.size() + this.dataFiles.size() + 1 + this.openedFiles.size() < this.minFiles) {
            if (trace) {
                JournalFilesRepository.trace("Adding free file " + file);
            }
            JournalFile jf = this.reinitializeFile(file);
            if (renameTmp) {
                jf.getFile().renameTo(JournalImpl.renameExtensionFile(jf.getFile().getFileName(), ".tmp"));
            }
            this.freeFiles.add(jf);
        } else {
            file.getFile().delete();
        }
    }

    public Collection<JournalFile> getFreeFiles() {
        return this.freeFiles;
    }

    public JournalFile getFreeFile() {
        return (JournalFile)this.freeFiles.remove();
    }

    public int getOpenedFilesCount() {
        return this.openedFiles.size();
    }

    public void drainClosedFiles() {
        try {
            JournalFile file;
            while ((file = (JournalFile)this.pendingCloseFiles.poll()) != null) {
                file.getFile().close();
            }
        }
        catch (Exception e) {
            log.warn(e.getMessage(), e);
        }
    }

    public JournalFile openFile() throws InterruptedException {
        if (trace) {
            JournalFilesRepository.trace("enqueueOpenFile with openedFiles.size=" + this.openedFiles.size());
        }
        Runnable run = new Runnable(){

            @Override
            public void run() {
                try {
                    JournalFilesRepository.this.pushOpenedFile();
                }
                catch (Exception e) {
                    log.error(e.getMessage(), e);
                }
            }
        };
        if (this.filesExecutor == null) {
            run.run();
        } else {
            this.filesExecutor.execute(run);
        }
        JournalFile nextFile = null;
        while (nextFile == null) {
            nextFile = this.openedFiles.poll(5L, TimeUnit.SECONDS);
            if (nextFile != null) continue;
            log.warn("Couldn't open a file in 60 Seconds", new Exception("Warning: Couldn't open a file in 60 Seconds"));
        }
        if (trace) {
            JournalFilesRepository.trace("Returning file " + nextFile);
        }
        return nextFile;
    }

    public void pushOpenedFile() throws Exception {
        JournalFile nextOpenedFile = this.takeFile(true, true, true, false);
        if (trace) {
            JournalFilesRepository.trace("pushing openFile " + nextOpenedFile);
        }
        this.openedFiles.offer(nextOpenedFile);
    }

    public void closeFile(JournalFile file) {
        this.fileFactory.deactivateBuffer();
        this.pendingCloseFiles.add(file);
        this.dataFiles.add(file);
        Runnable run = new Runnable(){

            @Override
            public void run() {
                JournalFilesRepository.this.drainClosedFiles();
            }
        };
        if (this.filesExecutor == null) {
            run.run();
        } else {
            this.filesExecutor.execute(run);
        }
    }

    public JournalFile takeFile(boolean keepOpened, boolean multiAIO, boolean initFile, boolean tmpCompactExtension) throws Exception {
        JournalFile nextFile = null;
        nextFile = this.freeFiles.poll();
        if (nextFile == null) {
            nextFile = this.createFile(keepOpened, multiAIO, initFile, tmpCompactExtension);
        } else {
            if (tmpCompactExtension) {
                SequentialFile sequentialFile = nextFile.getFile();
                sequentialFile.renameTo(sequentialFile.getFileName() + ".cmp");
            }
            if (keepOpened) {
                this.openFile(nextFile, multiAIO);
            }
        }
        return nextFile;
    }

    private JournalFile createFile(boolean keepOpened, boolean multiAIO, boolean init, boolean tmpCompact) throws Exception {
        long fileID = this.generateFileID();
        String fileName = this.createFileName(tmpCompact, fileID);
        if (trace) {
            JournalFilesRepository.trace("Creating file " + fileName);
        }
        String tmpFileName = fileName + ".tmp";
        SequentialFile sequentialFile = this.fileFactory.createSequentialFile(tmpFileName, this.maxAIO);
        sequentialFile.open(1, false);
        if (init) {
            sequentialFile.fill(0, this.fileSize, (byte)74);
            JournalImpl.initFileHeader(this.fileFactory, sequentialFile, this.userVersion, fileID);
        }
        long position = sequentialFile.position();
        sequentialFile.close();
        if (trace) {
            JournalFilesRepository.trace("Renaming file " + tmpFileName + " as " + fileName);
        }
        sequentialFile.renameTo(fileName);
        if (keepOpened) {
            if (multiAIO) {
                sequentialFile.open();
            } else {
                sequentialFile.open(1, false);
            }
            sequentialFile.position(position);
        }
        return new JournalFileImpl(sequentialFile, fileID, 2);
    }

    private String createFileName(boolean tmpCompact, long fileID) {
        String fileName = tmpCompact ? this.filePrefix + "-" + fileID + "." + this.fileExtension + ".cmp" : this.filePrefix + "-" + fileID + "." + this.fileExtension;
        return fileName;
    }

    private long generateFileID() {
        return this.nextFileID.incrementAndGet();
    }

    private long getFileNameID(String fileName) {
        try {
            return Long.parseLong(fileName.substring(this.filePrefix.length() + 1, fileName.indexOf(46)));
        }
        catch (Throwable e) {
            log.warn("Impossible to get the ID part of the file name " + fileName, e);
            return 0L;
        }
    }

    private JournalFile reinitializeFile(JournalFile file) throws Exception {
        long newFileID = this.generateFileID();
        SequentialFile sf = file.getFile();
        sf.open(1, false);
        int position = JournalImpl.initFileHeader(this.fileFactory, sf, this.userVersion, newFileID);
        JournalFileImpl jf = new JournalFileImpl(sf, newFileID, 2);
        sf.position(position);
        sf.close();
        return jf;
    }
}

