/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist;

import java.lang.annotation.Annotation;
import java.lang.reflect.Modifier;

import javassist.CannotCompileException;
import javassist.CtBehavior;
import javassist.CtClass;
import javassist.NotFoundException;
import javassist.bytecode.SignatureAttribute.MethodSignature;

import org.jboss.reflect.plugins.AnnotationHelper;
import org.jboss.reflect.spi.AnnotationInfo;
import org.jboss.reflect.spi.AnnotationValue;
import org.jboss.reflect.spi.Body;
import org.jboss.reflect.spi.ClassInfo;
import org.jboss.reflect.spi.ModifierInfo;
import org.jboss.reflect.spi.ParameterInfo;
import org.jboss.reflect.spi.TypeInfo;

/**
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 103926 $
 */
public abstract class JavassistAnnotatedParameterInfo extends JavassistAnnotatedInfo
{
   /** The serialVersionUID */
   private static final long serialVersionUID = -494071110672611729L;

   /** The parameters */
   protected transient volatile ParameterInfo[] parameters;
   
   /** The parameter types */
   protected transient volatile TypeInfo[] parameterTypes;
   
   /** The exception types */
   protected transient volatile ClassInfo[] exceptionTypes;
   
   /** The type info */
   protected final JavassistTypeInfo typeInfo;
   
   /** The constructor or method */
   protected final CtBehavior ctBehavior;


   public JavassistAnnotatedParameterInfo(AnnotationHelper annotationHelper, JavassistTypeInfo typeInfo, CtBehavior ctBehavior)
   {
      super(annotationHelper);
      this.typeInfo = typeInfo;
      this.ctBehavior = ctBehavior;
   }

   public int getModifiers()
   {
      return ctBehavior.getModifiers();
   }

   public boolean isPublic()
   {
      return Modifier.isPublic(ctBehavior.getModifiers());
   }

   public boolean isStatic()
   {
      return Modifier.isStatic(ctBehavior.getModifiers());
   }

   public boolean isVolatile()
   {
      return Modifier.isVolatile(ctBehavior.getModifiers());
   }

   public ClassInfo getDeclaringClass()
   {
      return typeInfo;
   }

   public ClassInfo[] getExceptionTypes()
   {
      if (exceptionTypes == null)
      {
         try
         {
            CtClass[] types = ctBehavior.getExceptionTypes();
            ClassInfo[] exceptionTypes = new ClassInfo[types.length];
            for (int i = 0; i < types.length; ++i)
               exceptionTypes[i] = (ClassInfo) typeInfo.getFactory().getTypeInfo(types[i]);
            
            this.exceptionTypes = exceptionTypes;
         }
         catch (NotFoundException e)
         {
            throw JavassistTypeInfoFactoryImpl.raiseClassNotFound("for exception types of constructor", e);
         }
      }
      return exceptionTypes;
   }

   protected synchronized void generateParameters()
   {
      if (parameters != null)
         return;
      
      try
      {
         MethodSignature sig = JavassistHelper.getMethodSignature(ctBehavior);
         if (sig != null && sig.getParameterTypes().length == ctBehavior.getParameterTypes().length)
         {
            parameterTypes = JavassistHelper.createParameterTypes(ctBehavior, sig, typeInfo);
         }
         else
         {
            CtClass[] types = ctBehavior.getParameterTypes();
            parameterTypes = new TypeInfo[types.length];
            for (int i = 0; i < types.length; ++i)
               parameterTypes[i] = typeInfo.getFactory().getTypeInfo(types[i]);
         }
         parameters = new ParameterInfo[parameterTypes.length];
         for (int i = 0; i < parameterTypes.length; ++i)
            parameters[i] = new JavassistParameterInfo(annotationHelper, this, i, parameterTypes[i]);
      }
      catch (NotFoundException e)
      {
         throw JavassistTypeInfoFactoryImpl.raiseClassNotFound("for parameters of constructor", e);
      }
   }

   public ParameterInfo[] getParameters()
   {
      if (parameters == null)
         generateParameters();
      return parameters;
   }

   public TypeInfo[] getParameterTypes()
   {
      if (parameterTypes == null)
         generateParameters();
      return parameterTypes;
   }

   public AnnotationValue[] getAnnotations()
   {
      return getAnnotations(ctBehavior);
   }
      
   protected void setupParameterAnnotations(Object[][] annotations)
   {
      for (int param = 0 ; param < annotations.length ; param++)
      {
         AnnotationValue[] annotationValues = new AnnotationValue[annotations[param].length];
         for (int ann = 0 ; ann < annotationValues.length ; ann++)
         {
            Class<?> clazz = ((Annotation)annotations[param][ann]).annotationType();

            AnnotationInfo info = (AnnotationInfo)((JavassistTypeInfoFactoryImpl)annotationHelper).getTypeInfo(clazz);
            annotationValues[ann] = annotationHelper.createAnnotationValue(info, annotations[param][ann]);
         }
         ((JavassistParameterInfo)parameters[param]).setAnnotations(annotationValues);
      }
   }
   
   protected void createParameterAnnotations()
   {
      try
      {
         Object[][] parameterAnnotations = ctBehavior.getParameterAnnotations();
         setupParameterAnnotations(parameterAnnotations);
      }
      catch (ClassNotFoundException e)
      {
         // AutoGenerated
         throw new RuntimeException(e);
      }
   }

   public void setBody(Body body)
   {
      typeInfo.clearMethodCache();
      if (body instanceof AbstractJavassistBody == false)
      {
         throw new IllegalArgumentException("Body is not an instance of AbstractJavassistBody");
      }
      ((AbstractJavassistBody)body).createBody(ctBehavior);
   }

   public void setExceptions(String[] exceptions)
   {
      typeInfo.clearMethodCache();
      try
      {
         ctBehavior.setExceptionTypes(JavassistUtil.toCtClass(typeInfo.getCtClass().getClassPool(), exceptions));
      }
      catch (NotFoundException e)
      {
         throw new org.jboss.reflect.spi.NotFoundException(e.toString());
      }
   }

   public void setExceptions(ClassInfo[] exceptions)
   {
      try
      {
         ctBehavior.setExceptionTypes(JavassistUtil.toCtClass(exceptions));
      }
      catch (NotFoundException e)
      {
         throw new org.jboss.reflect.spi.NotFoundException(e.toString());
      }
      typeInfo.clearMethodCache();
   }

   public void setModifier(ModifierInfo mi)
   {
      ctBehavior.setModifiers(mi.getModifiers());
     typeInfo.clearMethodCache();
   }

   public void setParameters(String[] parameters)
   {
      for(String p : parameters)
      {
         try
         {
            ctBehavior.addParameter(JavassistUtil.toCtClass(typeInfo.getCtClass().getClassPool(), p));
         }
         catch (CannotCompileException e)
         {
            throw new org.jboss.reflect.spi.CannotCompileException(e.toString());
         }
      }
      typeInfo.clearMethodCache();
   }

   public void setParameters(ClassInfo[] parameters)
   {
      for(ClassInfo clazz : parameters)
      {
         try
         {
            ctBehavior.addParameter(JavassistUtil.toCtClass(clazz));
         }
         catch (CannotCompileException e)
         {
            throw new org.jboss.reflect.spi.CannotCompileException(e.toString());
         }
      }
      typeInfo.clearMethodCache();
   }

}
