/*
  $Id: Response.java 3266 2018-09-12 18:42:07Z daniel_fisher $

  Copyright (C) 2003-2014 Virginia Tech.
  All rights reserved.

  SEE LICENSE FOR MORE INFORMATION

  Author:  Middleware Services
  Email:   middleware@vt.edu
  Version: $Revision: 3266 $
  Updated: $Date: 2018-09-12 14:42:07 -0400 (Wed, 12 Sep 2018) $
*/
package org.ldaptive;

import java.util.Arrays;
import org.ldaptive.control.ResponseControl;

/**
 * Wrapper class for all operation responses.
 *
 * @param  <T>  type of ldap result contained in this response
 *
 * @author  Middleware Services
 * @version  $Revision: 3266 $ $Date: 2018-09-12 14:42:07 -0400 (Wed, 12 Sep 2018) $
 */
public class Response<T> implements ResponseMessage
{

  /**
   * Property to configure the encoding of control characters in the response
   * message.
   */
  public static final String ENCODE_CNTRL_CHARS =
    "org.ldaptive.response.encodeCntrlChars";

  /** Whether to encode control characters. */
  protected static boolean encodeCntrlChars;

  /**
   * statically initialize encoding of control characters.
   */
  static {
    final String ecc = System.getProperty(ENCODE_CNTRL_CHARS);
    if (ecc != null) {
      encodeCntrlChars = Boolean.valueOf(ecc);
    }
  }

  /** Operation response. */
  private final T result;

  /** Operation result code. */
  private final ResultCode resultCode;

  /** Response message. */
  private final String message;

  /** Response matched DN. */
  private final String matchedDn;

  /** Response controls. */
  private final ResponseControl[] responseControls;

  /** Referral URLs. */
  private final String[] referralURLs;

  /** Message ID. */
  private final int messageId;


  /**
   * Creates a new ldap response.
   *
   * @param  t  response type
   * @param  rc  result code
   */
  public Response(final T t, final ResultCode rc)
  {
    result = t;
    resultCode = rc;
    message = null;
    matchedDn = null;
    responseControls = null;
    referralURLs = null;
    messageId = -1;
  }


  /**
   * Creates a new ldap response.
   *
   * @param  t  response type
   * @param  rc  result code
   * @param  msg  result message
   * @param  dn  matched dn
   * @param  c  response controls
   * @param  urls  referral urls
   * @param  id  message id
   */
  public Response(
    final T t,
    final ResultCode rc,
    final String msg,
    final String dn,
    final ResponseControl[] c,
    final String[] urls,
    final int id)
  {
    result = t;
    resultCode = rc;
    message = msg;
    matchedDn = dn;
    responseControls = c;
    referralURLs = urls;
    messageId = id;
  }


  /**
   * Returns the result of the ldap operation.
   *
   * @return  operation result
   */
  public T getResult()
  {
    return result;
  }


  /**
   * Returns the result code of the ldap operation.
   *
   * @return  operation result code
   */
  public ResultCode getResultCode()
  {
    return resultCode;
  }


  /**
   * Returns any error or diagnostic message produced by the ldap operation.
   *
   * @return  message
   */
  public String getMessage()
  {
    return message;
  }


  /**
   * Returns the matched DN produced by the ldap operation.
   *
   * @return  matched DN
   */
  public String getMatchedDn()
  {
    return matchedDn;
  }


  /** {@inheritDoc} */
  @Override
  public ResponseControl[] getControls()
  {
    return responseControls;
  }


  /** {@inheritDoc} */
  @Override
  public ResponseControl getControl(final String oid)
  {
    if (getControls() != null) {
      for (ResponseControl c : getControls()) {
        if (c.getOID().equals(oid)) {
          return c;
        }
      }
    }
    return null;
  }


  /**
   * Returns the referral URLs produced by the ldap operation.
   *
   * @return  referral urls
   */
  public String[] getReferralURLs()
  {
    return referralURLs;
  }


  /** {@inheritDoc} */
  @Override
  public int getMessageId()
  {
    return messageId;
  }


  /** {@inheritDoc} */
  @Override
  public String toString()
  {
    return
      String.format(
        "[%s@%d::result=%s, resultCode=%s, message=%s, matchedDn=%s, " +
        "responseControls=%s, referralURLs=%s, messageId=%s]",
        getClass().getName(),
        hashCode(),
        result,
        resultCode,
        encodeCntrlChars ?
          LdapUtils.percentEncodeControlChars(message) :
          message,
        matchedDn,
        Arrays.toString(responseControls),
        Arrays.toString(referralURLs),
        messageId);
  }
}
