/*
  $Id: AbstractAuthenticationHandler.java 3235 2017-08-28 19:48:23Z daniel_fisher $

  Copyright (C) 2003-2014 Virginia Tech.
  All rights reserved.

  SEE LICENSE FOR MORE INFORMATION

  Author:  Middleware Services
  Email:   middleware@vt.edu
  Version: $Revision: 3235 $
  Updated: $Date: 2017-08-28 15:48:23 -0400 (Mon, 28 Aug 2017) $
*/
package org.ldaptive.auth;

import org.ldaptive.Connection;
import org.ldaptive.LdapException;
import org.ldaptive.LdapUtils;
import org.ldaptive.control.RequestControl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Base class for an LDAP authentication implementations.
 *
 * @author  Middleware Services
 * @version  $Revision: 3235 $ $Date: 2017-08-28 15:48:23 -0400 (Mon, 28 Aug 2017) $
 */
public abstract class AbstractAuthenticationHandler
  implements AuthenticationHandler
{

  /** Logger for this class. */
  protected final Logger logger = LoggerFactory.getLogger(getClass());

  /** controls used by this handler. */
  private RequestControl[] authenticationControls;


  /**
   * Returns the controls for this authentication handler.
   *
   * @return  controls
   */
  public RequestControl[] getAuthenticationControls()
  {
    return authenticationControls;
  }


  /**
   * Sets the controls for this authentication handler.
   *
   * @param  c  controls to set
   */
  public void setAuthenticationControls(final RequestControl... c)
  {
    authenticationControls = c;
  }


  /** {@inheritDoc} */
  @Override
  public AuthenticationHandlerResponse authenticate(
    final AuthenticationCriteria ac)
    throws LdapException
  {
    logger.debug("authenticate criteria={}", ac);

    AuthenticationHandlerResponse response = null;
    final Connection conn = getConnection();
    boolean closeConn = false;
    try {
      response = authenticateInternal(conn, ac);
    } catch (LdapException e) {
      closeConn = true;
      throw e;
    } catch (RuntimeException e) {
      closeConn = true;
      throw e;
    } finally {
      if (closeConn) {
        conn.close();
      }
    }
    logger.debug("authenticate response={} for criteria={}", response, ac);
    return response;
  }


  /**
   * Returns a connection that the authentication operation should be performed
   * on.
   *
   * @return  connection
   *
   * @throws  LdapException  if an error occurs provisioning the connection
   */
  protected abstract Connection getConnection()
    throws LdapException;


  /**
   * Authenticate on the supplied connection using the supplied criteria.
   *
   * @param  c  to authenticate on
   * @param  criteria  criteria to authenticate with
   *
   * @return  authentication handler response
   *
   * @throws  LdapException  if the authentication fails
   */
  protected abstract AuthenticationHandlerResponse authenticateInternal(
    final Connection c,
    final AuthenticationCriteria criteria)
    throws LdapException;


  /**
   * Combines request controls in the {@link AuthenticationRequest} with {@link
   * #authenticationControls}.
   *
   * @param  criteria  containing request controls
   *
   * @return  combined request controls or null
   */
  protected RequestControl[] processRequestControls(
    final AuthenticationCriteria criteria)
  {
    RequestControl[] ctls;
    if (criteria.getAuthenticationRequest().getControls() != null) {
      if (getAuthenticationControls() != null) {
        ctls = LdapUtils.concatArrays(
          criteria.getAuthenticationRequest().getControls(),
          getAuthenticationControls());
      } else {
        ctls = criteria.getAuthenticationRequest().getControls();
      }
    } else {
      ctls = getAuthenticationControls();
    }
    return ctls;
  }
}
