/*
  $Id: AuthenticationHandlerResponse.java 3266 2018-09-12 18:42:07Z daniel_fisher $

  Copyright (C) 2003-2014 Virginia Tech.
  All rights reserved.

  SEE LICENSE FOR MORE INFORMATION

  Author:  Middleware Services
  Email:   middleware@vt.edu
  Version: $Revision: 3266 $
  Updated: $Date: 2018-09-12 14:42:07 -0400 (Wed, 12 Sep 2018) $
*/
package org.ldaptive.auth;

import java.util.Arrays;
import org.ldaptive.Connection;
import org.ldaptive.LdapUtils;
import org.ldaptive.Response;
import org.ldaptive.ResultCode;
import org.ldaptive.control.ResponseControl;

/**
 * Response object for authentication handlers.
 *
 * @author  Middleware Services
 * @version  $Revision: 3266 $ $Date: 2018-09-12 14:42:07 -0400 (Wed, 12 Sep 2018) $
 */
public class AuthenticationHandlerResponse extends Response<Boolean>
{

  /** Connection that authentication occurred on. */
  private final Connection connection;


  /**
   * Creates a new authentication response.
   *
   * @param  success  authentication result
   * @param  rc  result code from the underlying ldap operation
   * @param  conn  connection the authentication occurred on
   */
  public AuthenticationHandlerResponse(
    final boolean success,
    final ResultCode rc,
    final Connection conn)
  {
    super(success, rc);
    connection = conn;
  }


  /**
   * Creates a new authentication response.
   *
   * @param  success  authentication result
   * @param  rc  result code from the underlying ldap operation
   * @param  conn  connection the authentication occurred on
   * @param  msg  authentication message
   */
  public AuthenticationHandlerResponse(
    final boolean success,
    final ResultCode rc,
    final Connection conn,
    final String msg)
  {
    super(success, rc, msg, null, null, null, -1);
    connection = conn;
  }


  /**
   * Creates a new ldap response.
   *
   * @param  success  authentication result
   * @param  rc  result code from the underlying ldap operation
   * @param  conn  connection the authentication occurred on
   * @param  msg  authentication message
   * @param  controls  response controls from the underlying ldap operation
   * @param  msgId  message id from the underlying ldap operation
   */
  public AuthenticationHandlerResponse(
    final boolean success,
    final ResultCode rc,
    final Connection conn,
    final String msg,
    final ResponseControl[] controls,
    final int msgId)
  {
    super(success, rc, msg, null, controls, null, msgId);
    connection = conn;
  }


  /**
   * Returns the connection that the ldap operation occurred on.
   *
   * @return  connection
   */
  public Connection getConnection()
  {
    return connection;
  }


  /** {@inheritDoc} */
  @Override
  public String toString()
  {
    return
      String.format(
        "[%s@%d::connection=%s, result=%s, resultCode=%s, message=%s, " +
        "controls=%s]",
        getClass().getName(),
        hashCode(),
        connection,
        getResult(),
        getResultCode(),
        encodeCntrlChars ?
          LdapUtils.percentEncodeControlChars(getMessage()) :
          getMessage(),
        Arrays.toString(getControls()));
  }
}
