/*
  $Id: ActiveDirectoryAuthenticationResponseHandler.java 3189 2016-11-01 21:21:43Z daniel_fisher $

  Copyright (C) 2003-2014 Virginia Tech.
  All rights reserved.

  SEE LICENSE FOR MORE INFORMATION

  Author:  Middleware Services
  Email:   middleware@vt.edu
  Version: $Revision: 3189 $
  Updated: $Date: 2016-11-01 17:21:43 -0400 (Tue, 01 Nov 2016) $
*/
package org.ldaptive.auth.ext;

import java.util.Calendar;
import java.util.concurrent.TimeUnit;
import org.ldaptive.LdapAttribute;
import org.ldaptive.LdapEntry;
import org.ldaptive.LdapUtils;
import org.ldaptive.ad.io.FileTimeValueTranscoder;
import org.ldaptive.auth.AuthenticationResponse;
import org.ldaptive.auth.AuthenticationResponseHandler;

/**
 * Attempts to parse the authentication response message and set the account
 * state using data associated with active directory.
 *
 * @author  Middleware Services
 * @version  $Revision: 3189 $ $Date: 2016-11-01 17:21:43 -0400 (Tue, 01 Nov 2016) $
 */
public class ActiveDirectoryAuthenticationResponseHandler
  implements AuthenticationResponseHandler
{

  /** hash code seed. */
  private static final int HASH_CODE_SEED = 1201;

  /** Amount of time in milliseconds since a password was set until it will
   * expire. Used if msDS-UserPasswordExpiryTimeComputed cannot be read. */
  private long maxPasswordAge = -1;

  /** Number of hours before expiration to produce a warning. */
  private int warningHours;


  /** Default constructor. */
  public ActiveDirectoryAuthenticationResponseHandler() {}


  /**
   * Creates a new active directory authentication response handler.
   *
   * @param  passwordAge  length of time in milliseconds that a password is
   * valid
   */
  public ActiveDirectoryAuthenticationResponseHandler(final long passwordAge)
  {
    if (passwordAge < 0) {
      throw new IllegalArgumentException("Password age must be >= 0");
    }
    maxPasswordAge = passwordAge;
  }


  /**
   * Creates a new active directory authentication response handler.
   *
   * @param  warning  length of time before expiration that should produce a
   * warning
   */
  public ActiveDirectoryAuthenticationResponseHandler(final int warning)
  {
    if (warning < 0) {
      throw new IllegalArgumentException("Warning hours must be >= 0");
    }
    warningHours = warning;
  }


  /**
   * Creates a new active directory authentication response handler.
   *
   * @param  warning  length of time before expiration that should produce a
   * warning
   * @param  passwordAge  length of time in milliseconds that a password is
   * valid
   */
  public ActiveDirectoryAuthenticationResponseHandler(
    final int warning,
    final long passwordAge)
  {
    if (warning < 0) {
      throw new IllegalArgumentException("Warning hours must be >= 0");
    }
    warningHours = warning;
    if (passwordAge < 0) {
      throw new IllegalArgumentException("Password age must be >= 0");
    }
    maxPasswordAge = passwordAge;
  }


  /**
   * Creates a new active directory authentication response handler.
   *
   * @param  warning  in duration syntax of time before expiration that should
   * produce a warning
   */
  public ActiveDirectoryAuthenticationResponseHandler(final String warning)
  {
    this((int) LdapUtils.durationDecode(warning, TimeUnit.HOURS));
  }


  /**
   * Creates a new active directory authentication response handler.
   *
   * @param  warning  in duration syntax of time before expiration that should
   * produce a warning
   * @param  passwordAge  in duration syntax of time in milliseconds that a
   * password is valid
   */
  public ActiveDirectoryAuthenticationResponseHandler(
    final String warning,
    final String passwordAge)
  {
    this(
      (int) LdapUtils.durationDecode(warning, TimeUnit.HOURS),
      LdapUtils.durationDecode(passwordAge, TimeUnit.MILLISECONDS));
  }


  @Override
  public void handle(final AuthenticationResponse response)
  {
    if (response.getResult()) {
      final LdapEntry entry = response.getLdapEntry();
      final LdapAttribute expTime = entry.getAttribute(
        "msDS-UserPasswordExpiryTimeComputed");
      final LdapAttribute pwdLastSet = entry.getAttribute("pwdLastSet");

      Calendar exp = null;
      // ignore expTime if account is set to never expire
      if (expTime != null &&
          !"9223372036854775807".equals(expTime.getStringValue())) {
        exp = expTime.getValue(new FileTimeValueTranscoder());
      } else if (maxPasswordAge >= 0 && pwdLastSet != null) {
        exp = pwdLastSet.getValue(new FileTimeValueTranscoder());
        exp.setTimeInMillis(exp.getTimeInMillis() + maxPasswordAge);
      }

      if (exp != null) {
        if (warningHours > 0) {
          final Calendar now = Calendar.getInstance();
          final Calendar warn = (Calendar) exp.clone();
          warn.add(Calendar.HOUR_OF_DAY, -warningHours);
          if (now.after(warn)) {
            response.setAccountState(new ActiveDirectoryAccountState(exp));
          }
        } else {
          response.setAccountState(new ActiveDirectoryAccountState(exp));
        }
      }
    } else {
      if (response.getMessage() != null) {
        final ActiveDirectoryAccountState.Error adError =
          ActiveDirectoryAccountState.Error.parse(response.getMessage());
        if (adError != null) {
          response.setAccountState(new ActiveDirectoryAccountState(adError));
        }
      }
    }
  }


  /** {@inheritDoc} */
  @Override
  public boolean equals(final Object o)
  {
    if (o == this) {
      return true;
    }
    if (o instanceof ActiveDirectoryAuthenticationResponseHandler) {
      final ActiveDirectoryAuthenticationResponseHandler v =
        (ActiveDirectoryAuthenticationResponseHandler) o;
      return LdapUtils.areEqual(maxPasswordAge, v.maxPasswordAge) &&
             LdapUtils.areEqual(warningHours, v.warningHours);
    }
    return false;
  }


  /** {@inheritDoc} */
  @Override
  public int hashCode()
  {
    return
      LdapUtils.computeHashCode(HASH_CODE_SEED, maxPasswordAge, warningHours);
  }


  @Override
  public String toString()
  {
    return String.format(
      "[%s@%d::maxPasswordAge=%s, warningHours=%s]",
      getClass().getName(),
      hashCode(),
      maxPasswordAge,
      warningHours);
  }
}
