/*
  $Id: EDirectoryAuthenticationResponseHandler.java 3189 2016-11-01 21:21:43Z daniel_fisher $

  Copyright (C) 2003-2014 Virginia Tech.
  All rights reserved.

  SEE LICENSE FOR MORE INFORMATION

  Author:  Middleware Services
  Email:   middleware@vt.edu
  Version: $Revision: 3189 $
  Updated: $Date: 2016-11-01 17:21:43 -0400 (Tue, 01 Nov 2016) $
*/
package org.ldaptive.auth.ext;

import java.util.Calendar;
import java.util.concurrent.TimeUnit;
import org.ldaptive.LdapAttribute;
import org.ldaptive.LdapEntry;
import org.ldaptive.LdapUtils;
import org.ldaptive.auth.AuthenticationResponse;
import org.ldaptive.auth.AuthenticationResponseHandler;
import org.ldaptive.io.GeneralizedTimeValueTranscoder;

/**
 * Attempts to parse the authentication response and set the account state using
 * data associated with eDirectory. The {@link org.ldaptive.auth.Authenticator}
 * should be configured to return 'passwordExpirationTime' and
 * 'loginGraceRemaining' attributes so they can be consumed by this handler.
 *
 * @author  Middleware Services
 * @version  $Revision: 3189 $ $Date: 2016-11-01 17:21:43 -0400 (Tue, 01 Nov 2016) $
 */
public class EDirectoryAuthenticationResponseHandler
  implements AuthenticationResponseHandler
{

  /** hash code seed. */
  private static final int HASH_CODE_SEED = 1213;

  /** Number of hours before expiration to produce a warning. */
  private int warningHours;


  /** Default constructor. */
  public EDirectoryAuthenticationResponseHandler() {}


  /**
   * Creates a new edirectory authentication response handler.
   *
   * @param  hours  length of time before expiration that should produce a
   *                warning
   */
  public EDirectoryAuthenticationResponseHandler(final int hours)
  {
    if (hours < 0) {
      throw new IllegalArgumentException("Hours must be >= 0");
    }
    warningHours = hours;
  }


  /**
   * Creates a new edirectory authentication response handler.
   *
   * @param  hours  in duration syntax of time before expiration that should
   * produce a warning
   */
  public EDirectoryAuthenticationResponseHandler(final String hours)
  {
    this((int) LdapUtils.durationDecode(hours, TimeUnit.HOURS));
  }


  /** {@inheritDoc} */
  @Override
  public void handle(final AuthenticationResponse response)
  {
    if (response.getMessage() != null) {
      final EDirectoryAccountState.Error edError =
        EDirectoryAccountState.Error.parse(response.getMessage());
      if (edError != null) {
        response.setAccountState(new EDirectoryAccountState(edError));
      }
    } else if (response.getResult()) {
      final LdapEntry entry = response.getLdapEntry();
      final LdapAttribute expTime = entry.getAttribute(
        "passwordExpirationTime");
      final LdapAttribute loginRemaining = entry.getAttribute(
        "loginGraceRemaining");
      final int loginRemainingValue = loginRemaining != null ?
        Integer.parseInt(loginRemaining.getStringValue()) : 0;
      if (expTime != null) {
        final Calendar exp = expTime.getValue(
          new GeneralizedTimeValueTranscoder());
        if (warningHours > 0) {
          final Calendar now = Calendar.getInstance();
          final Calendar warn = (Calendar) exp.clone();
          warn.add(Calendar.HOUR_OF_DAY, -warningHours);
          if (now.after(warn)) {
            response.setAccountState(
              new EDirectoryAccountState(exp, loginRemainingValue));
          }
        } else {
          response.setAccountState(
            new EDirectoryAccountState(exp, loginRemainingValue));
        }
      } else if (loginRemaining != null) {
        response.setAccountState(
          new EDirectoryAccountState(null, loginRemainingValue));
      }
    }
  }


  /** {@inheritDoc} */
  @Override
  public boolean equals(final Object o)
  {
    if (o == this) {
      return true;
    }
    if (o instanceof EDirectoryAuthenticationResponseHandler) {
      final EDirectoryAuthenticationResponseHandler v =
        (EDirectoryAuthenticationResponseHandler) o;
      return LdapUtils.areEqual(warningHours, v.warningHours);
    }
    return false;
  }


  /** {@inheritDoc} */
  @Override
  public int hashCode()
  {
    return
      LdapUtils.computeHashCode(HASH_CODE_SEED, warningHours);
  }


  @Override
  public String toString()
  {
    return String.format(
      "[%s@%d::warningHours=%s]",
      getClass().getName(),
      hashCode(),
      warningHours);
  }
}
