/*
  $Id: AbstractSSLContextInitializer.java 3243 2017-09-12 01:58:31Z daniel_fisher $

  Copyright (C) 2003-2014 Virginia Tech.
  All rights reserved.

  SEE LICENSE FOR MORE INFORMATION

  Author:  Middleware Services
  Email:   middleware@vt.edu
  Version: $Revision: 3243 $
  Updated: $Date: 2017-09-11 21:58:31 -0400 (Mon, 11 Sep 2017) $
*/
package org.ldaptive.ssl;

import java.security.GeneralSecurityException;
import java.util.Arrays;
import javax.net.ssl.KeyManager;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import org.ldaptive.LdapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Provides common implementation for SSL context initializer.
 *
 * @author  Middleware Services
 * @version  $Revision: 3243 $ $Date: 2017-09-11 21:58:31 -0400 (Mon, 11 Sep 2017) $
 */
public abstract class AbstractSSLContextInitializer
  implements SSLContextInitializer
{

  /** Logger for this class. */
  protected final Logger logger = LoggerFactory.getLogger(getClass());

  /** Trust managers. */
  protected TrustManager[] trustManagers;

  /** Hostname verifier config. */
  protected HostnameVerifierConfig hostnameVerifierConfig;


  /** {@inheritDoc} */
  @Override
  public TrustManager[] getTrustManagers()
    throws GeneralSecurityException
  {
    final TrustManager[] tm = createTrustManagers();
    final TrustManager[] hostnameTrustManager = hostnameVerifierConfig != null
      ? new TrustManager[] {
        new HostnameVerifyingTrustManager(
          hostnameVerifierConfig.getCertificateHostnameVerifier(),
          hostnameVerifierConfig.getHostnames()),
      } : null;
    TrustManager[] aggregate = null;
    if (tm == null) {
      if (trustManagers == null) {
        if (hostnameTrustManager != null) {
          aggregate = aggregateTrustManagers(
            new DefaultTrustManager(),
            hostnameTrustManager[0]);
        }
      } else {
        aggregate = aggregateTrustManagers(
          LdapUtils.concatArrays(trustManagers, hostnameTrustManager));
      }
    } else {
      aggregate = aggregateTrustManagers(
        LdapUtils.concatArrays(tm, trustManagers, hostnameTrustManager));
    }
    return aggregate;
  }


  /**
   * Sets the trust managers.
   *
   * @param  managers  trust managers
   */
  @Override
  public void setTrustManagers(final TrustManager... managers)
  {
    trustManagers = managers;
  }


  @Override
  public HostnameVerifierConfig getHostnameVerifierConfig()
  {
    return hostnameVerifierConfig;
  }


  @Override
  public void setHostnameVerifierConfig(final HostnameVerifierConfig config)
  {
    hostnameVerifierConfig = config;
  }


  /**
   * Creates any trust managers specific to this context initializer.
   *
   * @return  trust managers
   *
   * @throws  GeneralSecurityException  if an errors occurs while loading the
   * TrustManagers
   */
  protected abstract TrustManager[] createTrustManagers()
    throws GeneralSecurityException;


  /** {@inheritDoc} */
  @Override
  public SSLContext initSSLContext(final String protocol)
    throws GeneralSecurityException
  {
    final KeyManager[] km = getKeyManagers();
    final TrustManager[] tm = getTrustManagers();
    logger.trace(
      "Initialize SSLContext with keyManagers={} and trustManagers={}",
      Arrays.toString(km),
      Arrays.toString(tm));
    final SSLContext ctx = SSLContext.getInstance(protocol);
    ctx.init(km, tm, null);
    return ctx;
  }


  /**
   * Creates an {@link AggregateTrustManager} containing the supplied trust
   * managers.
   *
   * @param  managers  to aggregate
   *
   * @return  array containing a single aggregate trust manager
   */
  protected TrustManager[] aggregateTrustManagers(
    final TrustManager... managers)
  {
    X509TrustManager[] x509Managers = null;
    if (managers != null) {
      x509Managers = new X509TrustManager[managers.length];
      for (int i = 0; i < managers.length; i++) {
        x509Managers[i] = (X509TrustManager) managers[i];
      }
    }
    return new TrustManager[] {new AggregateTrustManager(x509Managers)};
  }
}
