/*
  $Id$

  Copyright (C) 2003-2017 Virginia Tech.
  All rights reserved.

  SEE LICENSE FOR MORE INFORMATION

  Author:  Middleware Services
  Email:   middleware@vt.edu
  Version: $Revision$
  Updated: $Date$
*/
package org.ldaptive.ssl;

import java.security.cert.X509Certificate;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSession;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Adapts a {@link CertificateHostnameVerifier} for use as a {@link
 * HostnameVerifier}.
 *
 * @author  Middleware Services
 * @version  $Revision$ $Date$
 */
public class HostnameVerifierAdapter implements HostnameVerifier
{

  /** Logger for this class. */
  protected final Logger logger = LoggerFactory.getLogger(getClass());

  /** Hostname verifier to adapt. */
  private final CertificateHostnameVerifier hostnameVerifier;


  /**
   * Creates a new hostname verifier adapter.
   *
   * @param  verifier  verifier to adapt
   */
  public HostnameVerifierAdapter(final CertificateHostnameVerifier verifier)
  {
    hostnameVerifier = verifier;
  }


  @Override
  public boolean verify(final String hostname, final SSLSession session)
  {
    boolean b = false;
    try {
      String name = null;
      if (hostname != null) {
        // if IPv6 strip off the "[]"
        if (hostname.startsWith("[") && hostname.endsWith("]")) {
          name = hostname.substring(1, hostname.length() - 1).trim();
        } else {
          name = hostname.trim();
        }
      }
      b = hostnameVerifier.verify(
        name,
        (X509Certificate) session.getPeerCertificates()[0]);
    } catch (SSLPeerUnverifiedException e) {
      logger.warn(
        "Could not get certificate from SSL session {} for hostname {}",
        session,
        hostname,
        e);
    }
    return b;
  }


  @Override
  public String toString()
  {
    return String.format(
      "[%s@%d::hostnameVerifier=%s]",
      getClass().getName(),
      hashCode(),
      hostnameVerifier);
  }
}
