package liquibase.resource;

import liquibase.Scope;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.nio.file.*;
import java.util.Collections;

public class ZipResourceAccessor extends AbstractPathResourceAccessor {

    private FileSystem fileSystem;

    /**
     * Creates a FileSystemResourceAccessor with the given directories/files as the roots.
     */
    public ZipResourceAccessor(File file) throws FileNotFoundException {
        this(file.toPath());
    }

    public ZipResourceAccessor(Path file) throws FileNotFoundException {
        if (file == null) {
            throw new IllegalArgumentException("File must not be null");
        }
        Scope.getCurrentScope().getLog(getClass()).fine("Creating resourceAccessor for file " + file);
        if (!Files.exists(file)) {
            throw new FileNotFoundException("Non-existent file: " + file.toAbsolutePath());
        }
        if (!Files.isRegularFile(file)) {
            throw new IllegalArgumentException("Not a regular file: " + file.toAbsolutePath());
        }
        String lowerCaseName = file.toString().toLowerCase();
        if (!(lowerCaseName.endsWith(".jar") || lowerCaseName.endsWith(".zip"))) {
            throw new IllegalArgumentException("Not a jar or zip file: " + file.toAbsolutePath());
        }

        URI finalUri = URI.create("jar:" + file.toUri());

        try {
            try {
                this.fileSystem = FileSystems.getFileSystem(finalUri);
            } catch (FileSystemNotFoundException e) {
                this.fileSystem = FileSystems.newFileSystem(finalUri, Collections.emptyMap(), Scope.getCurrentScope().getClassLoader());
            }
        } catch (Throwable e) {
            throw new IllegalArgumentException(e.getMessage(), e);
        }
    }

    @Override
    public void close() throws Exception {
        //can't close the filesystem because they often get reused and/or are being used by other things
    }

    @Override
    protected Path getRootPath() {
        return this.fileSystem.getPath("/");
    }

    @Override
    protected Resource createResource(Path file, String pathToAdd) {
        return new PathResource(pathToAdd, file);
    }

}
