package liquibase.command;

import liquibase.Scope;
import liquibase.configuration.*;
import liquibase.exception.CommandExecutionException;
import liquibase.exception.CommandValidationException;
import liquibase.logging.mdc.MdcKey;
import liquibase.util.StringUtil;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.*;
import java.util.regex.Pattern;

/**
 * The primary facade used for executing commands.
 * This object gets configured with the command to run and the input arguments associated with it,
 * then is populated with the result output after {@link #execute()} is called.
 * <p>
 * Named similarly to {@link Scope} because they both define a self-contained unit of values, but this
 * scope is specific to a command rather than being a global scope.
 */
public class CommandScope {

    public static final Pattern NO_PREFIX_PATTERN = Pattern.compile(".*\\.");
    private final CommandDefinition commandDefinition;

    private final SortedMap<String, Object> argumentValues = new TreeMap<>();

    private final Map<Class<?>, Object> dependencies = new HashMap<>();

    /**
     * Config key including the command name. Example `liquibase.command.update`
     */
    private final String completeConfigPrefix;

    /**
     * Config key without the command name. Example `liquibase.command`
     */
    private final String shortConfigPrefix;

    private OutputStream outputStream;

    /**
     * Creates a new scope for the given command.
     */
    public CommandScope(String... commandName) throws CommandExecutionException {
        setOutput(System.out);

        final CommandFactory commandFactory = Scope.getCurrentScope().getSingleton(CommandFactory.class);

        this.commandDefinition = commandFactory.getCommandDefinition(commandName);

        completeConfigPrefix = "liquibase.command." + StringUtil.join(Arrays.asList(this.getCommand().getName()), ".");
        shortConfigPrefix = "liquibase.command";


    }

    /**
     * Returns the {@link CommandDefinition} for the command in this scope.
     */
    public CommandDefinition getCommand() {
        return commandDefinition;
    }

    /**
     * Returns the complete config prefix (without a trailing period) for the command in this scope.
     * @return
     */
    public String getCompleteConfigPrefix() {
        return completeConfigPrefix;
    }

    /**
     * Adds the given key/value pair to the stored argument data.
     *
     * @see #addArgumentValue(CommandArgumentDefinition, Object) for a type-safe version
     */
    public CommandScope addArgumentValue(String argument, Object value) {
        this.argumentValues.put(argument, value);

        return this;
    }

    /**
     * Adds the given key/value pair to the stored argument data.
     */
    public <T> CommandScope addArgumentValue(CommandArgumentDefinition<T> argument, T value) {
        this.argumentValues.put(argument.getName(), value);

        return this;
    }


    /**
     * Returns the detailed information about how an argument is set.
     * <br><br>
     * Prefers values set with {@link #addArgumentValue(String, Object)}, but will also check {@link liquibase.configuration.LiquibaseConfiguration}
     * for settings of liquibase.command.${commandName(s)}.${argumentName} or liquibase.command.${argumentName}
     */
    public <T> ConfiguredValue<T> getConfiguredValue(CommandArgumentDefinition<T> argument) {
        ConfigurationDefinition<T> configDef = createConfigurationDefinition(argument, true);
        ConfiguredValue<T> providedValue = configDef.getCurrentConfiguredValue(new CommandScopeValueProvider());

        if (!providedValue.found() || providedValue.wasDefaultValueUsed()) {
            ConfigurationDefinition<T> noCommandConfigDef = createConfigurationDefinition(argument, false);
            ConfiguredValue<T> noCommandNameProvidedValue = noCommandConfigDef.getCurrentConfiguredValue();
            if (noCommandNameProvidedValue.found() && !noCommandNameProvidedValue.wasDefaultValueUsed()) {
                providedValue = noCommandNameProvidedValue;
            }
        }

        return providedValue;
    }

    /**
     * Convenience method for {@link #getConfiguredValue(CommandArgumentDefinition)}, returning {@link ConfiguredValue#getValue()} along with any
     * {@link CommandArgumentDefinition#getValueConverter()} applied
     */
    public <T> T getArgumentValue(CommandArgumentDefinition<T> argument) {
        final T value = getConfiguredValue(argument).getValue();
        return argument.getValueConverter().convert(value);
    }

    /**
     * Assign a value to a given provided dependency. So if a CommandStep provides class X, at
     * {@link CommandStep#run(CommandResultsBuilder)} method it needs to provide the value for X using this method.
     * commandScope.provideDependency(LockService.class, lockService);
     *
     * Means that this class will LockService.class using object lock
     */
    public  CommandScope provideDependency(Class<?> clazz, Object value) {
        this.dependencies.put(clazz, value);

        return this;
    }

    /**
     * Retrieves the registered dependency object provided by this class identifier
     */
    public <T> Object getDependency(Class<T> clazz) {
        return this.dependencies.get(clazz);
    }

    /**
     * Sets the output stream for this command.
     * The command output sent to this stream should not include status/progress type output, only the actual output itself.
     * Think "what would be piped out", not "what the user is told about what is happening".
     */
    public CommandScope setOutput(OutputStream outputStream) {
        /*
        This is an UnclosableOutputStream because we do not want individual command steps to inadvertently (or
        intentionally) close the System.out OutputStream. Closing System.out renders it unusable for other command
        steps which expect it to still be open.  If the passed OutputStream is null then we do not create it.
         */
        if (outputStream != null) {
            this.outputStream = new UnclosableOutputStream(outputStream);
        } else {
            this.outputStream = null;
        }

        return this;
    }

    public void validate() throws CommandValidationException {
        for (ConfigurationValueProvider provider : Scope.getCurrentScope().getSingleton(LiquibaseConfiguration.class).getProviders()) {
            provider.validate(this);
        }

        for (CommandArgumentDefinition<?> definition : commandDefinition.getArguments().values()) {
            definition.validate(this);
        }

        final List<CommandStep> pipeline = commandDefinition.getPipeline();

        Scope.getCurrentScope().getLog(getClass()).fine("Pipeline for command '" + StringUtil.join(commandDefinition.getName(), " ") + ": " + StringUtil.join(pipeline, " then ", obj -> obj.getClass().getName()));

        for (CommandStep step : pipeline) {
            step.validate(this);
        }
    }

    /**
     * Executes the command in this scope, and returns the results.
     */
    public CommandResults execute() throws CommandExecutionException {
        CommandResultsBuilder resultsBuilder = new CommandResultsBuilder(this, outputStream);
        final List<CommandStep> pipeline = commandDefinition.getPipeline();
        validate();
        Scope.getCurrentScope().addMdcValue(MdcKey.LIQUIBASE_OPERATION, StringUtil.join(commandDefinition.getName(), "-"));
        try {
            for (CommandStep command : pipeline) {
                command.run(resultsBuilder);
            }

            // after executing our pipeline, runs cleanup in inverse order
            for (int i = pipeline.size() -1; i >= 0; i--) {
                CommandStep command = pipeline.get(i);
                if (command instanceof CleanUpCommandStep) {
                    ((CleanUpCommandStep)command).cleanUp(resultsBuilder);
                }
            }
        } catch (Exception e) {
            if (e instanceof CommandExecutionException) {
                throw (CommandExecutionException) e;
            } else {
                throw new CommandExecutionException(e);
            }
        } finally {
            try {
                if (this.outputStream != null) {
                    this.outputStream.flush();
                }
            } catch (Exception e) {
                Scope.getCurrentScope().getLog(getClass()).warning("Error flushing command output stream: " + e.getMessage(), e);
            }
        }

        return resultsBuilder.build();
    }

    private <T> ConfigurationDefinition<T> createConfigurationDefinition(CommandArgumentDefinition<T> argument, boolean includeCommandName) {
        final String key;
        if (includeCommandName) {
            key = completeConfigPrefix;
        } else {
            key = shortConfigPrefix;
        }

        return new ConfigurationDefinition.Builder(key)
                .define(argument.getName(), argument.getDataType())
                .addAliases(argument.getAliases())
                .setDefaultValue(argument.getDefaultValue())
                .setDescription(argument.getDescription())
                .setValueHandler(argument.getValueConverter())
                .setValueObfuscator(argument.getValueObfuscator())
                .buildTemporary();
    }

    /**
     * This class is a wrapper around OutputStreams, and makes them impossible for callers to close.
     */
    private static class UnclosableOutputStream extends FilterOutputStream {
        public UnclosableOutputStream(OutputStream out) {
            super(out);
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            out.write(b, off, len);
        }

        /**
         * This method does not actually close the underlying stream, but rather only flushes it. Callers should not be
         * closing the stream they are given.
         */
        @Override
        public void close() throws IOException {
            out.flush();
        }
    }

    /**
     * Adapts the command-scoped arguments into the overall ValueProvider system
     */
    private class CommandScopeValueProvider extends AbstractMapConfigurationValueProvider {

        @Override
        public int getPrecedence() {
            return -1;
        }

        @Override
        protected Map<?, ?> getMap() {
            return CommandScope.this.argumentValues;
        }

        @Override
        protected String getSourceDescription() {
            return "Command argument";
        }

        @Override
        protected boolean keyMatches(String wantedKey, String storedKey) {
            if (wantedKey.contains(".")) {
                return super.keyMatches(NO_PREFIX_PATTERN.matcher(wantedKey).replaceFirst(""), storedKey);
            } else {
                return super.keyMatches(wantedKey, storedKey);
            }
        }

    }
}
