/**
 * Copyright (c) 2012, Ben Fortuna
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *  o Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of Ben Fortuna nor the names of any other contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package net.fortuna.ical4j.model.property;

import net.fortuna.ical4j.model.Content;
import net.fortuna.ical4j.model.ParameterList;
import net.fortuna.ical4j.model.Property;
import net.fortuna.ical4j.model.PropertyFactory;
import net.fortuna.ical4j.model.component.Participant;
import net.fortuna.ical4j.util.Strings;
import net.fortuna.ical4j.util.Uris;
import net.fortuna.ical4j.validate.ValidationException;
import net.fortuna.ical4j.validate.ValidationResult;
import net.fortuna.ical4j.validate.property.RelationshipPropertyValidators;

import java.net.URI;
import java.net.URISyntaxException;

/**
 * $Id$
 * <p/>
 * Created: [Apr 6, 2004]
 * <p/>
 * Defines an ATTENDEE iCalendar component property.
 *
 * @author benf
 */
public class Attendee extends Property {

    private static final long serialVersionUID = 8430929418723298803L;

    private URI calAddress;

    /**
     * Default constructor.
     */
    public Attendee() {
        super(ATTENDEE);
    }

    /**
     * @param aValue a value string for this component
     * @throws IllegalArgumentException where the specified value string is not a valid uri
     */
    public Attendee(final String aValue) {
        super(ATTENDEE);
        setValue(aValue);
    }

    /**
     * @param aList  a list of parameters for this component
     * @param aValue a value string for this component
     */
    public Attendee(final ParameterList aList, final String aValue) {
        super(ATTENDEE, aList);
        setValue(aValue);
    }

    /**
     * @param aUri a URI
     */
    public Attendee(final URI aUri) {
        super(ATTENDEE);
        calAddress = aUri;
    }

    /**
     * @param aList a list of parameters for this component
     * @param aUri  a URI
     */
    public Attendee(final ParameterList aList, final URI aUri) {
        super(ATTENDEE, aList);
        calAddress = aUri;
    }

    /**
     * Construct an attendee from a participant. The expectation is that the participant is schedulable (i.e.
     * has a calendar address) and an exception will be thrown where this is not the case.
     *
     * @param participant a schedulable participant
     */
    public Attendee(Participant participant) {
        super(ATTENDEE);
        CalendarAddress calAddressProp = participant.getRequiredProperty(Property.CALENDAR_ADDRESS);
        calAddress = calAddressProp.getCalAddress();
    }

    /**
     * Construct an attendee from a participant. The expectation is that the participant is schedulable (i.e.
     * has a calendar address) and an exception will be thrown where this is not the case.
     *
     * @param participant a schedulable participant
     */
    public Attendee(ParameterList parameters, Participant participant) {
        super(ATTENDEE, parameters);
        CalendarAddress calAddressProp = participant.getRequiredProperty(Property.CALENDAR_ADDRESS);
        calAddress = calAddressProp.getCalAddress();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final void setValue(final String aValue) {
        try {
            calAddress = Uris.create(aValue);
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ValidationResult validate() throws ValidationException {
        return RelationshipPropertyValidators.ATTENDEE.validate(this);
    }

    /**
     * @return Returns the calAddress.
     */
    public final URI getCalAddress() {
        return calAddress;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final String getValue() {
        return Uris.decode(Strings.valueOf(getCalAddress()));
    }

    /**
     * @param calAddress The calAddress to set.
     */
    public final void setCalAddress(final URI calAddress) {
        this.calAddress = calAddress;
    }

    @Override
    protected PropertyFactory<Attendee> newFactory() {
        return new Factory();
    }

    public static class Factory extends Content.Factory implements PropertyFactory<Attendee> {
        private static final long serialVersionUID = 1L;

        public Factory() {
            super(ATTENDEE);
        }

        @Override
        public Attendee createProperty(final ParameterList parameters, final String value) {
            return new Attendee(parameters, value);
        }

        @Override
        public Attendee createProperty() {
            return new Attendee();
        }
    }

}
