/*
 *  Copyright (c) 2022, Ben Fortuna
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *   o Redistributions of source code must retain the above copyright
 *  notice, this list of conditions and the following disclaimer.
 *
 *   o Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *
 *   o Neither the name of Ben Fortuna nor the names of any other contributors
 *  may be used to endorse or promote products derived from this software
 *  without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 *  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 *  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

package net.fortuna.ical4j.model.property;

import net.fortuna.ical4j.model.Content;
import net.fortuna.ical4j.model.ParameterList;
import net.fortuna.ical4j.model.Property;
import net.fortuna.ical4j.model.PropertyFactory;
import net.fortuna.ical4j.validate.ValidationException;
import net.fortuna.ical4j.validate.ValidationResult;

/**
 * <pre>
 *     Purpose:
 *     This property value acts as a key for associated iCalendar entities.
 * Value type:
 *     TEXT
 * Property Parameters:
 *     Non-standard parameters can be specified on this property.
 * Conformance:
 *     This property can be specified zero or more times in any iCalendar component.
 * Description:
 *     The value of this property is free-form text that creates an identifier for associated components. All components that use the same REFID value are associated through that value and can be located or retrieved as a group. For example, all of the events in a travel itinerary would have the same REFID value, so as to be grouped together.
 * Format Definition:
 *
 *     This property is defined by the following notation:
 *
 *   refid      = "REFID" refidparam ":" text CRLF
 *
 *
 *   refidparam      = *(";" other-param)
 * </pre>
 * @see <a href="https://www.rfc-editor.org/rfc/rfc9253.html#name-refid">rfc9253</a>
 */
public class RefId extends Property {

    private static final String PROPERTY_NAME = "REFID";

    private String value;

    public RefId() {
        super(PROPERTY_NAME);
    }

    public RefId(String value) {
        super(PROPERTY_NAME);
        this.value = value;
    }

    public RefId(ParameterList aList, String value) {
        super(PROPERTY_NAME, aList);
        setValue(value);
    }

    @Override
    public String getValue() {
        return value;
    }

    @Override
    public void setValue(String aValue) {
        this.value = aValue;
    }

    @Override
    public ValidationResult validate() throws ValidationException {
        return ValidationResult.EMPTY;
    }

    public static class Factory extends Content.Factory implements PropertyFactory<RefId> {
        private static final long serialVersionUID = 1L;

        public Factory() {
            super(PROPERTY_NAME);
        }

        @Override
        public RefId createProperty(final ParameterList parameters, final String value) {
            return new RefId(parameters, value);
        }

        @Override
        public RefId createProperty() {
            return new RefId();
        }
    }

    @Override
    protected PropertyFactory<?> newFactory() {
        return new Factory();
    }
}
