import { LitElement, html, css } from 'lit';
import {EditorState} from "@codemirror/state"
import {EditorView, lineNumbers, keymap, highlightActiveLineGutter, highlightSpecialChars } from "@codemirror/view"
import {defaultKeymap} from "@codemirror/commands"
import { xml } from '@codemirror/lang-xml';
import { javascript } from '@codemirror/lang-javascript';
import { json } from '@codemirror/lang-json';
import { java } from '@codemirror/lang-java';
import { sql } from '@codemirror/lang-sql';
import { yaml } from '@codemirror/lang-yaml';
import { StreamLanguage } from "@codemirror/language"
import { properties } from './properties.js';
import { asciiArmor } from './asciiarmor.js';
import { powerShell } from './powershell.js';
import { basicLight } from './lightTheme.js';
import { basicDark } from './darkTheme.js';

/**
 * Code block UI Component 
 */
class QuiCodeBlock extends LitElement {
    
    static styles = css`
        :host {
            display: block;
            height: 100%;
        }

        #codeMirrorContainer {
            height: 100%;
        }

        slot {
            display:none;
        }
    `;

    static properties = {
        mode: { type: String },
        content: { type: String },
        src: { type: String },
        showLineNumbers: {type: Boolean},
        editable: {type: Boolean},
        value: {type: String, reflect: true },
        theme: {type: String},
        _basicTheme: {type: String}
    };

    constructor() {
        super();
        this.mode = null;
        this.content = '';
        this.showLineNumbers = false;
        this.editable = false;
        this.value = null;
        this._basicTheme = basicDark; // default
    }

    connectedCallback() {
        super.connectedCallback();
        if(this.src){
            // if mode is not provided, figure is out
            if(!this.mode){
                this.mode = this.src.split('.').pop()?.toLowerCase();
            }
            fetch(this.src).then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! Status: ${response.status}`);
                }            
                return response.text();
            }).then(text => {
                this.content = text;
            }).catch(error => {
                this.content = "Fetch error: " +  error;
            });
        }
        if(this.theme && this.theme==="light"){
            this._basicTheme = basicLight;
        }else{
            this._basicTheme = basicDark;
        }

    }

    firstUpdated() {
        
        // See if the content is provided in a slot
        const slotValue = this.shadowRoot.getElementById('slotContent');
        if(slotValue){
            const v = slotValue.assignedNodes()[1];
            if(v && v.textContent){
                this.content = this._removeEmptyLines(v.textContent);
            }
        }

        const codeMirrorContainer = this.shadowRoot.getElementById('codeMirrorContainer');

        // Setup codemirror
        
        const conf = [this._detectMode(), 
            this._basicTheme, 
            highlightActiveLineGutter(),
            highlightSpecialChars(),
            keymap.of([...defaultKeymap])];
        if(this.showLineNumbers){
            conf.push(lineNumbers());
        }

        if(this.editable){
            conf.push(
                EditorView.updateListener.of((v) => {
                    if (v.docChanged) {
                        this.value = v.state.doc.toString();
                    }
                })
            );
        }else{
            conf.push(EditorState.readOnly.of(true));
        }

        const state = EditorState.create({
            doc: this.content,
            extensions: conf
        });
        
        this.editorView = new EditorView({
            state: state,
            readonly: !this.editable,
            parent: codeMirrorContainer
        });        

        this.value = this.editorView.state.doc.toString();
    }

    updated(changedProperties) {
        if ((changedProperties.has('content')) && this.editorView) {
            this._updateContent();
        }else if(changedProperties.has('theme')){
            if(this.theme && this.theme==="light"){
                this._basicTheme = basicLight;
            }else {
                this._basicTheme = basicDark;
            }
        }else if(changedProperties.has('src')){
            if(this.src){
                // if mode is not provided, figure is out
                if(!this.mode){
                    this.mode = this.src.split('.').pop()?.toLowerCase();
                }
                fetch(this.src).then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP error! Status: ${response.status}`);
                    }            
                    return response.text();
                }).then(text => {
                    this.content = text;
                    this._updateContent();
                }).catch(error => {
                    this.content = "Fetch error: " +  error;
                    this._updateContent();
                });
            }
        }else {
            super.updated(changedProperties);
        }
    }

    _updateContent(){
        const transaction = this.editorView.state.update({
            changes: {
                from: 0,
                to: this.editorView.state.doc.length,
                insert: this.content,
            },
        });
        this.editorView.dispatch(transaction);
    }

    render() {
        return html`<div id="codeMirrorContainer">
                <slot id="slotContent"></slot>
            </div>`;
    }

    _detectMode() {
        switch (this.mode) {
            case 'js':
                return javascript();
            case 'pom':
            case 'xml':
                return xml();
            case 'json':
                return json();
            case 'java':
                return java();
            case 'sql':
                return sql();
            case 'yaml':
                return yaml();
            case 'asc':
                return StreamLanguage.define(asciiArmor);
            case 'properties':
                return StreamLanguage.define(properties);
            default:
                return StreamLanguage.define(powerShell);
        }
    }

    _removeEmptyLines(text) {
        // Remove empty lines at the beginning
        text = text.replace(/^\s*\n/, '');
      
        // Remove empty lines at the end
        text = text.replace(/\n\s*$/, '');
      
        return text;
    }
}

customElements.define('qui-code-block', QuiCodeBlock);
