/* Copyright 2011 Achim Nierbeck.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied.
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ops4j.pax.web.service.spi;

import java.util.Collection;

import org.osgi.framework.Bundle;

public class WebEvent {

	public enum WebTopic {
		DEPLOYING("org/osgi/service/web/DEPLOYING"),
		DEPLOYED("org/osgi/service/web/DEPLOYED"), 
		UNDEPLOYING("org/osgi/service/web/UNDEPLOYING"), 
		UNDEPLOYED("org/osgi/service/web/UNDEPLOYED"),
		FAILED("org/osgi/service/web/FAILED");

		private final String topic;

		WebTopic(String topic) {
			this.topic = topic;
		}

		@Override
		public String toString() {
			return topic;
		}

	}
	
	public static final int DEPLOYING = 1;
	public static final int DEPLOYED = 2;
	public static final int UNDEPLOYING = 3;
	public static final int UNDEPLOYED = 4;
	public static final int FAILED = 5;
	
	
	private boolean replay;
	private int type;
	private Bundle bundle;
	private Bundle extenderBundle;
	private Throwable cause;
	private long timestamp;
	private String contextPath;
	private Collection<Long> collisionIds;
	
	public WebEvent(WebEvent event, boolean replay) {
		this.type = event.getType();
		this.contextPath = event.getContextPath();
		this.bundle = event.getBundle();
		this.extenderBundle = event.getExtenderBundle();
		this.collisionIds = event.getCollisionIds();
		this.cause = event.getCause();
		this.timestamp = event.getTimestamp();
		this.replay = replay;
	}
	
	public WebEvent(int type, String contextPath, Bundle bundle, Bundle extenderBundle) {
		this.timestamp = System.currentTimeMillis();
		this.type = type;
		this.contextPath = contextPath;
		this.bundle = bundle;
		this.extenderBundle = extenderBundle;
	}
	
	public WebEvent(int type, String contextPath, Bundle bundle, Bundle extenderBundle, Throwable cause) {
		this(type, contextPath, bundle, extenderBundle);
		this.cause = cause;
	}
	
	public WebEvent(int type, String contextPath, Bundle bundle, Bundle extenderBundle, Collection<Long> ids) {
		this(type, contextPath, bundle, extenderBundle);
		this.collisionIds = ids;
	}
	
	/**
	 * @return the type
	 */
	public int getType() {
		return type;
	}

	/**
	 * @return the replay
	 */
	public boolean isReplay() {
		return replay;
	}

	/**
	 * @return the bundle
	 */
	public Bundle getBundle() {
		return bundle;
	}

	/**
	 * @return the extenderBundle
	 */
	public Bundle getExtenderBundle() {
		return extenderBundle;
	}

	/**
	 * @return the cause
	 */
	public Throwable getCause() {
		return cause;
	}

	/**
	 * @return the timestamp
	 */
	public long getTimestamp() {
		return timestamp;
	}

	/**
	 * @return the contextPath
	 */
	public String getContextPath() {
		return contextPath;
	}

	/**
	 * @return the collisionIds
	 */
	public Collection<Long> getCollisionIds() {
		return collisionIds;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return "WebEvent [replay=" + replay + ", type=" + type + ", bundle="
				+ bundle + ", extenderBundle=" + extenderBundle + ", cause="
				+ cause + ", timestamp=" + timestamp + ", contextPath="
				+ contextPath + ", collisionIds=" + collisionIds + "]";
	}

}
