/*
 * Copyright 2009 OW2 Chameleon
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.chameleon.hsql;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Properties;

import javax.sql.DataSource;

import org.osgi.service.jdbc.DataSourceFactory;
import org.ow2.chameleon.database.connection.ConnectionFactory;

/**
 * HSQL onnection Factory is a component creating HSQL database
 * connection.
 * @author <a href="mailto:chameleon-dev@ow2.org">Chameleon Project Team</a>
 */
public class HSQLConnectionFactory implements ConnectionFactory {

    /**
     * JDBC URL.
     */
    private String url;

    /**
     * User name.
     */
    private String user;

    /**
     * Database password.
     */
    private String password;

    /**
     * Datasource used to create connections.
     */
    private DataSource  source;

    /**
     * Bind the data source factory.
     * This creates a data source used to
     * create connections.
     * @param factory the factory.
     */
    public synchronized void bindFactory(DataSourceFactory factory) {
        Properties props = new Properties();
        props.put(JDBC_URL, url);
        if (user != null) {
            props.put(JDBC_USER, user);
        }
        if (password != null) {
            props.put(DataSourceFactory.JDBC_PASSWORD, password);
        }
        source = factory.createDataSource(props);
    }

    /**
     * Unbind the data source factory.
     */
    public synchronized void unbindFactory() {
        source = null;
    }


    public void setUser(String user) {
        this.user = user;
    }

    public void setPassword(String pwd) {
        this.password = pwd;
    }

    /**
     * Creates connections object.
     * IT relies on the data source created with the injection
     * data source factory.
     * @return a new connection object
     * @throws SQLException if the connection object cannot be created.
     * @see org.ow2.chameleon.database.connection.
     *  ConnectionFactory#createConnection()
     */
    public Connection createConnection() throws SQLException {
        return source.getConnection();
    }

}
