/**
 * PETALS - PETALS Services Platform. Copyright (c) 2005-2006 EBM WebSourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id: RemoteComponentContextImpl.java,v 1.1 2007/06/11 13:34:07 lam Exp $
 * -------------------------------------------------------------------------
 */
package org.objectweb.petals.tools.rmi.server.remote.implementations;

import java.io.IOException;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;

import javax.jbi.JBIException;
import javax.jbi.component.ComponentContext;
import javax.jbi.management.MBeanNames;
import javax.jbi.messaging.MessagingException;
import javax.jbi.servicedesc.ServiceEndpoint;
import javax.management.MBeanServer;
import javax.naming.InitialContext;
import javax.xml.namespace.QName;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import org.objectweb.petals.tools.rmi.server.remote.interfaces.RemoteComponentContext;
import org.objectweb.petals.tools.rmi.server.remote.interfaces.RemoteDeliveryChannel;
import org.objectweb.petals.tools.rmi.server.util.Convert;
import org.ow2.easywsdl.wsdl.WSDLFactory;
import org.ow2.easywsdl.wsdl.api.Description;
import org.ow2.easywsdl.wsdl.api.Endpoint;
import org.ow2.easywsdl.wsdl.api.InterfaceType;
import org.ow2.easywsdl.wsdl.api.Service;
import org.ow2.easywsdl.wsdl.api.WSDLException;
import org.ow2.easywsdl.wsdl.api.WSDLWriter;
import org.ow2.easywsdl.wsdl.api.abstractItf.AbsItfDescription.WSDLVersionConstants;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentFragment;
import org.xml.sax.SAXException;

/**
 * The RMI implementation of component context.
 * 
 * @author nsalatge
 * 
 */
public class RemoteComponentContextImpl extends UnicastRemoteObject implements
        RemoteComponentContext {

    /**
     * the serialVersionUID.
     */
    private static final long serialVersionUID = 7288972323223181501L;

    /**
     * the real component context.
     */
    private ComponentContext componentContext;

    /**
     * the rmi delivery channel.
     */
    private RemoteDeliveryChannel rmiDeliveryChannel;

    /**
     * the MBean Names.
     */
    private MBeanNames mbeanNames;

    /**
     * The map of WSDLs.
     */
    private Map<String, Document> wsdlMap;

    /**
     * The WSDL Factory lazy singleton.
     */
    private static WSDLFactory wsdlFactory;

    /**
     * The WSDL Writer lazy singleton.
     */
    private static WSDLWriter wsdlWriter;

    /**
     * the constructor.
     * 
     * @param c
     *            the component context
     * @param dl
     *            the rmi delivery channel
     * @throws RemoteException
     *             impossbile to realize a rmi access
     */
    public RemoteComponentContextImpl(final ComponentContext c, final RemoteDeliveryChannel dl)
            throws RemoteException {
        super();

        this.componentContext = c;
        this.rmiDeliveryChannel = dl;
        this.wsdlMap = new ConcurrentHashMap<String, Document>();
        this.mbeanNames = new org.objectweb.petals.tools.rmi.common.serializable.MBeanNames(
                this.componentContext.getMBeanNames().getJmxDomainName());
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#activateEndpoint(javax.xml.namespace.QName,
     * javax.xml.namespace.QName, java.lang.String)
     */
    public ServiceEndpoint activateEndpoint(final QName interfaceName, final QName serviceName,
            final String endpointName) throws JBIException, RemoteException {
        try {
            this.wsdlMap.put(endpointName, generateWSDL(interfaceName, serviceName, endpointName));
            ServiceEndpoint se = this.componentContext.activateEndpoint(serviceName, endpointName);
            return new org.objectweb.petals.tools.rmi.common.serializable.ServiceEndpoint(se
                    .getServiceName(), se.getEndpointName(), se.getInterfaces());
        } catch (final JBIException e) {
            throw Convert.convertJBIException(e);
        } catch (final WSDLException e) {
            RemoteException exception = new RemoteException(
                    "Error occured during the WSDL generation");
            exception.setStackTrace(e.getStackTrace());
            throw exception;
        } catch (final SAXException e) {
            RemoteException exception = new RemoteException(
                    "Error occured during the WSDL generation");
            exception.setStackTrace(e.getStackTrace());
            throw exception;
        } catch (final IOException e) {
            RemoteException exception = new RemoteException(
                    "Error occured during the WSDL generation");
            exception.setStackTrace(e.getStackTrace());
            throw exception;
        } catch (final ParserConfigurationException e) {
            RemoteException exception = new RemoteException(
                    "Error occured during the WSDL generation");
            exception.setStackTrace(e.getStackTrace());
            throw exception;
        } catch (TransformerException e) {
            RemoteException exception = new RemoteException(
                    "Error occured during the WSDL generation");
            exception.setStackTrace(e.getStackTrace());
            throw exception;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext
     * #deactivateEndpoint(javax.jbi.servicedesc.ServiceEndpoint)
     */
    public void deactivateEndpoint(final ServiceEndpoint endpoint) throws JBIException,
            RemoteException {
        try {
            ServiceEndpoint se = this.componentContext.getEndpoint(endpoint.getServiceName(),
                    endpoint.getEndpointName());
            this.componentContext.deactivateEndpoint(se);
            this.wsdlMap.remove(endpoint.getEndpointName());
        } catch (JBIException e) {
            throw Convert.convertJBIException(e);
        } finally {
            this.wsdlMap.remove(endpoint.getEndpointName());
        }

    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext
     * #deregisterExternalEndpoint(javax.jbi.servicedesc.ServiceEndpoint)
     */
    public void deregisterExternalEndpoint(final ServiceEndpoint externalEndpoint)
            throws JBIException, RemoteException {
        throw new NoSuchMethodError("deregisterExternalEndpoint: Not implemented.");
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#getComponentName()
     */
    public String getComponentName() throws RemoteException {
        return this.componentContext.getComponentName();
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#getDeliveryChannel()
     */
    public org.objectweb.petals.tools.rmi.server.remote.interfaces.RemoteDeliveryChannel getDeliveryChannel()
            throws MessagingException, RemoteException {
        return this.rmiDeliveryChannel;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#getEndpoint(javax.xml.namespace.QName,
     * java.lang.String)
     */
    public ServiceEndpoint getEndpoint(final QName service, final String name)
            throws RemoteException {
        ServiceEndpoint seC = this.componentContext.getEndpoint(service, name);
        ServiceEndpoint seF = null;
        if (seC != null) {
            seF = new org.objectweb.petals.tools.rmi.common.serializable.ServiceEndpoint(seC
                    .getServiceName(), seC.getEndpointName(), seC.getInterfaces());
        }
        return seF;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext
     * #getEndpointDescriptor(javax.jbi.servicedesc.ServiceEndpoint)
     */
    public Document getEndpointDescriptor(final ServiceEndpoint endpoint) throws JBIException,
            RemoteException {
        Document doc = null;
        ServiceEndpoint se = this.componentContext.getEndpoint(endpoint.getServiceName(), endpoint
                .getEndpointName());
        if (se != null) {
            doc = this.componentContext.getEndpointDescriptor(se);
        }
        return doc;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#getEndpoints(javax.xml.namespace.QName)
     */
    public ServiceEndpoint[] getEndpoints(final QName interfaceName) throws RemoteException {
        ServiceEndpoint[] sesTemp = this.componentContext.getEndpoints(interfaceName);
        ServiceEndpoint[] sesRes = null;
        if (sesTemp != null) {
            sesRes = new ServiceEndpoint[sesTemp.length];
            for (int i = 0; i < sesRes.length; i++) {
                sesRes[i] = new org.objectweb.petals.tools.rmi.common.serializable.ServiceEndpoint(
                        sesTemp[i].getServiceName(), sesTemp[i].getEndpointName(), sesTemp[i]
                                .getInterfaces());
            }
        }
        return sesRes;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#getEndpointsForService(javax.xml.namespace.QName)
     */
    public ServiceEndpoint[] getEndpointsForService(final QName serviceName) throws RemoteException {
        ServiceEndpoint[] sesTemp = this.componentContext.getEndpointsForService(serviceName);
        ServiceEndpoint[] sesRes = null;
        if (sesTemp != null) {
            sesRes = new ServiceEndpoint[sesTemp.length];
            for (int i = 0; i < sesRes.length; i++) {
                sesRes[i] = new org.objectweb.petals.tools.rmi.common.serializable.ServiceEndpoint(
                        sesTemp[i].getServiceName(), sesTemp[i].getEndpointName(), sesTemp[i]
                                .getInterfaces());
            }
        }
        return sesRes;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#getExternalEndpoints(javax.xml.namespace.QName)
     */
    public ServiceEndpoint[] getExternalEndpoints(final QName interfaceName) throws RemoteException {
        ServiceEndpoint[] sesTemp = this.componentContext.getExternalEndpoints(interfaceName);
        ServiceEndpoint[] sesRes = null;
        if (sesTemp != null) {
            sesRes = new ServiceEndpoint[sesTemp.length];
            for (int i = 0; i < sesRes.length; i++) {
                sesRes[i] = new org.objectweb.petals.tools.rmi.common.serializable.ServiceEndpoint(
                        sesTemp[i].getServiceName(), sesTemp[i].getEndpointName(), sesTemp[i]
                                .getInterfaces());
            }
        }
        return sesRes;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext
     * #getExternalEndpointsForService(javax.xml.namespace.QName)
     */
    public ServiceEndpoint[] getExternalEndpointsForService(final QName serviceName)
            throws RemoteException {
        ServiceEndpoint[] sesTemp = this.componentContext
                .getExternalEndpointsForService(serviceName);
        ServiceEndpoint[] sesRes = null;
        if (sesTemp != null) {
            sesRes = new ServiceEndpoint[sesTemp.length];
            for (int i = 0; i < sesRes.length; i++) {
                sesRes[i] = new org.objectweb.petals.tools.rmi.common.serializable.ServiceEndpoint(
                        sesTemp[i].getServiceName(), sesTemp[i].getEndpointName(), sesTemp[i]
                                .getInterfaces());
            }
        }
        return sesRes;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#getInstallRoot()
     */
    public String getInstallRoot() throws RemoteException {
        return this.componentContext.getInstallRoot();
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#getLogger(java.lang.String, java.lang.String)
     */
    public Logger getLogger(final String suffix, final String resourceBundleName)
            throws JBIException, RemoteException {
        throw new NoSuchMethodError("getLogger: Not implemented.");
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#getMBeanNames()
     */
    public MBeanNames getMBeanNames() throws RemoteException {
        return this.mbeanNames;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#getMBeanServer()
     */
    public MBeanServer getMBeanServer() throws RemoteException {
        throw new NoSuchMethodError("getMBeanServer: Not implemented.");
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#getNamingContext()
     */
    public InitialContext getNamingContext() throws RemoteException {
        throw new NoSuchMethodError("getNamingContext: Not implemented.");
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#getTransactionManager()
     */
    public Object getTransactionManager() throws RemoteException {
        throw new NoSuchMethodError("getTransactionManager: Not implemented.");
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext#getWorkspaceRoot()
     */
    public String getWorkspaceRoot() throws RemoteException {
        return this.componentContext.getWorkspaceRoot();
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext
     * #registerExternalEndpoint(javax.jbi.servicedesc.ServiceEndpoint)
     */
    public void registerExternalEndpoint(final ServiceEndpoint externalEndpoint)
            throws JBIException, RemoteException {
        throw new NoSuchMethodError("registerExternalEndpoint: Not implemented.");
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.objectweb.petals.tools.rmi.server.remote.interfaces.
     * RemoteComponentContext
     * #resolveEndpointReference(org.w3c.dom.DocumentFragment)
     */
    public ServiceEndpoint resolveEndpointReference(final DocumentFragment epr)
            throws RemoteException {
        ServiceEndpoint seC = this.componentContext.resolveEndpointReference(epr);
        ServiceEndpoint seF = null;
        if (seC != null) {
            seF = new org.objectweb.petals.tools.rmi.common.serializable.ServiceEndpoint(seC
                    .getServiceName(), seC.getEndpointName(), seC.getInterfaces());
        }
        return seF;
    }

    /**
     * Get the generated WSDL matching the endpoint activated from the
     * component.
     * 
     * @param endpointName
     */
    public Document getOwnedEndpointDescriptor(String endpointName) {
        return this.wsdlMap.get(endpointName);
    }

    /**
     * Allow to generate a WSDL file by providing, interface/service/endpoint
     * name
     * 
     * @param interfaceName
     * @param serviceName
     * @param endpointName
     * @return Document
     * @throws WSDLException
     * @throws SAXException
     * @throws IOException
     * @throws ParserConfigurationException
     * @throws TransformerException
     */
    private Document generateWSDL(final QName interfaceName, final QName serviceName,
            final String endpointName) throws WSDLException, SAXException, IOException,
            ParserConfigurationException, TransformerException {
        final Description description = getWSDLFactory()
                .newDescription(WSDLVersionConstants.WSDL20);

        // Add service/interface/endpoint to the description
        description.setTargetNamespace(serviceName.getNamespaceURI());
        final Service service = description.createService();
        service.setQName(serviceName);
        final InterfaceType _interface = description.createInterface();
        _interface.setQName(interfaceName);
        final Endpoint endpoint = service.createEndpoint();
        endpoint.setName(endpointName);
        service.addEndpoint(endpoint);
        service.setInterface(_interface);
        description.addService(service);
        description.addInterface(_interface);

        return getWSDLWriter().getDocument(description);
    }

    private static final WSDLFactory getWSDLFactory() throws WSDLException {
        if (wsdlFactory == null) {
            wsdlFactory = WSDLFactory.newInstance();
        }

        return wsdlFactory;
    }

    private static final WSDLWriter getWSDLWriter() throws WSDLException {
        if (wsdlWriter == null) {
            wsdlWriter = getWSDLFactory().newWSDLWriter();
        }

        return wsdlWriter;
    }
}
