/**
 * PETALS - PETALS Services Platform. Copyright (c) 2005-2006 EBM WebSourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id: Convert.java,v 1.1 2007/06/11 13:34:07 lam Exp $
 * -------------------------------------------------------------------------
 */

package org.objectweb.petals.tools.rmi.server.util;

import java.net.URI;

import javax.jbi.JBIException;
import javax.jbi.component.ComponentContext;
import javax.jbi.messaging.ExchangeStatus;
import javax.jbi.messaging.MessageExchange;
import javax.jbi.messaging.MessagingException;
import javax.jbi.messaging.NormalizedMessage;

/**
 * Static class that allow to convert fake client rmi object to concrete petals
 * object and vice and versa.
 * 
 * @author nsalatge
 * 
 */
public final class Convert {

    /**
     * IN_MSG.
     */
    public static final String IN_MSG = "in";

    /**
     * inOnly pattern.
     */
    public static final URI IN_ONLY_PATTERN = URI.create("http://www.w3.org/2004/08/wsdl/in-only");

    /**
     * inOptionalOut pattern.
     */
    public static final URI IN_OPTIONAL_OUT_PATTERN = URI
            .create("http://www.w3.org/2004/08/wsdl/in-opt-out");

    /**
     * inOut pattern.
     */
    public static final URI IN_OUT_PATTERN = URI.create("http://www.w3.org/2004/08/wsdl/in-out");

    /**
     * OUT_MSG.
     */
    public static final String OUT_MSG = "out";

    /**
     * RobustInOnly pattern.
     */
    public static final URI ROBUST_IN_ONLY_PATTERN = URI
            .create("http://www.w3.org/2004/08/wsdl/robust-in-only");

    /**
     * the default constructor.
     * 
     */
    private Convert() {

    }

    /**
     * Convert a concrete message exchange to a fake message exchange.
     * 
     * @param concrete
     *            the concrete message
     * @param pattern
     *            the pattern of message
     * @return the fake message
     * @throws MessagingException
     *             impossible to realize the conversion
     */
    public static javax.jbi.messaging.MessageExchange convertConcreteMessageExchangeToFakeMessageExchange(
            final javax.jbi.messaging.MessageExchange concrete, final URI pattern)
            throws MessagingException {
        // Create Fake Message Exchange
        final javax.jbi.messaging.MessageExchange fake = Convert.createAnEmptyFakeMessage(pattern);

        // Set all values in concrete message exchange to fake message exchange
        if (concrete.getEndpoint() != null) {
            fake
                    .setEndpoint(new org.objectweb.petals.tools.rmi.common.serializable.ServiceEndpoint(
                            concrete.getEndpoint().getServiceName(), concrete.getEndpoint()
                                    .getEndpointName(), concrete.getEndpoint().getInterfaces()));
        } else {
            fake.setEndpoint(null);
        }

        if (concrete.getError() != null) {
            fake.setError(Convert.convertException(concrete.getError()));
        }

        if (concrete.getFault() != null) {
            fake
                    .setFault((javax.jbi.messaging.Fault) Convert
                            .convertConcreteToFakeNormalizedMessage(
                                    (org.objectweb.petals.tools.rmi.common.serializable.MessageExchange) fake,
                                    concrete.getFault()));
        }
        fake.setInterfaceName(concrete.getInterfaceName());

        if (concrete.getMessage(Convert.IN_MSG) != null) {
            fake.setMessage(Convert.convertConcreteToFakeNormalizedMessage(
                    (org.objectweb.petals.tools.rmi.common.serializable.MessageExchange) fake,
                    concrete.getMessage(Convert.IN_MSG)), Convert.IN_MSG);
        }

        if (concrete.getMessage(Convert.OUT_MSG) != null) {
            fake.setMessage(Convert.convertConcreteToFakeNormalizedMessage(
                    (org.objectweb.petals.tools.rmi.common.serializable.MessageExchange) fake,
                    concrete.getMessage(Convert.OUT_MSG)), Convert.OUT_MSG);
        }

        fake.setOperation(concrete.getOperation());
        if (concrete.getPropertyNames() != null) {
            for (final Object property : concrete.getPropertyNames()) {
                fake.setProperty((String) property, concrete.getProperty((String) property));
            }
        }
        fake.setService(concrete.getService());
        fake.setStatus(concrete.getStatus());
        ((org.objectweb.petals.tools.rmi.common.serializable.MessageExchange) fake)
                .setExchangeId(concrete.getExchangeId());
        ((org.objectweb.petals.tools.rmi.common.serializable.MessageExchange) fake)
                .setPattern(concrete.getPattern());
        return fake;
    }

    /**
     * Convert a concrete messaging exception to fake messaging exception.
     * 
     * @param e
     *            the concrete messaging exception
     * @return the fake messaging exception
     */
    public static final JBIException convertJBIException(final JBIException e) {
        final JBIException me = new JBIException(e.getMessage());
        me.setStackTrace(e.getStackTrace());
        return me;
    }

    /**
     * Convert a concrete messaging exception to fake messaging exception.
     * 
     * @param e
     *            the concrete messaging exception
     * @return the fake messaging exception
     */
    public static final MessagingException convertMessagingException(final MessagingException e) {
        final MessagingException me = new MessagingException(e.getMessage());
        me.setStackTrace(e.getStackTrace());
        return me;
    }

    /**
     * Convert an error exception to a RMI exception.
     * 
     * @param e
     *            the concrete messaging exception
     * @return the fake messaging exception
     */
    public static Exception convertException(final Exception e) {
        final Exception exception = new Exception(e.getMessage());
        exception.setStackTrace(e.getStackTrace());
        return exception;
    }

    /**
     * Set Concrete Value of Message Exchange in fake message exchange.
     * 
     * @param cc
     *            the component context
     * @param fake
     *            the fake message exchange
     * @param concrete
     *            the concrete message exchange
     * @throws MessagingException
     *             impossible to realize the conversion
     */
    public static void setConcreteValueMEToFakeValueME(final ComponentContext cc,
            final MessageExchange concrete,
            final org.objectweb.petals.tools.rmi.common.serializable.MessageExchange fake)
            throws MessagingException {
        if (fake.getStatus() == ExchangeStatus.DONE) {
            // Status done
            concrete.setStatus(fake.getStatus());
        } else if (fake.getStatus() == ExchangeStatus.ERROR) {
            // Status error
            concrete.setStatus(fake.getStatus());
            concrete.setError(fake.getError());
        } else {
            // Status active
            // Set all values in fake message exchange to concrete message
            // exchange
            if (concrete.getEndpoint() != null) {
                fake
                        .setEndpoint(new org.objectweb.petals.tools.rmi.common.serializable.ServiceEndpoint(
                                concrete.getEndpoint().getServiceName(), concrete.getEndpoint()
                                        .getEndpointName(), concrete.getEndpoint().getInterfaces()));
            } else {
                fake.setEndpoint(null);
            }

            fake.setError(concrete.getError());

            if (concrete.getFault() != null) {
                fake.setFault((javax.jbi.messaging.Fault) Convert
                        .convertConcreteToFakeNormalizedMessage(fake, concrete.getFault()));
            } else {
                fake.setFault(null);
            }

            fake.setInterfaceName(concrete.getInterfaceName());

            if (concrete.getMessage(Convert.IN_MSG) != null) {
                fake.setMessage(Convert.convertConcreteToFakeNormalizedMessage(fake, concrete
                        .getMessage(Convert.IN_MSG)), Convert.IN_MSG);
            }
            if (concrete.getMessage(Convert.OUT_MSG) != null) {
                fake.setMessage(Convert.convertConcreteToFakeNormalizedMessage(fake, concrete
                        .getMessage(Convert.OUT_MSG)), Convert.OUT_MSG);
            }

            fake.setOperation(concrete.getOperation());

            if (concrete.getPropertyNames() != null) {
                for (final Object property : concrete.getPropertyNames()) {
                    fake.setProperty((String) property, concrete.getProperty((String) property));
                }
            }

            fake.setService(concrete.getService());

            fake.setStatus(concrete.getStatus());
        }
    }

    /**
     * Set Fake Value of Message Exchange in concrete message exchange.
     * 
     * @param cc
     *            the component context
     * @param fake
     *            the fake message exchange
     * @param concrete
     *            the concrete message exchange
     * @throws MessagingException
     *             impossible to realize the conversion
     */
    public static void setFakeValueMEToConcreteValueME(final ComponentContext cc,
            final org.objectweb.petals.tools.rmi.common.serializable.MessageExchange fake,
            final MessageExchange concrete) throws MessagingException {

        if (fake.getStatus() == ExchangeStatus.DONE) {
            // Status done
            concrete.setStatus(fake.getStatus());
        } else if (fake.getStatus() == ExchangeStatus.ERROR) {
            // Status error
            concrete.setStatus(fake.getStatus());
            concrete.setError(fake.getError());
        } else {
            // Status active

            // Set all values in fake message exchange to concrete message
            // exchange
            if (fake.getEndpoint() != null) {
                concrete.setEndpoint(cc.getEndpoint(fake.getEndpoint().getServiceName(), fake
                        .getEndpoint().getEndpointName()));
            } else {
                concrete.setEndpoint(null);
            }

            if (fake.getError() != null) {
                concrete.setError(fake.getError());
            }

            if (fake.getFault() != null) {
                concrete
                        .setFault((javax.jbi.messaging.Fault) Convert
                                .convertFakeToConcreteNormalizedMessage(
                                        concrete,
                                        (org.objectweb.petals.tools.rmi.common.serializable.NormalizedMessage) fake
                                                .getFault()));
            }

            concrete.setInterfaceName(fake.getInterfaceName());

            Convert.convertFakeToConcreteNormalizedMessage(fake, concrete);

            concrete.setOperation(fake.getOperation());

            if (fake.getPropertyNames() != null) {
                for (final Object property : fake.getPropertyNames()) {
                    concrete.setProperty((String) property, fake.getProperty((String) property));
                }
            }

            concrete.setService(fake.getService());

            concrete.setStatus(fake.getStatus());
        }
    }

    /**
     * convert a concrete NormalizedMessage to a fake NormalizedMessage.
     * 
     * @param fakeME
     *            the fake message exchange
     * @param concrete
     *            the concrete NormalizedMessage
     * @return the fake NormalizedMessage
     * @throws MessagingException
     *             impossible to realize the conversion
     */
    private static final NormalizedMessage convertConcreteToFakeNormalizedMessage(
            final org.objectweb.petals.tools.rmi.common.serializable.MessageExchange fakeME,
            final NormalizedMessage concrete) throws MessagingException {
        NormalizedMessage fake = null;
        if (concrete != null) {
            if (concrete instanceof javax.jbi.messaging.Fault) {
                // Create a fake FaultMessage
                fake = fakeME.createFault();
            } else {
                // Create a fake NormalizedMessage
                fake = fakeME.createMessage();
            }

            // Set all values in fake normalized message to concrete normalized
            // message
            fake.setContent(concrete.getContent());

            if (concrete.getPropertyNames() != null) {
                for (final Object property : concrete.getPropertyNames()) {
                    fake.setProperty((String) property, concrete.getProperty((String) property));
                }
            }

            fake.setSecuritySubject(concrete.getSecuritySubject());

            if (concrete.getAttachmentNames() != null) {
                Convert.deleteAllAttachments(fake);
                for (final Object attachmentName : concrete.getAttachmentNames()) {
                    fake.addAttachment((String) attachmentName, concrete
                            .getAttachment((String) attachmentName));
                }
            } else {
                Convert.deleteAllAttachments(fake);
            }
        }
        return fake;
    }

    /**
     * convert a fake NormalizedMessage to a concrete NormalizedMessage.
     * 
     * @param concreteME
     *            the concrete message exchange
     * @param fake
     *            the fake NormalizedMessage
     * @return the concrete NormalizedMessage
     * @throws MessagingException
     *             impossible to realize the conversion
     */
    private static final NormalizedMessage convertFakeToConcreteNormalizedMessage(
            final MessageExchange concreteME,
            final org.objectweb.petals.tools.rmi.common.serializable.NormalizedMessage fake)
            throws MessagingException {
        NormalizedMessage concrete = null;

        if (fake != null) {
            if (fake instanceof javax.jbi.messaging.Fault) {
                // Create a concrete FaultMessage
                concrete = concreteME.createFault();
            } else {
                // Create a concrete NormalizedMessage
                concrete = concreteME.createMessage();
            }

            // Set all values in fake normalized message to concrete normalized
            // message
            concrete.setContent(fake.getContent());

            if (fake.getPropertyNames() != null) {
                for (final Object property : fake.getPropertyNames()) {
                    concrete.setProperty((String) property, fake.getProperty((String) property));
                }
            }

            concrete.setSecuritySubject(fake.getSecuritySubject());

            if (fake.getAttachmentNames() != null) {
                Convert.deleteAllAttachments(concrete);
                for (final Object attachmentName : fake.getAttachmentNames()) {
                    concrete.addAttachment((String) attachmentName, fake
                            .getAttachment((String) attachmentName));
                }
            } else {
                Convert.deleteAllAttachments(concrete);
            }
        }
        return concrete;
    }

    /**
     * Convert fake to concrete message.
     * 
     * @param fake
     *            the fake message
     * @param concrete
     *            the concrete message
     * @throws MessagingException
     *             impossible to convert the message
     */
    private static final void convertFakeToConcreteNormalizedMessage(
            final org.objectweb.petals.tools.rmi.common.serializable.MessageExchange fake,
            final MessageExchange concrete) throws MessagingException {
        if (fake.getMessage(Convert.IN_MSG) != null) {
            if (concrete.getMessage(Convert.IN_MSG) == null) {
                concrete.setMessage(Convert.convertFakeToConcreteNormalizedMessage(concrete,
                        (org.objectweb.petals.tools.rmi.common.serializable.NormalizedMessage) fake
                                .getMessage(Convert.IN_MSG)), Convert.IN_MSG);
            }
        }

        if (fake.getMessage(Convert.OUT_MSG) != null) {
            if (concrete.getMessage(Convert.OUT_MSG) == null) {
                concrete.setMessage(Convert.convertFakeToConcreteNormalizedMessage(concrete,
                        (org.objectweb.petals.tools.rmi.common.serializable.NormalizedMessage) fake
                                .getMessage(Convert.OUT_MSG)), Convert.OUT_MSG);
            }
        }
    }

    /**
     * Create an empty fake message.
     * 
     * @param pattern
     *            the pattern
     * @return the empty fake message
     */
    private static final javax.jbi.messaging.MessageExchange createAnEmptyFakeMessage(
            final URI pattern) {
        javax.jbi.messaging.MessageExchange fake = null;
        if (pattern != null) {
            if (pattern.equals(Convert.IN_ONLY_PATTERN)) {
                fake = new org.objectweb.petals.tools.rmi.common.serializable.messagingtypes.InOnly();
            } else if (pattern.equals(Convert.IN_OUT_PATTERN)) {
                fake = new org.objectweb.petals.tools.rmi.common.serializable.messagingtypes.InOut();
            } else if (pattern.equals(Convert.IN_OPTIONAL_OUT_PATTERN)) {
                fake = new org.objectweb.petals.tools.rmi.common.serializable.messagingtypes.InOptionalOut();
            } else if (pattern.equals(Convert.ROBUST_IN_ONLY_PATTERN)) {
                fake = new org.objectweb.petals.tools.rmi.common.serializable.messagingtypes.RobustInOnly();
            } else if (pattern == null) {
                fake = new org.objectweb.petals.tools.rmi.common.serializable.MessageExchange();
            }
        }
        return fake;
    }

    /**
     * Delete all the attachments.
     * 
     * @param nm
     *            the normalized message
     * @throws MessagingException
     *             impossible to delete the message
     */
    private static final void deleteAllAttachments(final NormalizedMessage nm)
            throws MessagingException {
        // Delete all attachments
        if (nm.getAttachmentNames() != null) {
            for (final Object attachmentName : nm.getAttachmentNames()) {
                nm.removeAttachment((String) attachmentName);
            }
        }
    }
}
