/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.picketlink.idm.internal.util;

import org.picketlink.idm.model.Agent;
import org.picketlink.idm.model.Group;
import org.picketlink.idm.model.IdentityType;
import org.picketlink.idm.model.Role;
import org.picketlink.idm.model.User;
import org.picketlink.idm.query.QueryParameter;

/**
 * General purpose Util
 *
 * @author anil saldhana
 * @since Sep 13, 2012
 *
 */
public class IDMUtil {

    public static boolean isGroupType(Class<? extends IdentityType> identityType) {
        return Group.class.isAssignableFrom(identityType);
    }

    public static  boolean isRoleType(Class<? extends IdentityType> identityType) {
        return Role.class.isAssignableFrom(identityType);
    }

    public static  boolean isUserType(Class<? extends IdentityType> identityType) {
        return User.class.isAssignableFrom(identityType);
    }

    public static  boolean isAgentType(Class<? extends IdentityType> identityType) {
        return Agent.class.isAssignableFrom(identityType);
    }


    /**
     * Return default criterias for sorting query results. Those are used by default if there are not sorting criterias specified by user
     *
     * @param clazz identity type
     * @return default sorting criteria for particular identity type
     */
    public static QueryParameter[] getDefaultParamsForSorting(Class<? extends IdentityType> clazz) {
        if (isAgentType(clazz)) {
            return new QueryParameter[] { Agent.LOGIN_NAME };
        } else if (isGroupType(clazz)) {
            return new QueryParameter[] { Group.NAME };
        } else if (isRoleType(clazz)) {
            return new QueryParameter[] { Role.NAME };
        } else {
            return new QueryParameter[] { IdentityType.ID };
        }
    }

    /**
     * Match two arrays for equality
     *
     * @param arr1
     * @param arr2
     * @return
     */
    public static boolean arraysEqual(String[] arr1, String[] arr2) {
        if (arr1 != null && arr2 == null) {
            return false;
        }
        if (arr1 == null && arr2 == null) {
            return true;
        }
        if (arr1 == null && arr2 != null) {
            return false;
        }

        int length1 = arr1.length;
        int length2 = arr2.length;
        if (length1 != length2) {
            return false;
        }
        boolean foundMatch = false;
        for (int i = 0; i < length1; i++) {
            for (int j = 0; j < length2; j++) {
                if (arr1[i].equals(arr2[j])) {
                    foundMatch = true;
                    break;
                }
            }
            if (foundMatch == false) {
                return false;
            }
            // reset
            foundMatch = false;
        }
        return true;
    }

    /**
     * Sleep given number of milliseconds
     *
     * @param ms
     */
    public static void sleep(int ms) {
        try {
            Thread.sleep(ms);
        } catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
        }
    }

}