/*
 * Decompiled with CFR 0.152.
 */
package cn.nukkit.positiontracking;

import cn.nukkit.Player;
import cn.nukkit.Server;
import cn.nukkit.api.PowerNukkitOnly;
import cn.nukkit.api.Since;
import cn.nukkit.inventory.Inventory;
import cn.nukkit.item.Item;
import cn.nukkit.item.ItemCompassLodestone;
import cn.nukkit.network.protocol.DataPacket;
import cn.nukkit.network.protocol.PositionTrackingDBServerBroadcastPacket;
import cn.nukkit.positiontracking.NamedPosition;
import cn.nukkit.positiontracking.PositionTracking;
import cn.nukkit.positiontracking.PositionTrackingStorage;
import com.google.common.base.Preconditions;
import com.google.common.collect.MapMaker;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.ints.IntList;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.regex.Pattern;
import java.util.stream.IntStream;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

@ParametersAreNonnullByDefault
@PowerNukkitOnly
@Since(value="1.4.0.0-PN")
public class PositionTrackingService
implements Closeable {
    private static final Logger log = LogManager.getLogger(PositionTrackingService.class);
    private static final Pattern FILENAME_PATTERN = Pattern.compile("^\\d+\\.pnt$", 2);
    private static final FilenameFilter FILENAME_FILTER = (dir, name) -> FILENAME_PATTERN.matcher(name).matches() && new File(dir, name).isFile();
    private final TreeMap<Integer, WeakReference<PositionTrackingStorage>> storage = new TreeMap(Comparator.reverseOrder());
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private final File folder;
    private final Map<Player, Map<PositionTrackingStorage, IntSet>> tracking = new MapMaker().weakKeys().makeMap();

    public PositionTrackingService(File folder) throws FileNotFoundException {
        if (!folder.isDirectory() && !folder.mkdirs()) {
            throw new FileNotFoundException("Failed to create the folder " + folder);
        }
        this.folder = folder;
        WeakReference<Object> emptyRef = new WeakReference<Object>(null);
        Arrays.stream((Object[])Optional.ofNullable(folder.list(FILENAME_FILTER)).orElseThrow(() -> new FileNotFoundException("Invalid folder: " + folder))).map(name -> Integer.parseInt(name.substring(0, name.length() - 4))).forEachOrdered(startIndex -> this.storage.put((Integer)startIndex, (WeakReference<PositionTrackingStorage>)emptyRef));
    }

    private boolean hasTrackingDevice(Player player, @Nullable Inventory inventory, int trackingHandler) throws IOException {
        if (inventory == null) {
            return false;
        }
        int size = inventory.getSize();
        for (int i = 0; i < size; ++i) {
            if (!this.isTrackingDevice(player, inventory.getItem(i), trackingHandler)) continue;
            return true;
        }
        return false;
    }

    private boolean isTrackingDevice(Player player, @Nullable Item item, int trackingHandler) throws IOException {
        if (item == null || item.getId() != 741 || !(item instanceof ItemCompassLodestone)) {
            return false;
        }
        ItemCompassLodestone compassLodestone = (ItemCompassLodestone)item;
        if (compassLodestone.getTrackingHandle() != trackingHandler) {
            return false;
        }
        PositionTracking position = this.getPosition(trackingHandler);
        return position != null && position.getLevelName().equals(player.getLevelName());
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public boolean hasTrackingDevice(Player player, int trackingHandler) throws IOException {
        for (Inventory inventory : this.inventories(player)) {
            if (!this.hasTrackingDevice(player, inventory, trackingHandler)) continue;
            return true;
        }
        return false;
    }

    private void sendTrackingUpdate(Player player, int trackingHandler, PositionTracking pos) {
        if (player.getLevelName().equals(pos.getLevelName())) {
            PositionTrackingDBServerBroadcastPacket packet = new PositionTrackingDBServerBroadcastPacket();
            packet.setAction(PositionTrackingDBServerBroadcastPacket.Action.UPDATE);
            packet.setPosition(pos);
            packet.setDimension(player.getLevel().getDimension());
            packet.setTrackingId(trackingHandler);
            packet.setStatus(0);
            player.dataPacket(packet);
        } else {
            this.sendTrackingDestroy(player, trackingHandler);
        }
    }

    private void sendTrackingDestroy(Player player, int trackingHandler) {
        PositionTrackingDBServerBroadcastPacket packet = this.destroyPacket(trackingHandler);
        player.dataPacket(packet);
    }

    @Nullable
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized PositionTracking startTracking(Player player, int trackingHandler, boolean validate) throws IOException {
        Preconditions.checkArgument(trackingHandler >= 0, "Tracking handler must be positive");
        if (this.isTracking(player, trackingHandler, validate)) {
            PositionTracking position = this.getPosition(trackingHandler);
            if (position != null) {
                this.sendTrackingUpdate(player, trackingHandler, position);
                return position;
            }
            this.stopTracking(player, trackingHandler);
            return null;
        }
        if (validate && !this.hasTrackingDevice(player, trackingHandler)) {
            return null;
        }
        PositionTrackingStorage storage = this.getStorageForHandler(trackingHandler);
        if (storage == null) {
            return null;
        }
        PositionTracking position = storage.getPosition(trackingHandler);
        if (position == null) {
            return null;
        }
        this.tracking.computeIfAbsent(player, p -> new HashMap()).computeIfAbsent(storage, s -> new IntOpenHashSet(3)).add(trackingHandler);
        return position;
    }

    private PositionTrackingDBServerBroadcastPacket destroyPacket(int trackingHandler) {
        PositionTrackingDBServerBroadcastPacket packet = new PositionTrackingDBServerBroadcastPacket();
        packet.setAction(PositionTrackingDBServerBroadcastPacket.Action.DESTROY);
        packet.setTrackingId(trackingHandler);
        packet.setDimension(0);
        packet.setPosition(0, 0, 0);
        packet.setStatus(2);
        return packet;
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized boolean stopTracking(Player player) {
        Map<PositionTrackingStorage, IntSet> toRemove = this.tracking.remove(player);
        if (toRemove != null && player.isOnline()) {
            DataPacket[] packets = (DataPacket[])toRemove.values().stream().flatMapToInt(handlers -> IntStream.of(handlers.toIntArray())).mapToObj(this::destroyPacket).toArray(DataPacket[]::new);
            player.getServer().batchPackets(new Player[]{player}, packets);
        }
        return toRemove != null;
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized boolean stopTracking(Player player, int trackingHandler) {
        Map<PositionTrackingStorage, IntSet> tracking = this.tracking.get(player);
        if (tracking == null) {
            return false;
        }
        for (Map.Entry<PositionTrackingStorage, IntSet> entry : tracking.entrySet()) {
            if (!entry.getValue().remove(trackingHandler)) continue;
            if (entry.getValue().isEmpty()) {
                tracking.remove(entry.getKey());
            }
            player.dataPacket(this.destroyPacket(trackingHandler));
            return true;
        }
        return false;
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized boolean isTracking(Player player, int trackingHandler, boolean validate) throws IOException {
        Map<PositionTrackingStorage, IntSet> tracking = this.tracking.get(player);
        if (tracking == null) {
            return false;
        }
        for (IntSet value : tracking.values()) {
            if (!value.contains(trackingHandler)) continue;
            if (validate && !this.hasTrackingDevice(player, trackingHandler)) {
                this.stopTracking(player, trackingHandler);
                return false;
            }
            return true;
        }
        return false;
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized void forceRecheckAllPlayers() {
        this.tracking.keySet().removeIf(p -> !p.isOnline());
        HashMap<Player, IntList> toRemove = new HashMap<Player, IntList>(2);
        for (Map.Entry<Player, Map<PositionTrackingStorage, IntSet>> entry : this.tracking.entrySet()) {
            Player player2 = entry.getKey();
            for (Map.Entry<PositionTrackingStorage, IntSet> entry2 : entry.getValue().entrySet()) {
                entry2.getValue().forEach(trackingHandler -> {
                    try {
                        if (!this.hasTrackingDevice(player2, trackingHandler)) {
                            toRemove.computeIfAbsent(player2, p -> new IntArrayList(2)).add(trackingHandler);
                        }
                    }
                    catch (IOException e) {
                        log.error("Failed to update the tracking handler " + trackingHandler + " for player " + player2.getName(), (Throwable)e);
                    }
                });
            }
        }
        toRemove.forEach((player, list) -> list.forEach(handler -> this.stopTracking((Player)player, handler)));
        Server.getInstance().getOnlinePlayers().values().forEach(this::detectNeededUpdates);
    }

    private Iterable<Inventory> inventories(final Player player) {
        return () -> new Iterator<Inventory>(){
            int next = 0;

            @Override
            public boolean hasNext() {
                return this.next <= 4;
            }

            @Override
            public Inventory next() {
                switch (this.next++) {
                    case 0: {
                        return player.getInventory();
                    }
                    case 1: {
                        return player.getCursorInventory();
                    }
                    case 2: {
                        return player.getOffhandInventory();
                    }
                    case 3: {
                        return player.getCraftingGrid();
                    }
                    case 4: {
                        return player.getTopWindow().orElse(null);
                    }
                }
                throw new NoSuchElementException();
            }
        };
    }

    private void detectNeededUpdates(Player player) {
        for (Inventory inventory : this.inventories(player)) {
            if (inventory == null) continue;
            int size = inventory.getSize();
            for (int slot = 0; slot < size; ++slot) {
                ItemCompassLodestone compass;
                int trackingHandle;
                Item item = inventory.getItem(slot);
                if (item.getId() != 741 || !(item instanceof ItemCompassLodestone) || (trackingHandle = (compass = (ItemCompassLodestone)item).getTrackingHandle()) == 0) continue;
                try {
                    PositionTracking pos = this.getPosition(trackingHandle);
                    if (pos == null || !pos.getLevelName().equals(player.getLevelName())) continue;
                    this.startTracking(player, trackingHandle, false);
                    continue;
                }
                catch (IOException e) {
                    log.error("Failed to get the position of the tracking handler " + trackingHandle, (Throwable)e);
                }
            }
        }
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public void forceRecheck(Player player) {
        Map<PositionTrackingStorage, IntSet> tracking = this.tracking.get(player);
        if (tracking != null) {
            IntArrayList toRemove = new IntArrayList(2);
            for (Map.Entry<PositionTrackingStorage, IntSet> entry2 : tracking.entrySet()) {
                entry2.getValue().forEach(trackingHandler -> {
                    try {
                        if (!this.hasTrackingDevice(player, trackingHandler)) {
                            toRemove.add(trackingHandler);
                        }
                    }
                    catch (IOException e) {
                        log.error("Failed to update the tracking handler " + trackingHandler + " for player " + player.getName(), (Throwable)e);
                    }
                });
            }
            toRemove.forEach(handler -> this.stopTracking(player, handler));
        }
        this.detectNeededUpdates(player);
    }

    @Nullable
    private synchronized Integer findStorageForHandler(@Nonnull Integer handler) {
        Integer best = null;
        for (Integer startIndex : this.storage.keySet()) {
            int comp = startIndex.compareTo(handler);
            if (comp == 0) {
                return startIndex;
            }
            if (comp >= 0 || best != null && best.compareTo(startIndex) >= 0) continue;
            best = startIndex;
        }
        return best;
    }

    @Nonnull
    private synchronized PositionTrackingStorage loadStorage(@Nonnull Integer startIndex) throws IOException {
        PositionTrackingStorage trackingStorage = (PositionTrackingStorage)this.storage.get(startIndex).get();
        if (trackingStorage != null) {
            return trackingStorage;
        }
        PositionTrackingStorage positionTrackingStorage = new PositionTrackingStorage(startIndex, new File(this.folder, startIndex + ".pnt"));
        this.storage.put(startIndex, new WeakReference<PositionTrackingStorage>(positionTrackingStorage));
        return positionTrackingStorage;
    }

    @Nullable
    private synchronized PositionTrackingStorage getStorageForHandler(@Nonnull Integer trackingHandler) throws IOException {
        Integer startIndex = this.findStorageForHandler(trackingHandler);
        if (startIndex == null) {
            return null;
        }
        PositionTrackingStorage storage = this.loadStorage(startIndex);
        if (trackingHandler > storage.getMaxHandler()) {
            return null;
        }
        return storage;
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized int addOrReusePosition(NamedPosition position) throws IOException {
        this.checkClosed();
        OptionalInt trackingHandler = this.findTrackingHandler(position);
        if (trackingHandler.isPresent()) {
            return trackingHandler.getAsInt();
        }
        return this.addNewPosition(position);
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized int addNewPosition(NamedPosition position) throws IOException {
        return this.addNewPosition(position, true);
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized int addNewPosition(NamedPosition position, boolean enabled) throws IOException {
        PositionTrackingStorage trackingStorage;
        this.checkClosed();
        int next = 1;
        if (!this.storage.isEmpty()) {
            trackingStorage = this.loadStorage(this.storage.firstKey());
            OptionalInt handler = trackingStorage.addNewPosition(position, enabled);
            if (handler.isPresent()) {
                return handler.getAsInt();
            }
            next = trackingStorage.getMaxHandler();
        }
        trackingStorage = new PositionTrackingStorage(next, new File(this.folder, next + ".pnt"));
        this.storage.put(next, new WeakReference<PositionTrackingStorage>(trackingStorage));
        return trackingStorage.addNewPosition(position, enabled).orElseThrow(InternalError::new);
    }

    @Nonnull
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public OptionalInt findTrackingHandler(NamedPosition position) throws IOException {
        IntList handlers = this.findTrackingHandlers(position, true, 1);
        if (!handlers.isEmpty()) {
            return OptionalInt.of(handlers.getInt(0));
        }
        return OptionalInt.empty();
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized boolean invalidateHandler(int trackingHandler) throws IOException {
        this.checkClosed();
        PositionTrackingStorage storage = this.getStorageForHandler(trackingHandler);
        if (storage == null) {
            return false;
        }
        if (!storage.hasPosition(trackingHandler, false)) {
            return false;
        }
        storage.invalidateHandler(trackingHandler);
        this.handlerDisabled(trackingHandler);
        return true;
    }

    private void handlerDisabled(int trackingHandler) {
        ArrayList<Player> players = new ArrayList<Player>();
        block0: for (Map.Entry<Player, Map<PositionTrackingStorage, IntSet>> playerMapEntry : this.tracking.entrySet()) {
            for (IntSet value : playerMapEntry.getValue().values()) {
                if (!value.contains(trackingHandler)) continue;
                players.add(playerMapEntry.getKey());
                continue block0;
            }
        }
        if (!players.isEmpty()) {
            Server.getInstance().batchPackets(players.toArray(Player.EMPTY_ARRAY), new DataPacket[]{this.destroyPacket(trackingHandler)});
        }
    }

    private void handlerEnabled(int trackingHandler) throws IOException {
        Server server = Server.getInstance();
        for (Player player : server.getOnlinePlayers().values()) {
            if (!this.hasTrackingDevice(player, trackingHandler) || this.isTracking(player, trackingHandler, false)) continue;
            this.startTracking(player, trackingHandler, false);
        }
    }

    @Nullable
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public PositionTracking getPosition(int trackingHandle) throws IOException {
        return this.getPosition(trackingHandle, true);
    }

    @Nullable
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public PositionTracking getPosition(int trackingHandle, boolean onlyEnabled) throws IOException {
        this.checkClosed();
        PositionTrackingStorage trackingStorage = this.getStorageForHandler(trackingHandle);
        if (trackingStorage == null) {
            return null;
        }
        return trackingStorage.getPosition(trackingHandle, onlyEnabled);
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized boolean isEnabled(int trackingHandler) throws IOException {
        this.checkClosed();
        PositionTrackingStorage trackingStorage = this.getStorageForHandler(trackingHandler);
        return trackingStorage != null && trackingStorage.isEnabled(trackingHandler);
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized boolean setEnabled(int trackingHandler, boolean enabled) throws IOException {
        this.checkClosed();
        PositionTrackingStorage trackingStorage = this.getStorageForHandler(trackingHandler);
        if (trackingStorage == null) {
            return false;
        }
        if (trackingStorage.setEnabled(trackingHandler, enabled)) {
            if (enabled) {
                this.handlerEnabled(trackingHandler);
            } else {
                this.handlerDisabled(trackingHandler);
            }
            return true;
        }
        return false;
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized boolean hasPosition(int trackingHandler) throws IOException {
        return this.hasPosition(trackingHandler, true);
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized boolean hasPosition(int trackingHandler, boolean onlyEnabled) throws IOException {
        this.checkClosed();
        Integer startIndex = this.findStorageForHandler(trackingHandler);
        if (startIndex == null) {
            return false;
        }
        if (!this.storage.containsKey(startIndex)) {
            return false;
        }
        return this.loadStorage(startIndex).hasPosition(trackingHandler, onlyEnabled);
    }

    @Nonnull
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized IntList findTrackingHandlers(NamedPosition pos) throws IOException {
        return this.findTrackingHandlers(pos, true);
    }

    @Nonnull
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized IntList findTrackingHandlers(NamedPosition pos, boolean onlyEnabled) throws IOException {
        return this.findTrackingHandlers(pos, onlyEnabled, Integer.MAX_VALUE);
    }

    @Nonnull
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized IntList findTrackingHandlers(NamedPosition pos, boolean onlyEnabled, int limit) throws IOException {
        this.checkClosed();
        IntArrayList list = new IntArrayList();
        for (Integer startIndex : this.storage.descendingKeySet()) {
            list.addAll(this.loadStorage(startIndex).findTrackingHandlers(pos, onlyEnabled, limit - list.size()));
            if (list.size() < limit) continue;
            break;
        }
        return list;
    }

    @Override
    public synchronized void close() throws IOException {
        this.closed.set(true);
        Throwable exception = null;
        for (WeakReference<PositionTrackingStorage> ref : this.storage.values()) {
            PositionTrackingStorage positionTrackingStorage = (PositionTrackingStorage)ref.get();
            if (positionTrackingStorage == null) continue;
            try {
                positionTrackingStorage.close();
            }
            catch (Throwable e) {
                if (exception == null) {
                    exception = new IOException(e);
                    continue;
                }
                exception.addSuppressed(e);
            }
        }
        if (exception != null) {
            throw exception;
        }
    }

    protected void finalize() throws Throwable {
        this.close();
    }

    private void checkClosed() throws IOException {
        if (this.closed.get()) {
            throw new IOException("The service is closed");
        }
    }
}

