/*
 * Decompiled with CFR 0.152.
 */
package cn.nukkit.positiontracking;

import cn.nukkit.api.PowerNukkitOnly;
import cn.nukkit.api.Since;
import cn.nukkit.math.NukkitMath;
import cn.nukkit.positiontracking.NamedPosition;
import cn.nukkit.positiontracking.PositionTracking;
import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.ints.IntList;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

@ParametersAreNonnullByDefault
@PowerNukkitOnly
@Since(value="1.4.0.0-PN")
public class PositionTrackingStorage
implements Closeable {
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public static final int DEFAULT_MAX_STORAGE = 500;
    private static final byte[] HEADER = new byte[]{12, 32, 32, 80, 78, 80, 84, 68, 66, 49};
    private final int startIndex;
    private final int maxStorage;
    private final long garbagePos;
    private final long stringHeapPos;
    private final RandomAccessFile persistence;
    private final Cache<Integer, Optional<PositionTracking>> cache = CacheBuilder.newBuilder().expireAfterAccess(5L, TimeUnit.MINUTES).concurrencyLevel(1).build();
    private int nextIndex;

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public PositionTrackingStorage(int startIndex, File persistenceFile) throws IOException {
        this(startIndex, persistenceFile, 0);
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public PositionTrackingStorage(int startIndex, File persistenceFile, int maxStorage) throws IOException {
        Preconditions.checkArgument((startIndex > 0 ? 1 : 0) != 0, (String)"Start index must be positive. Got {}", (int)startIndex);
        this.startIndex = startIndex;
        if (maxStorage <= 0) {
            maxStorage = 500;
        }
        boolean created = false;
        if (!persistenceFile.isFile()) {
            if (!persistenceFile.getParentFile().isDirectory() && !persistenceFile.getParentFile().mkdirs()) {
                throw new FileNotFoundException("Could not create the directory " + persistenceFile.getParent());
            }
            if (!persistenceFile.createNewFile()) {
                throw new FileNotFoundException("Could not create the file " + persistenceFile);
            }
            created = true;
        } else if (persistenceFile.length() == 0L) {
            created = true;
        }
        this.persistence = new RandomAccessFile(persistenceFile, "rwd");
        try {
            if (created) {
                this.persistence.write(ByteBuffer.allocate(HEADER.length + 4 + 4 + 4).put(HEADER).putInt(maxStorage).putInt(startIndex).putInt(startIndex).array());
                this.maxStorage = maxStorage;
                this.nextIndex = startIndex;
            } else {
                int start;
                int next;
                int max;
                byte[] check = new byte[HEADER.length];
                EOFException eof = null;
                try {
                    this.persistence.readFully(check);
                    byte[] buf = new byte[12];
                    this.persistence.readFully(buf);
                    ByteBuffer buffer = ByteBuffer.wrap(buf);
                    max = buffer.getInt();
                    next = buffer.getInt();
                    start = buffer.getInt();
                }
                catch (EOFException e) {
                    eof = e;
                    max = 0;
                    next = 0;
                    start = 0;
                }
                if (eof != null || max <= 0 || next <= 0 || start <= 0 || !Arrays.equals(check, HEADER)) {
                    throw new IOException("The file " + persistenceFile + " is not a valid PowerNukkit TrackingPositionDB persistence file.", eof);
                }
                if (start != startIndex) {
                    throw new IllegalArgumentException("The start index " + startIndex + " was given but the file " + persistenceFile + " has start index " + start);
                }
                this.maxStorage = maxStorage = max;
                this.nextIndex = next;
            }
            this.garbagePos = this.getAxisPos(startIndex + maxStorage);
            this.stringHeapPos = this.garbagePos + 4L + 180L;
            if (created) {
                this.persistence.seek(this.stringHeapPos - 1L);
                this.persistence.writeByte(0);
            }
        }
        catch (Throwable e) {
            try {
                this.persistence.close();
            }
            catch (Throwable e2) {
                e.addSuppressed(e2);
            }
            throw e;
        }
    }

    private long getAxisPos(int trackingHandler) {
        return (long)(HEADER.length + 4 + 4 + 4) + 37L * (long)(trackingHandler - this.startIndex);
    }

    private void validateHandler(int trackingHandler) {
        Preconditions.checkArgument((trackingHandler >= this.startIndex ? 1 : 0) != 0, (String)"The trackingHandler {} is too low for this storage (starts at {})", (int)trackingHandler, (int)this.startIndex);
        int limit = this.startIndex + this.maxStorage;
        Preconditions.checkArgument((trackingHandler <= limit ? 1 : 0) != 0, (String)"The trackingHandler {} is too high for this storage (ends at {})", (int)trackingHandler, (int)limit);
    }

    @Nullable
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public PositionTracking getPosition(int trackingHandler) throws IOException {
        this.validateHandler(trackingHandler);
        try {
            return ((Optional)this.cache.get((Object)trackingHandler, () -> this.loadPosition(trackingHandler, true))).map(PositionTracking::clone).orElse(null);
        }
        catch (ExecutionException e) {
            throw this.handleExecutionException(e);
        }
    }

    @Nullable
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public PositionTracking getPosition(int trackingHandler, boolean onlyEnabled) throws IOException {
        if (onlyEnabled) {
            return this.getPosition(trackingHandler);
        }
        this.validateHandler(trackingHandler);
        return this.loadPosition(trackingHandler, false).orElse(null);
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public OptionalInt addOrReusePosition(NamedPosition position) throws IOException {
        OptionalInt handler = this.findTrackingHandler(position);
        if (handler.isPresent()) {
            return handler;
        }
        return this.addNewPosition(position);
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized OptionalInt addNewPosition(NamedPosition position) throws IOException {
        return this.addNewPosition(position, true);
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized OptionalInt addNewPosition(NamedPosition position, boolean enabled) throws IOException {
        OptionalInt handler = this.addNewPos(position, enabled);
        if (!handler.isPresent()) {
            return handler;
        }
        if (enabled) {
            this.cache.put((Object)handler.getAsInt(), Optional.of(new PositionTracking(position)));
        }
        return handler;
    }

    @Nonnull
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public OptionalInt findTrackingHandler(NamedPosition position) throws IOException {
        OptionalInt cached = this.cache.asMap().entrySet().stream().filter(e -> ((Optional)e.getValue()).filter(position::matchesNamedPosition).isPresent()).mapToInt(Map.Entry::getKey).findFirst();
        if (cached.isPresent()) {
            return cached;
        }
        IntList handlers = this.findTrackingHandlers(position, true, 1);
        if (handlers.isEmpty()) {
            return OptionalInt.empty();
        }
        int found = handlers.getInt(0);
        this.cache.put((Object)found, Optional.of(new PositionTracking(position)));
        return OptionalInt.of(found);
    }

    private IOException handleExecutionException(ExecutionException e) {
        Throwable cause = e.getCause();
        if (cause instanceof IOException) {
            return (IOException)cause;
        }
        return new IOException(e);
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized void invalidateHandler(int trackingHandler) throws IOException {
        this.validateHandler(trackingHandler);
        this.invalidatePos(trackingHandler);
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized boolean isEnabled(int trackingHandler) throws IOException {
        this.validateHandler(trackingHandler);
        this.persistence.seek(this.getAxisPos(trackingHandler));
        return this.persistence.readBoolean();
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized boolean setEnabled(int trackingHandler, boolean enabled) throws IOException {
        this.validateHandler(trackingHandler);
        long pos = this.getAxisPos(trackingHandler);
        this.persistence.seek(pos);
        if (this.persistence.readBoolean() == enabled) {
            return false;
        }
        if (this.persistence.readLong() == 0L && enabled) {
            return false;
        }
        this.persistence.seek(pos);
        this.persistence.writeBoolean(enabled);
        this.cache.invalidate((Object)trackingHandler);
        return true;
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized boolean hasPosition(int trackingHandler) throws IOException {
        return this.hasPosition(trackingHandler, true);
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized boolean hasPosition(int trackingHandler, boolean onlyEnabled) throws IOException {
        this.validateHandler(trackingHandler);
        this.persistence.seek(this.getAxisPos(trackingHandler));
        boolean enabled = this.persistence.readBoolean();
        if (!enabled && onlyEnabled) {
            return false;
        }
        return this.persistence.readLong() != 0L;
    }

    private synchronized void invalidatePos(int trackingHandler) throws IOException {
        long pos = this.getAxisPos(trackingHandler);
        this.persistence.seek(pos);
        this.persistence.writeBoolean(false);
        byte[] buf = new byte[12];
        this.persistence.readFully(buf);
        ByteBuffer buffer = ByteBuffer.wrap(buf);
        long namePos = buffer.getLong();
        int nameLen = buffer.getInt();
        this.persistence.seek(pos + 1L);
        this.persistence.write(new byte[12]);
        this.cache.put((Object)trackingHandler, Optional.empty());
        this.addGarbage(namePos, nameLen);
    }

    private synchronized void addGarbage(long pos, int len) throws IOException {
        long garbage;
        int attempt;
        this.persistence.seek(this.garbagePos);
        int count = this.persistence.readInt();
        if (count >= 15) {
            return;
        }
        byte[] buf = new byte[12];
        ByteBuffer buffer = ByteBuffer.wrap(buf);
        if (count > 0) {
            for (attempt = 0; attempt < 15; ++attempt) {
                this.persistence.readFully(buf);
                buffer.rewind();
                garbage = buffer.getLong();
                int garbageLen = buffer.getInt();
                if (garbage == 0L) continue;
                if (garbage + (long)garbageLen == pos) {
                    this.persistence.seek(this.persistence.getFilePointer() - 4L - 8L);
                    buffer.rewind();
                    buffer.putLong(garbage).putInt(garbageLen + len);
                    this.persistence.write(buf);
                    return;
                }
                if (pos + (long)len != garbage) continue;
                this.persistence.seek(this.persistence.getFilePointer() - 4L - 8L);
                buffer.rewind();
                buffer.putLong(pos).putInt(garbageLen + len);
                this.persistence.write(buf);
                return;
            }
            this.persistence.seek(this.garbagePos + 4L);
        }
        for (attempt = 0; attempt < 15; ++attempt) {
            this.persistence.readFully(buf);
            buffer.rewind();
            garbage = buffer.getLong();
            if (garbage != 0L) continue;
            this.persistence.seek(this.persistence.getFilePointer() - 4L - 8L);
            buffer.rewind();
            buffer.putLong(pos).putInt(len);
            this.persistence.write(buf);
            this.persistence.seek(this.garbagePos);
            this.persistence.writeInt(count + 1);
            return;
        }
    }

    private synchronized long findSpaceInStringHeap(int len) throws IOException {
        this.persistence.seek(this.garbagePos);
        int remaining = this.persistence.readInt();
        if (remaining <= 0) {
            return this.persistence.length();
        }
        byte[] buf = new byte[12];
        ByteBuffer buffer = ByteBuffer.wrap(buf);
        for (int attempt = 0; attempt < 15; ++attempt) {
            this.persistence.readFully(buf);
            buffer.rewind();
            long garbage = buffer.getLong();
            int garbageLen = buffer.getInt();
            if (garbage < this.stringHeapPos || len > garbageLen) continue;
            this.persistence.seek(this.persistence.getFilePointer() - 4L - 8L);
            if (garbageLen == len) {
                this.persistence.write(new byte[12]);
                this.persistence.seek(this.garbagePos);
                this.persistence.writeInt(remaining - 1);
            } else {
                buffer.rewind();
                buffer.putLong(garbage + (long)len).putInt(garbageLen - len);
                this.persistence.write(buf);
            }
            return garbage;
        }
        return this.persistence.length();
    }

    private synchronized OptionalInt addNewPos(NamedPosition pos, boolean enabled) throws IOException {
        if (this.nextIndex - this.startIndex >= this.maxStorage) {
            return OptionalInt.empty();
        }
        int handler = this.nextIndex++;
        this.writePos(handler, pos, enabled);
        this.persistence.seek(HEADER.length + 4);
        this.persistence.writeInt(this.nextIndex);
        return OptionalInt.of(handler);
    }

    private synchronized void writePos(int trackingHandler, NamedPosition pos, boolean enabled) throws IOException {
        byte[] name = pos.getLevelName().getBytes(StandardCharsets.UTF_8);
        long namePos = this.addLevelName(name);
        this.persistence.seek(this.getAxisPos(trackingHandler));
        this.persistence.write(ByteBuffer.allocate(37).put(enabled ? (byte)1 : 0).putLong(namePos).putInt(name.length).putDouble(pos.x).putDouble(pos.y).putDouble(pos.z).array());
    }

    private synchronized long addLevelName(byte[] name) throws IOException {
        long pos = this.findSpaceInStringHeap(name.length);
        this.persistence.seek(pos);
        this.persistence.write(name);
        return pos;
    }

    @Nonnull
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized IntList findTrackingHandlers(NamedPosition pos) throws IOException {
        return this.findTrackingHandlers(pos, true);
    }

    @Nonnull
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized IntList findTrackingHandlers(NamedPosition pos, boolean onlyEnabled) throws IOException {
        return this.findTrackingHandlers(pos, onlyEnabled, Integer.MAX_VALUE);
    }

    @Nonnull
    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public synchronized IntList findTrackingHandlers(NamedPosition pos, boolean onlyEnabled, int limit) throws IOException {
        this.persistence.seek(HEADER.length + 4 + 4 + 4);
        int handler = this.startIndex - 1;
        double lookingX = pos.x;
        double lookingY = pos.y;
        double lookingZ = pos.z;
        byte[] lookingName = pos.getLevelName().getBytes(StandardCharsets.UTF_8);
        IntArrayList results = new IntArrayList(NukkitMath.clamp(limit, 1, 16));
        byte[] buf = new byte[36];
        ByteBuffer buffer = ByteBuffer.wrap(buf);
        while (++handler < this.nextIndex) {
            boolean enabled = this.persistence.readBoolean();
            if (onlyEnabled && !enabled) {
                if (this.persistence.skipBytes(36) == 36) continue;
                throw new EOFException();
            }
            this.persistence.readFully(buf);
            buffer.rewind();
            long namePos = buffer.getLong();
            int nameLen = buffer.getInt();
            double x = buffer.getDouble();
            double y = buffer.getDouble();
            double z = buffer.getDouble();
            if (namePos <= 0L || nameLen <= 0 || x != lookingX || y != lookingY || z != lookingZ) continue;
            long fp = this.persistence.getFilePointer();
            byte[] nameBytes = new byte[nameLen];
            this.persistence.seek(namePos);
            this.persistence.readFully(nameBytes);
            if (Arrays.equals(lookingName, nameBytes)) {
                results.add(handler);
                if (results.size() >= limit) {
                    return results;
                }
            }
            this.persistence.seek(fp);
        }
        return results;
    }

    private synchronized Optional<PositionTracking> loadPosition(int trackingHandler, boolean onlyEnabled) throws IOException {
        boolean enabled;
        if (trackingHandler >= this.nextIndex) {
            return Optional.empty();
        }
        this.persistence.seek(this.getAxisPos(trackingHandler));
        byte[] buf = new byte[37];
        this.persistence.readFully(buf);
        boolean bl = enabled = buf[0] == 1;
        if (!enabled && onlyEnabled) {
            return Optional.empty();
        }
        ByteBuffer buffer = ByteBuffer.wrap(buf, 1, buf.length - 1);
        long namePos = buffer.getLong();
        if (namePos == 0L) {
            return Optional.empty();
        }
        int nameLen = buffer.getInt();
        double x = buffer.getDouble();
        double y = buffer.getDouble();
        double z = buffer.getDouble();
        byte[] nameBytes = new byte[nameLen];
        this.persistence.seek(namePos);
        this.persistence.readFully(nameBytes);
        String name = new String(nameBytes, StandardCharsets.UTF_8);
        return Optional.of(new PositionTracking(name, x, y, z));
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public int getStartingHandler() {
        return this.startIndex;
    }

    @PowerNukkitOnly
    @Since(value="1.4.0.0-PN")
    public int getMaxHandler() {
        return this.startIndex + this.maxStorage - 1;
    }

    @Override
    public synchronized void close() throws IOException {
        this.persistence.close();
    }

    protected void finalize() throws Throwable {
        if (this.persistence != null) {
            this.persistence.close();
        }
    }
}

