/*! Idle Timer - v1.1.0 - 2016-03-21
* https://github.com/thorst/jquery-idletimer
* Copyright (c) 2016 Paul Irish; Licensed MIT */
/*
	mousewheel (deprecated) -> IE6.0, Chrome, Opera, Safari
	DOMMouseScroll (deprecated) -> Firefox 1.0
	wheel (standard) -> Chrome 31, Firefox 17, IE9, Firefox Mobile 17.0

	//No need to use, use DOMMouseScroll
	MozMousePixelScroll -> Firefox 3.5, Firefox Mobile 1.0

	//Events
	WheelEvent -> see wheel
	MouseWheelEvent -> see mousewheel
	MouseScrollEvent -> Firefox 3.5, Firefox Mobile 1.0
*/
(function ($) {

    $.idleTimer = function (firstParam, elem, uniqueId) {
        var opts;
        if ( typeof firstParam === "object" ) {
            opts = firstParam;
            firstParam = null;
        } else if (typeof firstParam === "number") {
            opts = { timeout: firstParam };
            firstParam = null;
        }

        // element to watch
        elem = elem || document;

        // defaults that are to be stored as instance props on the elem
        opts = $.extend({
            idle: false,                // indicates if the user is idle
            timeout: 30000,             // the amount of time (ms) before the user is considered idle
            events: "mousemove keydown wheel DOMMouseScroll mousewheel mousedown touchstart touchmove MSPointerDown MSPointerMove" // define active events
        }, opts);

        var jqElem = $(elem),
            obj = jqElem.data("idleTimerObj" + uniqueId) || {},

            /* (intentionally not documented)
             * Toggles the idle state and fires an appropriate event.
             * @return {void}
             */
            toggleIdleState = function (e) {
                var obj = $.data(elem, "idleTimerObj" + uniqueId) || {};

                // toggle the state
                obj.idle = !obj.idle;

                // store toggle state date time
                obj.olddate = +new Date();

                // create a custom event, with state and name space
                var event = $.Event((obj.idle ? "idle" : "active") + ".idleTimer" + uniqueId);

                // trigger event on object with elem and copy of obj
                $(elem).trigger(event, [elem, $.extend({}, obj), e]);
            },
            /**
             * Handle event triggers
             * @return {void}
             * @method event
             * @static
             */
            handleEvent = function (e) {
                var obj = $.data(elem, "idleTimerObj" + uniqueId) || {};

                if (e.type === "storage" && e.originalEvent.key !== obj.timerSyncId) {
                    return;
                }

                // this is already paused, ignore events for now
                if (obj.remaining != null) { return; }

                /*
                mousemove is kinda buggy, it can be triggered when it should be idle.
                Typically is happening between 115 - 150 milliseconds after idle triggered.
                @psyafter & @kaellis report "always triggered if using modal (jQuery ui, with overlay)"
                @thorst has similar issues on ios7 "after $.scrollTop() on text area"
                */
                if (e.type === "mousemove") {
                    // if coord are same, it didn't move
                    if (e.pageX === obj.pageX && e.pageY === obj.pageY) {
                        return;
                    }
                    // if coord don't exist how could it move
                    if (typeof e.pageX === "undefined" && typeof e.pageY === "undefined") {
                        return;
                    }
                    // under 200 ms is hard to do, and you would have to stop, as continuous activity will bypass this
                    var elapsed = (+new Date()) - obj.olddate;
                    if (elapsed < 200) {
                        return;
                    }
                }

                // clear any existing timeout
                clearTimeout(obj.tId);

                // if the idle timer is enabled, flip
                if (obj.idle) {
                    toggleIdleState(e);
                }

                // store when user was last active
                obj.lastActive = +new Date();

                // update mouse coord
                obj.pageX = e.pageX;
                obj.pageY = e.pageY;

                // sync lastActive
                if (e.type !== "storage" && obj.timerSyncId) {
                  if (typeof(localStorage) !== "undefined") {
                    localStorage.setItem(obj.timerSyncId, obj.lastActive);
                  }
                }

                // set a new timeout
                obj.tId = setTimeout(toggleIdleState, obj.timeout);
            },
            /**
             * Restore initial settings and restart timer
             * @return {void}
             * @method reset
             * @static
             */
            reset = function () {

                var obj = $.data(elem, "idleTimerObj" + uniqueId) || {};

                // reset settings
                obj.idle = obj.idleBackup;
                obj.olddate = +new Date();
                obj.lastActive = obj.olddate;
                obj.remaining = null;

                // reset Timers
                clearTimeout(obj.tId);
                if (!obj.idle) {
                    obj.tId = setTimeout(toggleIdleState, obj.timeout);
                }

            },
            /**
             * Store remaining time, stop timer
             * You can pause from an idle OR active state
             * @return {void}
             * @method pause
             * @static
             */
            pause = function () {

                var obj = $.data(elem, "idleTimerObj" + uniqueId) || {};

                // this is already paused
                if ( obj.remaining != null ) { return; }

                // define how much is left on the timer
                obj.remaining = obj.timeout - ((+new Date()) - obj.olddate);

                // clear any existing timeout
                clearTimeout(obj.tId);
            },
            /**
             * Start timer with remaining value
             * @return {void}
             * @method resume
             * @static
             */
            resume = function () {

                var obj = $.data(elem, "idleTimerObj" + uniqueId) || {};

                // this isn't paused yet
                if ( obj.remaining == null ) { return; }

                // start timer
                if ( !obj.idle ) {
                    obj.tId = setTimeout(toggleIdleState, obj.remaining);
                }

                // clear remaining
                obj.remaining = null;
            },
            /**
             * Stops the idle timer. This removes appropriate event handlers
             * and cancels any pending timeouts.
             * @return {void}
             * @method destroy
             * @static
             */
            destroy = function () {

                var obj = $.data(elem, "idleTimerObj" + uniqueId) || {};

                //clear any pending timeouts
                clearTimeout(obj.tId);

                //Remove data
                jqElem.removeData("idleTimerObj" + uniqueId);

                //detach the event handlers
                jqElem.off("._idleTimer" + uniqueId);
            },
            /**
            * Returns the time until becoming idle
            * @return {number}
            * @method remainingtime
            * @static
            */
            remainingtime = function () {

                var obj = $.data(elem, "idleTimerObj" + uniqueId) || {};

                //If idle there is no time remaining
                if ( obj.idle ) { return 0; }

                //If its paused just return that
                if ( obj.remaining != null ) { return obj.remaining; }

                //Determine remaining, if negative idle didn't finish flipping, just return 0
                var remaining = obj.timeout - ((+new Date()) - obj.lastActive);
                if (remaining < 0) { remaining = 0; }

                //If this is paused return that number, else return current remaining
                return remaining;
            };


        // determine which function to call
        if (firstParam === null && typeof obj.idle !== "undefined") {
            // they think they want to init, but it already is, just reset
            reset();
            return jqElem;
        } else if (firstParam === null) {
            // they want to init
        } else if (firstParam !== null && typeof obj.idle === "undefined") {
            // they want to do something, but it isnt init
            // not sure the best way to handle this
            return false;
        } else if (firstParam === "destroy") {
            destroy();
            return jqElem;
        } else if (firstParam === "pause") {
            pause();
            return jqElem;
        } else if (firstParam === "resume") {
            resume();
            return jqElem;
        } else if (firstParam === "reset") {
            reset();
            return jqElem;
        } else if (firstParam === "getRemainingTime") {
            return remainingtime();
        } else if (firstParam === "getElapsedTime") {
            return (+new Date()) - obj.olddate;
        } else if (firstParam === "getLastActiveTime") {
            return obj.lastActive;
        } else if (firstParam === "isIdle") {
            return obj.idle;
        }

        /* (intentionally not documented)
         * Handles a user event indicating that the user isn't idle. namespaced with internal idleTimer
         * @param {Event} event A DOM2-normalized event object.
         * @return {void}
         */
        jqElem.on(PrimeFaces.trim((opts.events + " ").split(" ").join("._idleTimer" + uniqueId + " ")), function (e) {
            handleEvent(e);
        });

        if (opts.timerSyncId) {
            $(window).on("storage", handleEvent);
        }

        // Internal Object Properties, This isn't all necessary, but we
        // explicitly define all keys here so we know what we are working with
        obj = $.extend({}, {
            olddate : +new Date(),          // the last time state changed
            lastActive: +new Date(),       // the last time timer was active
            idle : opts.idle,               // current state
            idleBackup : opts.idle,         // backup of idle parameter since it gets modified
            timeout : opts.timeout,         // the interval to change state
            remaining : null,               // how long until state changes
            timerSyncId : opts.timerSyncId, // localStorage key to use for syncing this timer
            tId : null,                     // the idle timer setTimeout
            pageX : null,                   // used to store the mouse coord
            pageY : null
        });

        // set a timeout to toggle state. May wish to omit this in some situations
        if (!obj.idle) {
            obj.tId = setTimeout(toggleIdleState, obj.timeout);
        }

        // store our instance on the object
        $.data(elem, "idleTimerObj" + uniqueId, obj);

        return jqElem;
    };

    // This allows binding to element
    $.fn.idleTimer = function (firstParam, uniqueId) {
        if (this[0]) {
            return $.idleTimer(firstParam, this[0], uniqueId);
        }

        return this;
    };

})(jQuery);
/**
 * __PrimeFaces IdleMonitor Widget__
 * 
 * IdleMonitor watches user actions on a page and notify callbacks in case they go idle or active again.
 * 
 * @typedef PrimeFaces.widget.IdleMonitor.OnActiveCallback Client side callback to execute when the user comes back. See
 * also {@link IdleMonitorCfg.onactive}.
 * @this {PrimeFaces.widget.IdleMonitor} PrimeFaces.widget.IdleMonitor.OnActiveCallback 
 * 
 * @typedef PrimeFaces.widget.IdleMonitor.OnIdleCallback Client side callback to execute when the user goes idle. See
 * also {@link IdleMonitorCfg.onidle}.
 * @this {PrimeFaces.widget.IdleMonitor} PrimeFaces.widget.IdleMonitor.OnIdleCallback 
 * 
 * @prop {number} timer The set-interval ID used for the timer.
 * 
 * @interface {PrimeFaces.widget.IdleMonitorCfg} cfg The configuration for the {@link  IdleMonitor| IdleMonitor widget}.
 * You can access this configuration via {@link PrimeFaces.widget.BaseWidget.cfg|BaseWidget.cfg}. Please note that this
 * configuration is usually meant to be read-only and should not be modified.
 * @extends {PrimeFaces.widget.BaseWidgetCfg} cfg
 * 
 * @prop {boolean} cfg.multiWindowSupport When set to true, the lastAccessed state will be shared between all browser
 * windows for the same servlet context.
 * @prop {PrimeFaces.widget.IdleMonitor.OnActiveCallback} cfg.onactive Client side callback to execute when the user
 * comes back.
 * @prop {PrimeFaces.widget.IdleMonitor.OnIdleCallback} cfg.onidle Client side callback to execute when the user goes
 * idle.
 * @prop {number} cfg.timeout Time to wait in milliseconds until deciding if the user is idle. Default is 5 minutes.
 */
PrimeFaces.widget.IdleMonitor = PrimeFaces.widget.BaseWidget.extend({

    /**
     * @override
     * @inheritdoc
     * @param {PrimeFaces.PartialWidgetCfg<TCfg>} cfg
     */
    init: function(cfg) {
        this._super(cfg);

        var $this = this;

        $(document).on("idle.idleTimer" + this.cfg.id, function(){

            if($this.cfg.onidle) {
                $this.cfg.onidle.call($this);
            }

            $this.callBehavior('idle');
        })
        .on("active.idleTimer" + this.cfg.id, function(){
            if($this.cfg.onactive) {
                $this.cfg.onactive.call($this);
            }

            $this.callBehavior('active');
        });

        $.idleTimer(this.cfg.timeout, document, this.cfg.id);


        if (cfg.multiWindowSupport) {
            var globalLastActiveKey = PrimeFaces.createStorageKey(this.cfg.id, 'IdleMonitor_lastActive');

            // always reset with current time on init
            localStorage.setItem(globalLastActiveKey, $(document).data('idleTimerObj' + this.cfg.id).lastActive);

            $this.timer = setInterval(function() {

                var idleTimerObj = $(document).data('idleTimerObj' + $this.cfg.id);

                var globalLastActive = parseInt(localStorage.getItem(globalLastActiveKey));
                var localLastActive = idleTimerObj.lastActive;

                // reset local state
                if (globalLastActive > localLastActive) {
                    // pause timer
                    $.idleTimer('pause', document, $this.cfg.id);

                    // overwrite real state
                    idleTimerObj.idle = false;
                    idleTimerObj.olddate = globalLastActive;
                    idleTimerObj.lastActive = globalLastActive;
                    idleTimerObj.remaining = $this.cfg.timeout;

                    // resume timer
                    $.idleTimer('resume', document, $this.cfg.id);
                }
                // update global state
                else if (localLastActive > globalLastActive) {
                    localStorage.setItem(globalLastActiveKey, localLastActive);
                }

            }, 2000);
        }
    },

    /**
     * @override
     * @inheritdoc
     */
    destroy: function() {
        this._super();

        if (this.cfg.multiWindowSupport) {
            clearInterval(this.timer);
        }
    },

    /**
     * Pauses the monitor, saving the remaining time.
     */
    pause: function() {
        $.idleTimer('pause', document, this.cfg.id);
    },

    /**
     * Resumes the monitor, with the remaining time as it was saved when the monitor was paused.
     */
    resume: function() {
        $.idleTimer('resume', document, this.cfg.id);
    },

    /**
     * Resets the monitor and restarts the timer.
     */
    reset: function() {
        $.idleTimer('reset', document, this.cfg.id);
    }

});

