/*
 * Decompiled with CFR 0.152.
 */
package org.primefaces.util;

import java.awt.Color;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Date;
import java.util.Locale;
import java.util.Objects;
import java.util.function.Supplier;
import org.apache.poi.hssf.usermodel.HSSFPalette;
import org.apache.poi.hssf.usermodel.HSSFRichTextString;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.ss.usermodel.BuiltinFormats;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.RichTextString;
import org.apache.poi.ss.usermodel.VerticalAlignment;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellUtil;
import org.primefaces.component.api.UIColumn;
import org.primefaces.component.export.ColumnValue;
import org.primefaces.component.export.ExcelOptions;
import org.primefaces.util.BigDecimalValidator;
import org.primefaces.util.CalendarUtils;
import org.primefaces.util.CurrencyValidator;
import org.primefaces.util.ExcelXmlStylesManager;
import org.primefaces.util.LangUtils;

public class ExcelStylesManager {
    protected static final String DEFAULT_FONT = "Arial";
    private static final short DEFAULT_DATE_EXCEL_FORMAT = 14;
    private static final short DEFAULT_TIME_EXCEL_FORMAT = 20;
    private static final short DEFAULT_DATETIME_EXCEL_FORMAT = 22;
    protected final Workbook wb;
    protected final ExcelOptions options;
    private final Locale locale;
    private final boolean stronglyTypedCells;
    private final DecimalFormat numberFormat;
    private final DecimalFormat currencyFormat;
    private final Styles facetStyles;
    private final Styles cellStyles;

    protected ExcelStylesManager(Workbook wb, Locale locale, ExcelOptions options) {
        this.wb = wb;
        this.locale = locale;
        this.options = options;
        this.stronglyTypedCells = options == null || options.isStronglyTypedCells();
        this.numberFormat = this.getNumberFormat();
        this.currencyFormat = this.getCurrencyFormat();
        this.facetStyles = new Styles(this::createDefaultFacetStyle);
        this.cellStyles = new Styles(this::createDefaultCellStyle);
    }

    public static ExcelStylesManager createExcelStylesManager(Workbook wb, Locale locale, ExcelOptions options) {
        if (wb instanceof HSSFWorkbook) {
            return new ExcelStylesManager(wb, locale, options);
        }
        return new ExcelXmlStylesManager(wb, locale, options);
    }

    public void updateFacetCell(Cell cell, ColumnValue value) {
        if (value.isStringValue()) {
            this.updateCellStringValue(cell, Objects.toString(value, ""), this.facetStyles);
        } else {
            this.updateCellObjectValue(cell, value.getValue(), this.facetStyles);
        }
    }

    public void updateCell(UIColumn column, Cell cell, ColumnValue value) {
        if (value.isStringValue()) {
            this.updateCellStringValue(cell, Objects.toString(value, ""), this.cellStyles);
        } else {
            this.updateCellObjectValue(cell, value.getValue(), this.cellStyles);
        }
        this.applyColumnAlignments(column, cell);
    }

    private void updateCellObjectValue(Cell cell, Object value, Styles styles) {
        if (value instanceof BigDecimal) {
            this.setBigDecimalValue(cell, this.numberFormat.format(value), (BigDecimal)value, styles);
        } else if (value instanceof Number) {
            this.setNumberValue(cell, (Number)value, styles);
        } else if (value instanceof LocalDate) {
            this.setLocalDateValue(cell, (LocalDate)value, styles);
        } else if (value instanceof LocalDateTime) {
            this.setLocalDateTimeValue(cell, (LocalDateTime)value, styles);
        } else if (value instanceof LocalTime) {
            this.setLocalTimeValue(cell, (LocalTime)value, styles);
        } else if (value instanceof Date) {
            this.setDateValue(cell, (Date)value, styles);
        } else {
            cell.setCellStyle(styles.getDefaultStyle());
            cell.setCellValue(this.createRichTextString(Objects.toString(value, "")));
        }
    }

    private void updateCellStringValue(Cell cell, String value, Styles styles) {
        boolean printed = false;
        if (this.stronglyTypedCells && !(printed = this.setNumberValueIfAppropiate(cell, value, styles))) {
            printed = this.setCurrencyValueIfAppropiate(cell, value, styles);
        }
        if (!printed) {
            cell.setCellStyle(styles.getDefaultStyle());
            cell.setCellValue(this.createRichTextString(value));
        }
    }

    private boolean setNumberValueIfAppropiate(Cell cell, String value, Styles styles) {
        BigDecimal bigDecimal = BigDecimalValidator.getInstance().validate(value, this.numberFormat);
        if (bigDecimal != null) {
            this.setBigDecimalValue(cell, value, bigDecimal, styles);
            return true;
        }
        if (LangUtils.isNumeric(value)) {
            double number = Double.parseDouble(value);
            this.setNumberValue(cell, number, styles);
            return true;
        }
        return false;
    }

    private void setBigDecimalValue(Cell cell, String value, BigDecimal bigDecimal, Styles styles) {
        CellStyle style;
        boolean withoutThousandSeparator;
        cell.setCellValue(bigDecimal.doubleValue());
        boolean bl = withoutThousandSeparator = value.indexOf(this.numberFormat.getDecimalFormatSymbols().getGroupingSeparator()) == -1;
        if (withoutThousandSeparator) {
            style = styles.getGeneralNumberStyle();
        } else {
            boolean hasDecimals = bigDecimal.stripTrailingZeros().scale() > 0;
            style = hasDecimals ? styles.getDecimalStyle() : styles.getIntegerStyle();
        }
        cell.setCellStyle(style);
    }

    private void setNumberValue(Cell cell, Number number, Styles styles) {
        cell.setCellValue(number.doubleValue());
        cell.setCellStyle(styles.getGeneralNumberStyle());
    }

    private void setLocalDateValue(Cell cell, LocalDate date, Styles styles) {
        cell.setCellValue(date);
        cell.setCellStyle(styles.getDateStyle());
    }

    private void setLocalDateTimeValue(Cell cell, LocalDateTime dateTime, Styles styles) {
        cell.setCellValue(dateTime);
        cell.setCellStyle(styles.getDateTimeStyle());
    }

    private void setLocalTimeValue(Cell cell, LocalTime localTime, Styles styles) {
        cell.setCellValue(CalendarUtils.convertLocalTime2Date(localTime));
        cell.setCellStyle(styles.getTimeStyle());
    }

    private void setDateValue(Cell cell, Date date, Styles styles) {
        cell.setCellValue(date);
        cell.setCellStyle(styles.getDateStyle());
    }

    private boolean setCurrencyValueIfAppropiate(Cell cell, String value, Styles styles) {
        BigDecimal currency = CurrencyValidator.getInstance().validate(value, this.currencyFormat);
        if (currency == null) {
            return false;
        }
        cell.setCellValue(((Number)currency).doubleValue());
        cell.setCellStyle(styles.getCurrencyStyle());
        return true;
    }

    private void applyColumnAlignments(UIColumn column, Cell cell) {
        String[] styles = new String[]{column.getStyle(), column.getStyleClass()};
        if (LangUtils.containsIgnoreCase(styles, "right")) {
            CellUtil.setAlignment((Cell)cell, (HorizontalAlignment)HorizontalAlignment.RIGHT);
        } else if (LangUtils.containsIgnoreCase(styles, "center")) {
            CellUtil.setAlignment((Cell)cell, (HorizontalAlignment)HorizontalAlignment.CENTER);
        } else if (LangUtils.containsIgnoreCase(styles, "left")) {
            CellUtil.setAlignment((Cell)cell, (HorizontalAlignment)HorizontalAlignment.LEFT);
        }
    }

    private CellStyle createDefaultFacetStyle() {
        CellStyle style = this.wb.createCellStyle();
        style.setFont(this.createFont());
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        style.setWrapText(true);
        this.applyFacetOptions(style);
        return style;
    }

    private CellStyle createDefaultCellStyle() {
        CellStyle style = this.wb.createCellStyle();
        style.setFont(this.createFont());
        this.applyCellOptions(style);
        return style;
    }

    protected void applyFacetOptions(CellStyle style) {
        if (this.options != null) {
            this.applyHssfFacetOptions(style);
        }
    }

    protected void applyCellOptions(CellStyle style) {
        if (this.options != null) {
            this.applyHssfCellOptions(style);
        }
    }

    private void applyHssfFacetOptions(CellStyle style) {
        String facetFontSize;
        String facetFontColor;
        Color color;
        Font facetFont = this.wb.getFontAt(style.getFontIndex());
        String facetFontStyle = this.options.getFacetFontStyle();
        if (facetFontStyle != null) {
            if ("BOLD".equalsIgnoreCase(facetFontStyle)) {
                facetFont.setBold(true);
            }
            if ("ITALIC".equalsIgnoreCase(facetFontStyle)) {
                facetFont.setItalic(true);
            }
        }
        HSSFPalette palette = ((HSSFWorkbook)this.wb).getCustomPalette();
        String facetBackground = this.options.getFacetBgColor();
        if (facetBackground != null) {
            color = Color.decode(facetBackground);
            HSSFColor backgroundColor = palette.findSimilarColor(color.getRed(), color.getGreen(), color.getBlue());
            style.setFillForegroundColor(backgroundColor.getIndex());
            style.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        }
        if ((facetFontColor = this.options.getFacetFontColor()) != null) {
            color = Color.decode(facetFontColor);
            HSSFColor facetColor = palette.findSimilarColor(color.getRed(), color.getGreen(), color.getBlue());
            facetFont.setColor(facetColor.getIndex());
        }
        if ((facetFontSize = this.options.getFacetFontSize()) != null) {
            facetFont.setFontHeightInPoints(Short.parseShort(facetFontSize));
        }
    }

    private void applyHssfCellOptions(CellStyle style) {
        String cellFontStyle;
        String cellFontSize;
        Font cellFont = this.wb.getFontAt(style.getFontIndex());
        String cellFontColor = this.options.getCellFontColor();
        if (cellFontColor != null) {
            HSSFPalette palette = ((HSSFWorkbook)this.wb).getCustomPalette();
            Color color = Color.decode(cellFontColor);
            HSSFColor cellColor = palette.findSimilarColor(color.getRed(), color.getGreen(), color.getBlue());
            cellFont.setColor(cellColor.getIndex());
        }
        if ((cellFontSize = this.options.getCellFontSize()) != null) {
            cellFont.setFontHeightInPoints(Short.parseShort(cellFontSize));
        }
        if ((cellFontStyle = this.options.getCellFontStyle()) != null) {
            if ("BOLD".equalsIgnoreCase(cellFontStyle)) {
                cellFont.setBold(true);
            }
            if ("ITALIC".equalsIgnoreCase(cellFontStyle)) {
                cellFont.setItalic(true);
            }
        }
    }

    private Font createFont() {
        Font font = this.wb.createFont();
        if (this.options != null) {
            String fontName = LangUtils.isBlank(this.options.getFontName()) ? DEFAULT_FONT : this.options.getFontName();
            font.setFontName(fontName);
        } else {
            font.setFontName(DEFAULT_FONT);
        }
        return font;
    }

    public RichTextString createRichTextString(String value) {
        return new HSSFRichTextString(value);
    }

    protected String getCurrencyExcelFormat() {
        String pattern = this.currencyFormat.toLocalizedPattern();
        String[] patterns = pattern.split(";");
        return this.toExcelPattern(patterns[0], this.currencyFormat.getDecimalFormatSymbols());
    }

    protected String getFormattedIntegerExcelFormat() {
        if (this.options == null || this.options.getNumberFormat() == null) {
            return BuiltinFormats.getBuiltinFormat((int)3);
        }
        return this.toExcelPattern(this.numberFormat.toLocalizedPattern(), this.numberFormat.getDecimalFormatSymbols());
    }

    protected String getFormattedDecimalExcelFormat() {
        if (this.options == null || this.options.getNumberFormat() == null) {
            return "#,##0.###";
        }
        return this.toExcelPattern(this.numberFormat.toLocalizedPattern(), this.numberFormat.getDecimalFormatSymbols());
    }

    private String toExcelPattern(String pattern, DecimalFormatSymbols symbols) {
        StringBuilder buffer = new StringBuilder(pattern.length());
        for (int i = 0; i < pattern.length(); ++i) {
            char character = pattern.charAt(i);
            if (character == symbols.getDecimalSeparator()) {
                buffer.append(".");
                continue;
            }
            if (character == symbols.getGroupingSeparator()) {
                buffer.append(",");
                continue;
            }
            if (character == symbols.getPatternSeparator()) {
                buffer.append(";");
                continue;
            }
            if (character == '\u00a4') {
                buffer.append("\"").append(symbols.getCurrencySymbol()).append("\"");
                continue;
            }
            buffer.append(character);
        }
        return buffer.toString();
    }

    private DecimalFormat getNumberFormat() {
        if (this.options != null && this.options.getNumberFormat() != null) {
            return this.options.getNumberFormat();
        }
        return (DecimalFormat)DecimalFormat.getInstance(this.locale);
    }

    private DecimalFormat getCurrencyFormat() {
        if (this.options != null && this.options.getCurrencyFormat() != null) {
            return this.options.getCurrencyFormat();
        }
        return (DecimalFormat)DecimalFormat.getCurrencyInstance(this.locale);
    }

    private class Styles {
        private final Supplier<CellStyle> defaultStyleSupplier;
        private CellStyle defaultStyle;
        private CellStyle generalNumberStyle;
        private CellStyle decimalStyle;
        private CellStyle integerStyle;
        private CellStyle currencyStyle;
        private CellStyle dateStyle;
        private CellStyle timeStyle;
        private CellStyle dateTimeStyle;

        public Styles(Supplier<CellStyle> defaultStyleSupplier) {
            this.defaultStyleSupplier = defaultStyleSupplier;
        }

        public CellStyle getDefaultStyle() {
            if (this.defaultStyle == null) {
                this.defaultStyle = this.defaultStyleSupplier.get();
            }
            return this.defaultStyle;
        }

        public CellStyle getGeneralNumberStyle() {
            if (this.generalNumberStyle == null) {
                this.generalNumberStyle = this.createGeneralNumberStyle();
            }
            return this.generalNumberStyle;
        }

        private CellStyle getIntegerStyle() {
            if (this.integerStyle == null) {
                this.integerStyle = this.createIntegerStyle();
            }
            return this.integerStyle;
        }

        private CellStyle getDecimalStyle() {
            if (this.decimalStyle == null) {
                this.decimalStyle = this.createDecimalStyle();
            }
            return this.decimalStyle;
        }

        private CellStyle getCurrencyStyle() {
            if (this.currencyStyle == null) {
                this.currencyStyle = this.createCurrencyStyle();
            }
            return this.currencyStyle;
        }

        private CellStyle getDateStyle() {
            if (this.dateStyle == null) {
                this.dateStyle = this.createDateStyle();
            }
            return this.dateStyle;
        }

        private CellStyle getDateTimeStyle() {
            if (this.dateTimeStyle == null) {
                this.dateTimeStyle = this.createDateTimeStyle();
            }
            return this.dateTimeStyle;
        }

        private CellStyle getTimeStyle() {
            if (this.timeStyle == null) {
                this.timeStyle = this.createTimeStyle();
            }
            return this.timeStyle;
        }

        private CellStyle createDateTimeStyle() {
            short dataFormat;
            CellStyle style = this.defaultStyleSupplier.get();
            if (ExcelStylesManager.this.options.getExcelDateTimeFormat() != null) {
                String format = ExcelStylesManager.this.options.getExcelDateTimeFormat();
                dataFormat = ExcelStylesManager.this.wb.getCreationHelper().createDataFormat().getFormat(format);
            } else {
                dataFormat = 22;
            }
            style.setDataFormat(dataFormat);
            style.setAlignment(HorizontalAlignment.RIGHT);
            return style;
        }

        private CellStyle createTimeStyle() {
            short dataFormat;
            CellStyle style = this.defaultStyleSupplier.get();
            if (ExcelStylesManager.this.options.getExcelTimeFormat() != null) {
                String format = ExcelStylesManager.this.options.getExcelTimeFormat();
                dataFormat = ExcelStylesManager.this.wb.getCreationHelper().createDataFormat().getFormat(format);
            } else {
                dataFormat = 20;
            }
            style.setDataFormat(dataFormat);
            style.setAlignment(HorizontalAlignment.RIGHT);
            return style;
        }

        private CellStyle createGeneralNumberStyle() {
            CellStyle style = this.defaultStyleSupplier.get();
            style.setAlignment(HorizontalAlignment.RIGHT);
            return style;
        }

        private CellStyle createIntegerStyle() {
            CellStyle style = this.createGeneralNumberStyle();
            String format = ExcelStylesManager.this.getFormattedIntegerExcelFormat();
            short dataFormat = ExcelStylesManager.this.wb.getCreationHelper().createDataFormat().getFormat(format);
            style.setDataFormat(dataFormat);
            return style;
        }

        private CellStyle createDecimalStyle() {
            CellStyle style = this.defaultStyleSupplier.get();
            String format = ExcelStylesManager.this.getFormattedDecimalExcelFormat();
            short dataFormat = ExcelStylesManager.this.wb.getCreationHelper().createDataFormat().getFormat(format);
            style.setDataFormat(dataFormat);
            return style;
        }

        private CellStyle createCurrencyStyle() {
            CellStyle style = this.defaultStyleSupplier.get();
            String format = ExcelStylesManager.this.getCurrencyExcelFormat();
            short dataFormat = ExcelStylesManager.this.wb.getCreationHelper().createDataFormat().getFormat(format);
            style.setDataFormat(dataFormat);
            style.setAlignment(HorizontalAlignment.RIGHT);
            return style;
        }

        private CellStyle createDateStyle() {
            short dataFormat;
            CellStyle style = this.defaultStyleSupplier.get();
            if (ExcelStylesManager.this.options.getExcelDateFormat() != null) {
                String format = ExcelStylesManager.this.options.getExcelDateFormat();
                dataFormat = ExcelStylesManager.this.wb.getCreationHelper().createDataFormat().getFormat(format);
            } else {
                dataFormat = 14;
            }
            style.setDataFormat(dataFormat);
            style.setAlignment(HorizontalAlignment.RIGHT);
            return style;
        }
    }
}

