/*
 * Decompiled with CFR 0.152.
 */
package org.sonarsource.scanner.lib.internal;

import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;
import com.google.gson.reflect.TypeToken;
import java.io.BufferedReader;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.Type;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import javax.annotation.Nullable;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonarsource.scanner.lib.System2;
import org.sonarsource.scanner.lib.Utils;
import org.sonarsource.scanner.lib.internal.JavaRunner;
import org.sonarsource.scanner.lib.internal.JreCacheHit;
import org.sonarsource.scanner.lib.internal.ProcessWrapperFactory;
import org.sonarsource.scanner.lib.internal.ResourceMetadata;
import org.sonarsource.scanner.lib.internal.cache.CachedFile;
import org.sonarsource.scanner.lib.internal.cache.FileCache;
import org.sonarsource.scanner.lib.internal.cache.HashMismatchException;
import org.sonarsource.scanner.lib.internal.http.ScannerHttpClient;
import org.sonarsource.scanner.lib.internal.util.CompressionUtils;

public class JavaRunnerFactory {
    private static final Logger LOG = LoggerFactory.getLogger(JavaRunnerFactory.class);
    static final String API_PATH_JRE = "/analysis/jres";
    private static final String EXTENSION_ZIP = "zip";
    private static final String EXTENSION_GZ = "gz";
    private final System2 system;
    private final ProcessWrapperFactory processWrapperFactory;

    public JavaRunnerFactory(System2 system, ProcessWrapperFactory processWrapperFactory) {
        this.system = system;
        this.processWrapperFactory = processWrapperFactory;
    }

    public JavaRunner createRunner(ScannerHttpClient scannerHttpClient, FileCache fileCache, Map<String, String> properties) {
        Path javaExecutable;
        String javaExecutablePropValue = properties.get("sonar.scanner.javaExePath");
        if (javaExecutablePropValue != null) {
            LOG.info("Using the configured java executable '{}'", (Object)javaExecutablePropValue);
            return new JavaRunner(Paths.get(javaExecutablePropValue, new String[0]), JreCacheHit.DISABLED);
        }
        boolean skipJreProvisioning = Boolean.parseBoolean(properties.get("sonar.scanner.skipJreProvisioning"));
        if (skipJreProvisioning) {
            LOG.info("JRE provisioning is disabled");
        } else {
            Optional<CachedFile> cachedFile = JavaRunnerFactory.getJreFromServer(scannerHttpClient, fileCache, properties, true);
            if (cachedFile.isPresent()) {
                return new JavaRunner(cachedFile.get().getPathInCache(), cachedFile.get().isCacheHit() ? JreCacheHit.HIT : JreCacheHit.MISS);
            }
        }
        String javaHome = this.system.getEnvironmentVariable("JAVA_HOME");
        String javaExe = "java" + (this.isOsWindows() ? ".exe" : "");
        if (javaHome != null && Files.exists(javaExecutable = Paths.get(javaHome, "bin", javaExe), new LinkOption[0])) {
            LOG.info("Using the java executable '{}' from JAVA_HOME", (Object)javaExecutable);
            return new JavaRunner(javaExecutable, JreCacheHit.DISABLED);
        }
        LOG.info("The java executable in the PATH will be used");
        return new JavaRunner(this.isOsWindows() ? this.findJavaInPath(javaExe) : Paths.get(javaExe, new String[0]), JreCacheHit.DISABLED);
    }

    private boolean isOsWindows() {
        String osName = this.system.getProperty("os.name");
        return osName != null && osName.startsWith("Windows");
    }

    private Path findJavaInPath(String javaExe) {
        try {
            Path javaExecutable;
            ProcessWrapperFactory.ProcessWrapper process = this.processWrapperFactory.create("C:\\Windows\\System32\\where.exe", "$PATH:" + javaExe);
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream(), StandardCharsets.UTF_8));){
                javaExecutable = Paths.get(reader.lines().findFirst().orElseThrow(), new String[0]);
                LOG.debug("Found java executable in PATH at '{}'", (Object)javaExecutable.toAbsolutePath());
            }
            int exit = process.waitFor();
            if (exit != 0) {
                throw new IllegalStateException(String.format("Command execution exited with code: %d", exit));
            }
            return javaExecutable;
        }
        catch (Exception e) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Cannot find java executable in PATH", e);
        }
    }

    private static Optional<CachedFile> getJreFromServer(ScannerHttpClient scannerHttpClient, FileCache fileCache, Map<String, String> properties, boolean retry) {
        String os = properties.get("sonar.scanner.os");
        String arch = properties.get("sonar.scanner.arch");
        LOG.info("JRE provisioning: os[{}], arch[{}]", (Object)os, (Object)arch);
        try {
            Optional<JreMetadata> jreMetadata = JavaRunnerFactory.getJreMetadata(scannerHttpClient, os, arch);
            if (jreMetadata.isEmpty()) {
                LOG.info("No JRE found for this OS/architecture");
                return Optional.empty();
            }
            CachedFile cachedFile = fileCache.getOrDownload(jreMetadata.get().getFilename(), jreMetadata.get().getSha256(), "SHA-256", new JreDownloader(scannerHttpClient, jreMetadata.get()));
            Path extractedDirectory = JavaRunnerFactory.extractArchive(cachedFile.getPathInCache());
            return Optional.of(new CachedFile(extractedDirectory.resolve(jreMetadata.get().javaPath), cachedFile.isCacheHit()));
        }
        catch (HashMismatchException e) {
            if (retry) {
                LOG.warn("Failed to get the JRE, retrying...");
                return JavaRunnerFactory.getJreFromServer(scannerHttpClient, fileCache, properties, false);
            }
            throw e;
        }
    }

    private static Optional<JreMetadata> getJreMetadata(ScannerHttpClient scannerHttpClient, String os, String arch) {
        try {
            String response = scannerHttpClient.callRestApi(String.format("/analysis/jres?os=%s&arch=%s", os, arch));
            Type listType = new TypeToken<ArrayList<JreMetadata>>(){}.getType();
            List jres = (List)new Gson().fromJson(response, listType);
            return jres.stream().findFirst();
        }
        catch (IOException e) {
            throw new IllegalStateException("Failed to get JRE metadata", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static Path extractArchive(Path cachedFile) {
        String filename = cachedFile.getFileName().toString();
        Path destDir = cachedFile.getParent().resolve(filename + "_extracted");
        Path lockFile = cachedFile.getParent().resolve(filename + "_extracted.lock");
        if (!Files.exists(destDir, new LinkOption[0])) {
            try (FileOutputStream out = new FileOutputStream(lockFile.toFile());){
                FileLock lock = JavaRunnerFactory.createLockWithRetries(out.getChannel());
                try {
                    if (!Files.exists(destDir, new LinkOption[0])) {
                        Path tempDir = Files.createTempDirectory(cachedFile.getParent(), "jre", new FileAttribute[0]);
                        JavaRunnerFactory.extract(cachedFile, tempDir);
                        Files.move(tempDir, destDir, new CopyOption[0]);
                    }
                }
                finally {
                    lock.release();
                }
            }
            catch (IOException e) {
                throw new IllegalStateException("Failed to extract archive", e);
            }
            finally {
                Utils.deleteQuietly(lockFile);
            }
        }
        return destDir;
    }

    private static FileLock createLockWithRetries(FileChannel channel) throws IOException {
        int tryCount = 0;
        while (tryCount < 10) {
            ++tryCount;
            try {
                return channel.lock();
            }
            catch (OverlappingFileLockException overlappingFileLockException) {
                try {
                    Thread.sleep(200L * (long)tryCount);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        }
        throw new IOException("Unable to get lock after " + tryCount + " tries");
    }

    private static void extract(Path compressedFile, Path targetDir) throws IOException {
        String extension;
        String filename = compressedFile.getFileName().toString();
        switch (extension = filename.substring(filename.lastIndexOf(46) + 1)) {
            case "zip": {
                CompressionUtils.unzip(compressedFile, targetDir);
                break;
            }
            case "gz": {
                CompressionUtils.extractTarGz(compressedFile, targetDir);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported compressed archive extension: " + extension);
            }
        }
    }

    static class JreMetadata
    extends ResourceMetadata {
        @SerializedName(value="id")
        private final String id;
        @SerializedName(value="javaPath")
        private final String javaPath;

        JreMetadata(String filename, String sha256, @Nullable String downloadUrl, String id, String javaPath) {
            super(filename, sha256, downloadUrl);
            this.id = id;
            this.javaPath = javaPath;
        }
    }

    static class JreDownloader
    implements FileCache.Downloader {
        private final ScannerHttpClient connection;
        private final JreMetadata jreMetadata;

        JreDownloader(ScannerHttpClient connection, JreMetadata jreMetadata) {
            this.connection = connection;
            this.jreMetadata = jreMetadata;
        }

        @Override
        public void download(String filename, Path toFile) throws IOException {
            if (StringUtils.isNotBlank((CharSequence)this.jreMetadata.getDownloadUrl())) {
                this.connection.downloadFromExternalUrl(this.jreMetadata.getDownloadUrl(), toFile);
            } else {
                this.connection.downloadFromRestApi("/analysis/jres/" + this.jreMetadata.id, toFile);
            }
        }
    }
}

