/*
 * Copyright 2013-2019 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cloud.bus;

import org.springframework.boot.actuate.autoconfigure.endpoint.condition.ConditionalOnEnabledEndpoint;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.bus.endpoint.RefreshBusEndpoint;
import org.springframework.cloud.bus.event.RefreshListener;
import org.springframework.cloud.context.refresh.ContextRefresher;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * @author Ryan Baxter
 */
@Configuration(proxyBeanMethods = false)
@ConditionalOnBusEnabled
@AutoConfigureAfter(
		name = { "org.springframework.cloud.autoconfigure.RefreshAutoConfiguration" })
public class BusRefreshAutoConfiguration {

	@Bean
	@ConditionalOnProperty(value = "spring.cloud.bus.refresh.enabled",
			matchIfMissing = true)
	@ConditionalOnBean(ContextRefresher.class)
	public RefreshListener refreshListener(ContextRefresher contextRefresher) {
		return new RefreshListener(contextRefresher);
	}

	@Configuration(proxyBeanMethods = false)
	@ConditionalOnBean(ContextRefresher.class)
	@ConditionalOnClass(
			name = { "org.springframework.boot.actuate.endpoint.annotation.Endpoint",
					"org.springframework.cloud.context.scope.refresh.RefreshScope" })
	protected static class BusRefreshEndpointConfiguration {

		@Bean
		@ConditionalOnEnabledEndpoint
		public RefreshBusEndpoint refreshBusEndpoint(ApplicationContext context,
				BusProperties bus) {
			return new RefreshBusEndpoint(context, bus.getId());
		}

	}

}
