/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package shaded.org.eclipse.aether.transport.file;

import java.io.File;
import java.nio.file.Files;

import org.eclipse.aether.repository.RemoteRepository;
import shaded.org.eclipse.aether.spi.connector.transport.AbstractTransporter;
import shaded.org.eclipse.aether.spi.connector.transport.GetTask;
import shaded.org.eclipse.aether.spi.connector.transport.PeekTask;
import shaded.org.eclipse.aether.spi.connector.transport.PutTask;
import shaded.org.eclipse.aether.spi.connector.transport.TransportTask;
import org.eclipse.aether.transfer.NoTransporterException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A transporter using {@link java.io.File}.
 */
final class FileTransporter extends AbstractTransporter {

    private static final Logger LOGGER = LoggerFactory.getLogger(FileTransporter.class);

    private final File basedir;

    FileTransporter(RemoteRepository repository) throws NoTransporterException {
        if (!"file".equalsIgnoreCase(repository.getProtocol())) {
            throw new NoTransporterException(repository);
        }
        basedir = new File(PathUtils.basedir(repository.getUrl())).getAbsoluteFile();
    }

    File getBasedir() {
        return basedir;
    }

    public int classify(Throwable error) {
        if (error instanceof ResourceNotFoundException) {
            return ERROR_NOT_FOUND;
        }
        return ERROR_OTHER;
    }

    @Override
    protected void implPeek(PeekTask task) throws Exception {
        getFile(task, true);
    }

    @Override
    protected void implGet(GetTask task) throws Exception {
        File file = getFile(task, true);
        utilGet(task, Files.newInputStream(file.toPath()), true, file.length(), false);
    }

    @Override
    protected void implPut(PutTask task) throws Exception {
        File file = getFile(task, false);
        file.getParentFile().mkdirs();
        try {
            utilPut(task, Files.newOutputStream(file.toPath()), true);
        } catch (Exception e) {
            if (!file.delete() && file.exists()) {
                LOGGER.debug("Could not delete partial file {}", file);
            }
            throw e;
        }
    }

    private File getFile(TransportTask task, boolean required) throws Exception {
        String path = task.getLocation().getPath();
        if (path.contains("../")) {
            throw new IllegalArgumentException("illegal resource path: " + path);
        }
        File file = new File(basedir, path);
        if (required && !file.exists()) {
            throw new ResourceNotFoundException("Could not locate " + file);
        }
        return file;
    }

    @Override
    protected void implClose() {}
}
