/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.cloud.gateway.filter.factory.cache;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.springframework.cloud.gateway.filter.factory.cache.keygenerator.CacheKeyGenerator;
import org.springframework.http.HttpCookie;
import org.springframework.http.HttpHeaders;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.mock.http.server.reactive.MockServerHttpRequest;
import org.springframework.util.MultiValueMap;

class CacheKeyGeneratorTest {
    final CacheKeyGenerator cacheKeyGenerator = new CacheKeyGenerator();

    CacheKeyGeneratorTest() {
    }

    @Test
    public void shouldGenerateSameKeyForSameUri() {
        MockServerHttpRequest request1 = MockServerHttpRequest.get((String)"http://this", (Object[])new Object[0]).build();
        MockServerHttpRequest request2 = MockServerHttpRequest.get((String)"http://this", (Object[])new Object[0]).build();
        String key1 = this.cacheKeyGenerator.generateKey((ServerHttpRequest)request1, new String[0]);
        String key2 = this.cacheKeyGenerator.generateKey((ServerHttpRequest)request2, new String[0]);
        Assertions.assertThat((String)key1).isEqualTo(key2);
    }

    @Test
    void shouldGenerateDifferentKeyWhenAuthAreDifferent() {
        String uri = "https://this";
        MockServerHttpRequest requestWithoutAuth = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).build();
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.put("Authorization", List.of("my-token"));
        MockServerHttpRequest requestWithAuth = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders).build();
        String keyWithoutAuth = this.cacheKeyGenerator.generateKey((ServerHttpRequest)requestWithoutAuth, new String[0]);
        String keyWithAuth = this.cacheKeyGenerator.generateKey((ServerHttpRequest)requestWithAuth, new String[0]);
        Assertions.assertThat((String)keyWithAuth).isNotEqualTo((Object)keyWithoutAuth);
    }

    @Test
    void shouldGenerateDifferentKeyWhenCookiesAreDifferent() {
        HttpHeaders httpHeaders = new HttpHeaders();
        String uri = "https://this";
        MockServerHttpRequest requestWithoutCookies = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders).build();
        HttpCookie[] cookies = new HttpCookie[]{new HttpCookie("user", "my-first-cookie")};
        MockServerHttpRequest requestWithCookies = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders).cookie(cookies).build();
        String keyWithoutCookies = this.cacheKeyGenerator.generateKey((ServerHttpRequest)requestWithoutCookies, new String[0]);
        String keyWithCookies = this.cacheKeyGenerator.generateKey((ServerHttpRequest)requestWithCookies, new String[0]);
        Assertions.assertThat((String)keyWithoutCookies).isNotEqualTo((Object)keyWithCookies);
    }

    @Test
    void shouldGenerateSameKeyWhenSameAuthAndCookieArePresent() {
        String uri = "https://this";
        HttpCookie[] cookies = new HttpCookie[]{new HttpCookie("user", "my-first-cookie")};
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.put("Authorization", List.of("my-token"));
        MockServerHttpRequest request1 = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders).cookie(cookies).build();
        MockServerHttpRequest request2 = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders).cookie(cookies).build();
        String key1 = this.cacheKeyGenerator.generateKey((ServerHttpRequest)request1, new String[0]);
        String key2 = this.cacheKeyGenerator.generateKey((ServerHttpRequest)request2, new String[0]);
        Assertions.assertThat((String)key1).isEqualTo(key2);
    }

    @Test
    void shouldGenerateSameKeyWhenVaryHeadersAreEqual() {
        String varyHeader = "X-MY-VARY";
        String uri = "https://this";
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.put("X-MY-VARY", List.of("VALUE1"));
        MockServerHttpRequest withFirstVary = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders).build();
        HttpHeaders httpHeaders2 = new HttpHeaders();
        httpHeaders2.put("X-MY-VARY", List.of("VALUE1"));
        MockServerHttpRequest withSecondVary = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders2).build();
        String keyWithFirstVary = this.cacheKeyGenerator.generateKey((ServerHttpRequest)withFirstVary, new String[]{"X-MY-VARY"});
        String keyWithSecondVary = this.cacheKeyGenerator.generateKey((ServerHttpRequest)withSecondVary, new String[]{"X-MY-VARY"});
        Assertions.assertThat((String)keyWithFirstVary).isEqualTo(keyWithSecondVary);
    }

    @Test
    void shouldGenerateDifferentKeyWhenVaryHeadersAreDifferent() {
        String varyHeader = "X-MY-VARY";
        String uri = "https://this";
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.put("X-MY-VARY", List.of("VALUE1"));
        MockServerHttpRequest withFirstVary = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders).build();
        HttpHeaders httpHeaders2 = new HttpHeaders();
        httpHeaders.put("X-MY-VARY", List.of("VALUE2"));
        MockServerHttpRequest withSecondVary = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders2).build();
        String keyWithFirstVary = this.cacheKeyGenerator.generateKey((ServerHttpRequest)withFirstVary, new String[]{"X-MY-VARY"});
        String keyWithSecondVary = this.cacheKeyGenerator.generateKey((ServerHttpRequest)withSecondVary, new String[]{"X-MY-VARY"});
        Assertions.assertThat((String)keyWithFirstVary).isNotEqualTo((Object)keyWithSecondVary);
    }

    @Test
    void shouldGenerateDifferentKeyWhenVaryHeaderIsMissingInSecondRequest() {
        String varyHeader = "X-MY-VARY";
        String uri = "https://this";
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.put("X-MY-VARY", List.of("VALUE1"));
        MockServerHttpRequest withFirstVary = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders).build();
        MockServerHttpRequest withSecondVary = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).build();
        String keyWithFirstVary = this.cacheKeyGenerator.generateKey((ServerHttpRequest)withFirstVary, new String[]{"X-MY-VARY"});
        String keyWithSecondVary = this.cacheKeyGenerator.generateKey((ServerHttpRequest)withSecondVary, new String[]{"X-MY-VARY"});
        Assertions.assertThat((String)keyWithFirstVary).isNotEqualTo((Object)keyWithSecondVary);
    }

    @Test
    void shouldGenerateDifferentKeyWhenOneOfMultipleVaryHeadersIsDifferent() {
        String varyHeader = "X-MY-VARY";
        String varyHeader2 = "X-MY-SEC-VARY";
        String uri = "https://this";
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.put("X-MY-VARY", List.of("VALUE1"));
        httpHeaders.put(varyHeader2, List.of("VALUE1"));
        MockServerHttpRequest withFirstVary = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders).build();
        HttpHeaders httpHeaders2 = new HttpHeaders();
        httpHeaders.put("X-MY-VARY", List.of("VALUE2"));
        MockServerHttpRequest withSecondVary = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders2).build();
        String keyWithFirstVary = this.cacheKeyGenerator.generateKey((ServerHttpRequest)withFirstVary, new String[]{"X-MY-VARY"});
        String keyWithSecondVary = this.cacheKeyGenerator.generateKey((ServerHttpRequest)withSecondVary, new String[]{"X-MY-VARY"});
        Assertions.assertThat((String)keyWithFirstVary).isNotEqualTo((Object)keyWithSecondVary);
    }

    @Test
    void shouldGenerateDifferentKeyWhenHeadersAreDifferentButValuesAreTheSame() {
        String uri = "https://this";
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.put("X-MY-VARY-1", List.of("VALUE1"));
        httpHeaders.put("X-MY-VARY-2", List.of("VALUE2"));
        HttpHeaders httpHeaders2 = new HttpHeaders();
        httpHeaders2.put("X-MY-VARY-3", List.of("VALUE1"));
        httpHeaders2.put("X-MY-VARY-4", List.of("VALUE2"));
        MockServerHttpRequest withFirstVary = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders).build();
        MockServerHttpRequest withSecondVary = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders2).build();
        String keyWithFirstVary = this.cacheKeyGenerator.generateKey((ServerHttpRequest)withFirstVary, new String[]{"X-MY-VARY-1", "X-MY-VARY-2"});
        String keyWithSecondVary = this.cacheKeyGenerator.generateKey((ServerHttpRequest)withSecondVary, new String[]{"X-MY-VARY-3", "X-MY-VARY-4"});
        Assertions.assertThat((String)keyWithFirstVary).isNotEqualTo((Object)keyWithSecondVary);
    }

    @Test
    void whenHeaderHasEmptyValue() {
        String uri = "https://this";
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.put("X-MY-VARY-1", List.of(""));
        MockServerHttpRequest withFirstVary = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).headers((MultiValueMap)httpHeaders).build();
        MockServerHttpRequest withoutVaryHeader = MockServerHttpRequest.get((String)uri, (Object[])new Object[0]).build();
        String keyWithFirstVary = this.cacheKeyGenerator.generateKey((ServerHttpRequest)withFirstVary, new String[]{"X-MY-VARY-1"});
        String keyWithoutVary = this.cacheKeyGenerator.generateKey((ServerHttpRequest)withFirstVary, new String[]{"X-MY-VARY-1"});
        Assertions.assertThat((String)keyWithoutVary).isEqualTo(keyWithFirstVary);
    }

    @Test
    public void shouldNotFailWhenRunningInParallel() throws InterruptedException {
        MockServerHttpRequest request = MockServerHttpRequest.get((String)"http://this", (Object[])new Object[0]).build();
        int numberOfThreads = 100;
        List<Exception> exceptions = this.executeInParallel(Executors.newFixedThreadPool(numberOfThreads), numberOfThreads, () -> this.cacheKeyGenerator.generateKey((ServerHttpRequest)request, new String[0]));
        Assertions.assertThat((int)exceptions.size()).isEqualTo(0);
    }

    private List<Exception> executeInParallel(Executor executor, int nThreads, Runnable action) throws InterruptedException {
        CountDownLatch ready = new CountDownLatch(nThreads);
        CountDownLatch start = new CountDownLatch(1);
        CountDownLatch done = new CountDownLatch(nThreads);
        ArrayList<Exception> exceptions = new ArrayList<Exception>(nThreads);
        for (int i = 0; i < nThreads; ++i) {
            executor.execute(() -> {
                ready.countDown();
                try {
                    start.await();
                    action.run();
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
                catch (RuntimeException e) {
                    exceptions.add(e);
                }
                finally {
                    done.countDown();
                }
            });
        }
        ready.await();
        start.countDown();
        done.await();
        return exceptions;
    }
}

