/*
 * Copyright 2015-2016 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cloud.stream.converter;

import java.io.IOException;
import java.nio.charset.Charset;

import com.fasterxml.jackson.databind.ObjectMapper;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageHeaders;
import org.springframework.messaging.converter.AbstractMessageConverter;
import org.springframework.tuple.Tuple;
import org.springframework.tuple.TupleBuilder;

/**
 * A {@link org.springframework.messaging.converter.MessageConverter}
 * to convert a {@link Tuple} to a JSON String
 * @author David Turanski
 * @author Ilayaperumal Gopinathan
 */
public class TupleJsonMessageConverter extends AbstractMessageConverter {

	@Value("${typeconversion.json.prettyPrint:false}")
	private volatile boolean prettyPrint;

	private final ObjectMapper objectMapper;

	public void setPrettyPrint(boolean prettyPrint) {
		this.prettyPrint = prettyPrint;
	}

	public TupleJsonMessageConverter(ObjectMapper objectMapper) {
		super(MessageConverterUtils.X_SPRING_TUPLE);
		this.objectMapper = (objectMapper != null) ? objectMapper : new ObjectMapper();
	}

	@Override
	protected boolean supports(Class<?> clazz) {
		return Tuple.class.isAssignableFrom(clazz);
	}

	@Override
	protected Object convertToInternal(Object payload, MessageHeaders headers, Object conversionHint) {
		Tuple t = (Tuple) payload;
		String json;
		if (this.prettyPrint) {
			try {
				Object tmp = this.objectMapper.readValue(t.toString(), Object.class);
				json = this.objectMapper.writerWithDefaultPrettyPrinter().writeValueAsString(tmp);
			}
			catch (IOException e) {
				this.logger.error(e.getMessage(), e);
				return null;
			}
		}
		else {
			json = t.toString();
		}
		return json;
	}

	@Override
	public Object convertFromInternal(Message<?> message, Class<?> targetClass, Object conversionHint) {
		String source;
		if (message.getPayload() instanceof byte[]) {
			source = new String((byte[]) message.getPayload(), Charset.forName("UTF-8"));
		}
		else {
			source = message.getPayload().toString();
		}
		return TupleBuilder.fromString(source);
	}

}
