/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.cloud.vault.config;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.DefaultAWSCredentialsProviderChain;
import java.beans.ConstructorProperties;
import java.net.URI;
import java.util.concurrent.atomic.AtomicReference;
import org.springframework.beans.BeanUtils;
import org.springframework.cloud.vault.config.KubernetesAuthentication;
import org.springframework.cloud.vault.config.KubernetesAuthenticationOptions;
import org.springframework.cloud.vault.config.KubernetesServiceAccountTokenFile;
import org.springframework.cloud.vault.config.VaultProperties;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;
import org.springframework.vault.authentication.AppIdAuthentication;
import org.springframework.vault.authentication.AppIdAuthenticationOptions;
import org.springframework.vault.authentication.AppIdUserIdMechanism;
import org.springframework.vault.authentication.AppRoleAuthentication;
import org.springframework.vault.authentication.AppRoleAuthenticationOptions;
import org.springframework.vault.authentication.AwsEc2Authentication;
import org.springframework.vault.authentication.AwsEc2AuthenticationOptions;
import org.springframework.vault.authentication.AwsIamAuthentication;
import org.springframework.vault.authentication.AwsIamAuthenticationOptions;
import org.springframework.vault.authentication.ClientAuthentication;
import org.springframework.vault.authentication.ClientCertificateAuthentication;
import org.springframework.vault.authentication.CubbyholeAuthentication;
import org.springframework.vault.authentication.CubbyholeAuthenticationOptions;
import org.springframework.vault.authentication.IpAddressUserId;
import org.springframework.vault.authentication.MacAddressUserId;
import org.springframework.vault.authentication.StaticUserId;
import org.springframework.vault.authentication.TokenAuthentication;
import org.springframework.vault.support.VaultToken;
import org.springframework.web.client.RestOperations;

class ClientAuthenticationFactory {
    private final VaultProperties vaultProperties;
    private final RestOperations restOperations;

    ClientAuthentication createClientAuthentication() {
        switch (this.vaultProperties.getAuthentication()) {
            case TOKEN: {
                Assert.hasText((String)this.vaultProperties.getToken(), (String)"Token (spring.cloud.vault.token) must not be empty");
                return new TokenAuthentication(this.vaultProperties.getToken());
            }
            case APPID: {
                return this.appIdAuthentication(this.vaultProperties);
            }
            case APPROLE: {
                return this.appRoleAuthentication(this.vaultProperties);
            }
            case CERT: {
                return new ClientCertificateAuthentication(this.restOperations);
            }
            case AWS_EC2: {
                return this.awsEc2Authentication(this.vaultProperties);
            }
            case AWS_IAM: {
                return this.awsIamAuthentication(this.vaultProperties);
            }
            case CUBBYHOLE: {
                return this.cubbyholeAuthentication();
            }
            case KUBERNETES: {
                return this.kubernetesAuthentication(this.vaultProperties);
            }
        }
        throw new UnsupportedOperationException(String.format("Client authentication %s not supported", new Object[]{this.vaultProperties.getAuthentication()}));
    }

    private ClientAuthentication appIdAuthentication(VaultProperties vaultProperties) {
        VaultProperties.AppIdProperties appId = vaultProperties.getAppId();
        Assert.hasText((String)appId.getUserId(), (String)"UserId (spring.cloud.vault.app-id.user-id) must not be empty");
        AppIdAuthenticationOptions authenticationOptions = AppIdAuthenticationOptions.builder().appId(vaultProperties.getApplicationName()).path(appId.getAppIdPath()).userIdMechanism(this.getClientAuthentication(appId)).build();
        return new AppIdAuthentication(authenticationOptions, this.restOperations);
    }

    private AppIdUserIdMechanism getClientAuthentication(VaultProperties.AppIdProperties appId) {
        try {
            Class userIdClass = ClassUtils.forName((String)appId.getUserId(), null);
            return (AppIdUserIdMechanism)BeanUtils.instantiateClass((Class)userIdClass);
        }
        catch (ClassNotFoundException ex) {
            switch (appId.getUserId().toUpperCase()) {
                case "IP_ADDRESS": {
                    return new IpAddressUserId();
                }
                case "MAC_ADDRESS": {
                    if (StringUtils.hasText((String)appId.getNetworkInterface())) {
                        try {
                            return new MacAddressUserId(Integer.parseInt(appId.getNetworkInterface()));
                        }
                        catch (NumberFormatException e) {
                            return new MacAddressUserId(appId.getNetworkInterface());
                        }
                    }
                    return new MacAddressUserId();
                }
            }
            return new StaticUserId(appId.getUserId());
        }
    }

    private ClientAuthentication appRoleAuthentication(VaultProperties vaultProperties) {
        VaultProperties.AppRoleProperties appRole = vaultProperties.getAppRole();
        Assert.hasText((String)appRole.getRoleId(), (String)"RoleId (spring.cloud.vault.app-role.role-id) must not be empty");
        AppRoleAuthenticationOptions.AppRoleAuthenticationOptionsBuilder builder = AppRoleAuthenticationOptions.builder().path(appRole.getAppRolePath()).roleId(appRole.getRoleId());
        if (StringUtils.hasText((String)appRole.getSecretId())) {
            builder = builder.secretId(appRole.getSecretId());
        }
        return new AppRoleAuthentication(builder.build(), this.restOperations);
    }

    private ClientAuthentication awsEc2Authentication(VaultProperties vaultProperties) {
        VaultProperties.AwsEc2Properties awsEc2 = vaultProperties.getAwsEc2();
        AwsEc2AuthenticationOptions.Nonce nonce = StringUtils.hasText((String)awsEc2.getNonce()) ? AwsEc2AuthenticationOptions.Nonce.provided((char[])awsEc2.getNonce().toCharArray()) : AwsEc2AuthenticationOptions.Nonce.generated();
        AwsEc2AuthenticationOptions authenticationOptions = AwsEc2AuthenticationOptions.builder().role(awsEc2.getRole()).path(awsEc2.getAwsEc2Path()).nonce(nonce).identityDocumentUri(URI.create(awsEc2.getIdentityDocument())).build();
        return new AwsEc2Authentication(authenticationOptions, this.restOperations, this.restOperations);
    }

    private ClientAuthentication awsIamAuthentication(VaultProperties vaultProperties) {
        VaultProperties.AwsIamProperties awsIam = vaultProperties.getAwsIam();
        AWSCredentialsProvider credentialsProvider = AwsCredentialProvider.getAwsCredentialsProvider();
        AwsIamAuthenticationOptions.AwsIamAuthenticationOptionsBuilder builder = AwsIamAuthenticationOptions.builder();
        if (StringUtils.hasText((String)awsIam.getRole())) {
            builder.role(awsIam.getRole());
        }
        if (StringUtils.hasText((String)awsIam.getServerName())) {
            builder.serverName(awsIam.getServerName());
        }
        builder.path(awsIam.getAwsPath()).credentialsProvider(credentialsProvider);
        AwsIamAuthenticationOptions options = builder.credentialsProvider(credentialsProvider).build();
        return new AwsIamAuthentication(options, this.restOperations);
    }

    private ClientAuthentication cubbyholeAuthentication() {
        Assert.hasText((String)this.vaultProperties.getToken(), (String)"Initial Token (spring.cloud.vault.token) for Cubbyhole authentication must not be empty");
        CubbyholeAuthenticationOptions options = CubbyholeAuthenticationOptions.builder().wrapped().initialToken(VaultToken.of((String)this.vaultProperties.getToken())).build();
        return new CubbyholeAuthentication(options, this.restOperations);
    }

    private ClientAuthentication kubernetesAuthentication(VaultProperties vaultProperties) {
        VaultProperties.KubernetesProperties kubernetes = vaultProperties.getKubernetes();
        Assert.hasText((String)kubernetes.getRole(), (String)"Role (spring.cloud.vault.kubernetes.role) must not be empty");
        Assert.hasText((String)kubernetes.getServiceAccountTokenFile(), (String)"Service account token file (spring.cloud.vault.kubernetes.service-account-token-file) must not be empty");
        KubernetesAuthenticationOptions options = KubernetesAuthenticationOptions.builder().path(kubernetes.getKubernetesPath()).role(kubernetes.getRole()).jwtSupplier(new KubernetesServiceAccountTokenFile(kubernetes.getServiceAccountTokenFile())).build();
        return new KubernetesAuthentication(options, this.restOperations);
    }

    @ConstructorProperties(value={"vaultProperties", "restOperations"})
    public ClientAuthenticationFactory(VaultProperties vaultProperties, RestOperations restOperations) {
        this.vaultProperties = vaultProperties;
        this.restOperations = restOperations;
    }

    private static class AwsCredentialProvider {
        private AwsCredentialProvider() {
        }

        private static AWSCredentialsProvider getAwsCredentialsProvider() {
            final DefaultAWSCredentialsProviderChain backingCredentialsProvider = DefaultAWSCredentialsProviderChain.getInstance();
            final AWSCredentials firstAccess = backingCredentialsProvider.getCredentials();
            final AtomicReference<AWSCredentials> once = new AtomicReference<AWSCredentials>(firstAccess);
            return new AWSCredentialsProvider(){

                public AWSCredentials getCredentials() {
                    if (once.compareAndSet(firstAccess, null)) {
                        return firstAccess;
                    }
                    return backingCredentialsProvider.getCredentials();
                }

                public void refresh() {
                    backingCredentialsProvider.refresh();
                }
            };
        }
    }
}

