/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.vault.core.lease;

import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.Trigger;
import org.springframework.scheduling.TriggerContext;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.vault.VaultException;
import org.springframework.vault.client.VaultResponses;
import org.springframework.vault.core.RestOperationsCallback;
import org.springframework.vault.core.VaultOperations;
import org.springframework.vault.core.lease.SecretLeaseEventPublisher;
import org.springframework.vault.core.lease.domain.Lease;
import org.springframework.vault.core.lease.domain.RequestedSecret;
import org.springframework.vault.support.VaultResponseSupport;
import org.springframework.web.client.HttpStatusCodeException;
import org.springframework.web.client.RestOperations;

public class SecretLeaseContainer
extends SecretLeaseEventPublisher
implements InitializingBean,
DisposableBean {
    private static final Log log = LogFactory.getLog(SecretLeaseContainer.class);
    private static final AtomicIntegerFieldUpdater<SecretLeaseContainer> UPDATER = AtomicIntegerFieldUpdater.newUpdater(SecretLeaseContainer.class, "status");
    private static final AtomicInteger poolId = new AtomicInteger();
    private static final int STATUS_INITIAL = 0;
    private static final int STATUS_STARTED = 1;
    private static final int STATUS_DESTROYED = 2;
    private final List<RequestedSecret> requestedSecrets = new CopyOnWriteArrayList<RequestedSecret>();
    private final Map<RequestedSecret, LeaseRenewalScheduler> renewals = new ConcurrentHashMap<RequestedSecret, LeaseRenewalScheduler>();
    private final VaultOperations operations;
    private int minRenewalSeconds = 10;
    private int expiryThresholdSeconds = 60;
    private TaskScheduler taskScheduler;
    private boolean manageTaskScheduler;
    private volatile boolean initialized;
    private volatile int status = 0;

    public SecretLeaseContainer(VaultOperations operations) {
        Assert.notNull((Object)operations, (String)"VaultOperations must not be null");
        this.operations = operations;
    }

    public SecretLeaseContainer(VaultOperations operations, TaskScheduler taskScheduler) {
        Assert.notNull((Object)operations, (String)"VaultOperations must not be null");
        Assert.notNull((Object)taskScheduler, (String)"TaskScheduler must not be null");
        this.operations = operations;
        this.setTaskScheduler(taskScheduler);
    }

    public void setExpiryThresholdSeconds(int expiryThresholdSeconds) {
        this.expiryThresholdSeconds = expiryThresholdSeconds;
    }

    public void setMinRenewalSeconds(int minRenewalSeconds) {
        this.minRenewalSeconds = minRenewalSeconds;
    }

    public int getMinRenewalSeconds() {
        return this.minRenewalSeconds;
    }

    public int getExpiryThresholdSeconds() {
        return this.expiryThresholdSeconds;
    }

    public void setTaskScheduler(TaskScheduler taskScheduler) {
        Assert.notNull((Object)taskScheduler, (String)"TaskScheduler must not be null");
        this.taskScheduler = taskScheduler;
    }

    public RequestedSecret requestRenewableSecret(String path) {
        return this.addRequestedSecret(RequestedSecret.renewable(path));
    }

    public RequestedSecret requestRotatingSecret(String path) {
        return this.addRequestedSecret(RequestedSecret.rotating(path));
    }

    public RequestedSecret addRequestedSecret(RequestedSecret requestedSecret) {
        Assert.notNull((Object)requestedSecret, (String)"RequestedSecret must not be null");
        this.requestedSecrets.add(requestedSecret);
        if (this.initialized) {
            LeaseRenewalScheduler leaseRenewalScheduler = new LeaseRenewalScheduler(this.taskScheduler);
            this.renewals.put(requestedSecret, leaseRenewalScheduler);
            if (this.status == 1) {
                this.start(requestedSecret, leaseRenewalScheduler);
            }
        }
        return requestedSecret;
    }

    public void start() {
        Assert.state((boolean)this.initialized, (String)"Container is not initialized");
        Assert.state((this.status != 2 ? 1 : 0) != 0, (String)"Container is destroyed and cannot be started");
        HashMap<RequestedSecret, LeaseRenewalScheduler> renewals = new HashMap<RequestedSecret, LeaseRenewalScheduler>(this.renewals);
        if (UPDATER.compareAndSet(this, 0, 1)) {
            for (Map.Entry entry : renewals.entrySet()) {
                this.start((RequestedSecret)entry.getKey(), (LeaseRenewalScheduler)entry.getValue());
            }
        }
    }

    private void start(RequestedSecret requestedSecret, LeaseRenewalScheduler renewalScheduler) {
        VaultResponseSupport<Map<String, Object>> secrets = this.doGetSecrets(requestedSecret);
        if (secrets != null) {
            Lease lease = !StringUtils.hasText((String)secrets.getLeaseId()) ? Lease.none() : Lease.of(secrets.getLeaseId(), secrets.getLeaseDuration(), secrets.isRenewable());
            this.potentiallyScheduleLeaseRenewal(requestedSecret, lease, renewalScheduler);
            this.onSecretsObtained(requestedSecret, lease, secrets.getData());
        }
    }

    public void stop() {
        if (UPDATER.compareAndSet(this, 1, 0)) {
            for (LeaseRenewalScheduler leaseRenewal : this.renewals.values()) {
                leaseRenewal.disableScheduleRenewal();
            }
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        if (!this.initialized) {
            super.afterPropertiesSet();
            this.initialized = true;
            if (this.taskScheduler == null) {
                ThreadPoolTaskScheduler scheduler = new ThreadPoolTaskScheduler();
                scheduler.setDaemon(true);
                scheduler.setThreadNamePrefix(String.format("%s-%d-", this.getClass().getSimpleName(), poolId.incrementAndGet()));
                scheduler.afterPropertiesSet();
                this.taskScheduler = scheduler;
                this.manageTaskScheduler = true;
            }
            for (RequestedSecret requestedSecret : this.requestedSecrets) {
                this.renewals.put(requestedSecret, new LeaseRenewalScheduler(this.taskScheduler));
            }
        }
    }

    public void destroy() throws Exception {
        int status = this.status;
        if ((status == 0 || status == 1) && UPDATER.compareAndSet(this, status, 2)) {
            for (Map.Entry<RequestedSecret, LeaseRenewalScheduler> entry : this.renewals.entrySet()) {
                Lease lease = entry.getValue().getLease();
                entry.getValue().disableScheduleRenewal();
                if (lease == null) continue;
                this.doRevokeLease(entry.getKey(), lease);
            }
            if (this.manageTaskScheduler && this.taskScheduler instanceof DisposableBean) {
                ((DisposableBean)this.taskScheduler).destroy();
                this.taskScheduler = null;
            }
        }
    }

    void potentiallyScheduleLeaseRenewal(final RequestedSecret requestedSecret, Lease lease, final LeaseRenewalScheduler leaseRenewal) {
        if (leaseRenewal.isLeaseRenewable(lease)) {
            if (log.isDebugEnabled()) {
                log.debug((Object)String.format("Lease %s qualified for renewal", lease.getLeaseId()));
            }
            leaseRenewal.scheduleRenewal(new RenewLease(){

                @Override
                public Lease renewLease(Lease lease) {
                    Lease newLease = SecretLeaseContainer.this.doRenewLease(requestedSecret, lease);
                    if (newLease == null) {
                        return null;
                    }
                    SecretLeaseContainer.this.potentiallyScheduleLeaseRenewal(requestedSecret, newLease, leaseRenewal);
                    SecretLeaseContainer.this.onAfterLeaseRenewed(requestedSecret, newLease);
                    return newLease;
                }
            }, lease, this.getMinRenewalSeconds(), this.getExpiryThresholdSeconds());
        }
    }

    protected VaultResponseSupport<Map<String, Object>> doGetSecrets(RequestedSecret requestedSecret) {
        try {
            return this.operations.read(requestedSecret.getPath());
        }
        catch (RuntimeException e) {
            this.onError(requestedSecret, Lease.none(), e);
            return null;
        }
    }

    protected Lease doRenewLease(RequestedSecret requestedSecret, final Lease lease) {
        try {
            ResponseEntity<Map<String, Object>> entity = this.operations.doWithSession(new RestOperationsCallback<ResponseEntity<Map<String, Object>>>(){

                @Override
                public ResponseEntity<Map<String, Object>> doWithRestOperations(RestOperations restOperations) {
                    return restOperations.exchange("/sys/renew/{leaseId}", HttpMethod.PUT, null, Map.class, new Object[]{lease.getLeaseId()});
                }
            });
            Map body = (Map)entity.getBody();
            String leaseId = (String)body.get("lease_id");
            Number leaseDuration = (Number)body.get("lease_duration");
            boolean renewable = (Boolean)body.get("renewable");
            if (leaseDuration == null || leaseDuration.intValue() < this.minRenewalSeconds) {
                this.onLeaseExpired(requestedSecret, lease);
                return null;
            }
            return Lease.of(leaseId, leaseDuration.longValue(), renewable);
        }
        catch (HttpStatusCodeException e) {
            if (e.getStatusCode() == HttpStatus.BAD_REQUEST) {
                this.onLeaseExpired(requestedSecret, lease);
            }
            this.onError(requestedSecret, lease, (Exception)((Object)new VaultException(String.format("Cannot renew lease: %s", VaultResponses.getError(e.getResponseBodyAsString())))));
        }
        catch (RuntimeException e) {
            this.onError(requestedSecret, lease, e);
        }
        return null;
    }

    @Override
    protected void onLeaseExpired(RequestedSecret requestedSecret, Lease lease) {
        super.onLeaseExpired(requestedSecret, lease);
        if (requestedSecret.getMode() == RequestedSecret.Mode.ROTATE) {
            this.start(requestedSecret, this.renewals.get(requestedSecret));
        }
    }

    protected void doRevokeLease(RequestedSecret requestedSecret, final Lease lease) {
        try {
            this.onBeforeLeaseRevocation(requestedSecret, lease);
            this.operations.doWithSession(new RestOperationsCallback<ResponseEntity<Map<String, Object>>>(){

                @Override
                public ResponseEntity<Map<String, Object>> doWithRestOperations(RestOperations restOperations) {
                    return restOperations.exchange("/sys/revoke/{leaseId}", HttpMethod.PUT, null, Map.class, new Object[]{lease.getLeaseId()});
                }
            });
            this.onAfterLeaseRevocation(requestedSecret, lease);
        }
        catch (HttpStatusCodeException e) {
            this.onError(requestedSecret, lease, (Exception)((Object)new VaultException(String.format("Cannot revoke lease: %s", VaultResponses.getError(e.getResponseBodyAsString())))));
        }
        catch (RuntimeException e) {
            this.onError(requestedSecret, lease, e);
        }
    }

    static interface RenewLease {
        public Lease renewLease(Lease var1) throws VaultException;
    }

    static class OneShotTrigger
    implements Trigger {
        private static final AtomicIntegerFieldUpdater<OneShotTrigger> UPDATER = AtomicIntegerFieldUpdater.newUpdater(OneShotTrigger.class, "status");
        private static final int STATUS_ARMED = 0;
        private static final int STATUS_FIRED = 1;
        private volatile int status = 0;
        private final long seconds;

        OneShotTrigger(long seconds) {
            this.seconds = seconds;
        }

        public Date nextExecutionTime(TriggerContext triggerContext) {
            if (UPDATER.compareAndSet(this, 0, 1)) {
                return new Date(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(this.seconds));
            }
            return null;
        }
    }

    static class LeaseRenewalScheduler {
        private static final Log log = LogFactory.getLog(LeaseRenewalScheduler.class);
        private final TaskScheduler taskScheduler;
        final AtomicReference<Lease> currentLeaseRef = new AtomicReference();
        final Map<Lease, ScheduledFuture<?>> schedules = new ConcurrentHashMap();

        LeaseRenewalScheduler(TaskScheduler taskScheduler) {
            this.taskScheduler = taskScheduler;
        }

        void scheduleRenewal(final RenewLease renewLease, final Lease lease, int minRenewalSeconds, int expiryThresholdSeconds) {
            if (log.isDebugEnabled()) {
                log.debug((Object)String.format("Scheduling renewal for lease %s, lease duration %d", lease.getLeaseId(), lease.getLeaseDuration()));
            }
            Lease currentLease = this.currentLeaseRef.get();
            this.currentLeaseRef.set(lease);
            if (currentLease != null) {
                this.cancelSchedule(currentLease);
            }
            ScheduledFuture scheduledFuture = this.taskScheduler.schedule(new Runnable(){

                @Override
                public void run() {
                    try {
                        LeaseRenewalScheduler.this.schedules.remove(lease);
                        if (LeaseRenewalScheduler.this.currentLeaseRef.get() != lease) {
                            log.debug((Object)"Current lease has changed. Skipping renewal");
                            return;
                        }
                        if (log.isDebugEnabled()) {
                            log.debug((Object)String.format("Renewing lease %s", lease.getLeaseId()));
                        }
                        LeaseRenewalScheduler.this.currentLeaseRef.compareAndSet(lease, renewLease.renewLease(lease));
                    }
                    catch (Exception e) {
                        log.error((Object)String.format("Cannot renew lease %s", lease.getLeaseId()), (Throwable)e);
                    }
                }
            }, (Trigger)new OneShotTrigger(this.getRenewalSeconds(lease, minRenewalSeconds, expiryThresholdSeconds)));
            this.schedules.put(lease, scheduledFuture);
        }

        private void cancelSchedule(Lease lease) {
            ScheduledFuture<?> scheduledFuture = this.schedules.get(lease);
            if (scheduledFuture != null) {
                if (log.isDebugEnabled()) {
                    log.debug((Object)String.format("Canceling previously registered schedule for lease %s", lease.getLeaseId()));
                }
                scheduledFuture.cancel(false);
            }
        }

        void disableScheduleRenewal() {
            this.currentLeaseRef.set(null);
            HashSet<Lease> leases = new HashSet<Lease>(this.schedules.keySet());
            for (Lease lease : leases) {
                this.cancelSchedule(lease);
                this.schedules.remove(lease);
            }
        }

        private long getRenewalSeconds(Lease lease, int minRenewalSeconds, int expiryThresholdSeconds) {
            return Math.max((long)minRenewalSeconds, lease.getLeaseDuration() - (long)expiryThresholdSeconds);
        }

        private boolean isLeaseRenewable(Lease lease) {
            return lease != null && lease.isRenewable();
        }

        public Lease getLease() {
            return this.currentLeaseRef.get();
        }
    }
}

