/*
 * Copyright 2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springunit.framework;

import java.util.Map;

/**
 * A bean that holds data for a single SpringUnitTest.<br/>
 * The bean contains a single attribute, data,
 * that is a map of strings to objects.  The string keys
 * are either the names of tests or arbitrary identifiers
 * for data values.  For keys that name tests,
 * the corresponding value is itself a map of string keys
 * to object values.<br/>
 * 
 * @author <a href="mailto:ted@velkoff.com">Ted Velkoff</a>
 *
 */
public class SpringUnitContext<T> {

	/**
	 * Test data values associated with a TestCase.
	 * By convention, keys of the form "testXxx"
	 * have associated values of the type
	 * Map<? extends String, ? extends T>.
	 * All such elements comprise data items
	 * in the scope of the test having the same
	 * name as the key.
	 * Keys of any other form have values of type T,
	 * and represent data in the scope of the test case.<br/>
	 * @return Map<? extends String, ? extends T>
	 */
	public Map<? extends String, ? extends T> getData() {
		return this.data;
	}
	
	/**
	 * Find the data value identified by <code>key</code>,
	 * searching first in the scope of the name
	 * of the test (<code>fName</code>), then outward
	 * until found or <code>null</code> is returned.<br/>
	 * @param key Identifier of data value to be found
	 * @param fName Name of test in whose scope search
	 * should begin
	 * @return Object of type T if found, null otherwise
	 */
	public T getObject(String key, String fName) {
		return getObject(key, fName, getData());
	}
	
	/**
	 * Make <code>data</code> the test data values
	 * associcated with TestCase.<br/>
	 * @param data Data values indexed by string names
	 */
	public void setData(Map<? extends String, ? extends T> data) {
		this.data = data;
	}
	
	/**
	 * Find the data value identified by <code>key</code>
	 * in the <code>map</code> by searching first
	 * in the scope of the test (<code>fName</code>).
	 * If <code>fName</code> is the name of a map
	 * <code>testMap</code>
	 * contained in <code>map</code>, then return
	 * the value in <code>testMap</code> associated
	 * with <code>key</code>.  If there is no
	 * value in <code>testMap</code> associated with
	 * <code>key</code>, then return the value 
	 * associated with <code>key</code> in <code>map</code>,
	 * which could be <code>null</code>.
	 * @param key Identifier of data value
	 * @param fName Identifier of test; must exactly match
	 * the name of a test procedure in a SpringUnitTest.
	 * @param map Map of string identifiers to objects
	 * where the string is either an fName or a key
	 * @return Object of type T if found, otherwise null
	 */
	protected T getObject(String key, String fName, Map<? extends String, ? extends T> map) {
		T testData = map.get(fName);
		if ((testData != null) && testData instanceof Map) {
			Map<? extends String, ? extends T> testMap = (Map<? extends String, ? extends T>)testData;
			T obj = testMap.get(key);
			if (obj != null) {
				return obj;
			}
		}
		return map.get(key);
	}
	
	/**
	 * Map containing data that comprises the test context.<br/>
	 */
	private Map<? extends String, ? extends T> data;
	
}
